<?php
/**
 * Admin Panel - Forum
 */

if (!defined('ROOT_PATH')) {
    http_response_code(403);
    exit;
}

use NewSite\Admin\AdminLayout;
use NewSite\Auth\AdminAuth;
use NewSite\Database\DatabaseManager;
use NewSite\Database\DbHelper;
use NewSite\Forum\ForumHelper;
use NewSite\Settings\SettingsService;
use NewSite\Upload\ImageUrlHelper;

AdminAuth::requireLogin();

$db = DatabaseManager::getWriteConnection();
$defaultAvatar = '/assets/images/default-avatar.svg';

if (!function_exists('toPublicFilePath')) {
    function toPublicFilePath($path)
    {
        if (empty($path)) {
            return $path;
        }
        return str_replace('/admin-file/', '/site-file/', $path);
    }
}

function forumJson(array $data, int $status = 200): void
{
    header('Content-Type: application/json');
    http_response_code($status);
    echo json_encode($data);
    exit;
}

function forumSlug(string $value): string
{
    $value = strtolower(trim($value));
    $value = preg_replace('/[^a-z0-9\s-]/', '', $value);
    $value = preg_replace('/\s+/', '-', $value);
    $value = preg_replace('/-+/', '-', $value);
    return trim($value, '-');
}

function forumGetCategories(PDO $db): array
{
    return $db->query("SELECT * FROM forum_categories ORDER BY sort_order ASC, name ASC")->fetchAll();
}

function forumGetAllSubcategories(PDO $db, int $categoryId): array
{
    if ($categoryId <= 0) {
        return [];
    }
    $stmt = $db->prepare("SELECT * FROM forum_subcategories WHERE category_id = ? ORDER BY sort_order ASC, name ASC");
    $stmt->execute([$categoryId]);
    return $stmt->fetchAll();
}

function forumGetSubcategories(PDO $db, int $categoryId): array
{
    $allSubs = forumGetAllSubcategories($db, $categoryId);
    return ForumHelper::buildSubcategoryTree($allSubs, null);
}

function forumGetCategoryIdForSubcategory(PDO $db, int $subcategoryId): int
{
    if ($subcategoryId <= 0) {
        return 0;
    }
    $stmt = $db->prepare("SELECT category_id FROM forum_subcategories WHERE id = ?");
    $stmt->execute([$subcategoryId]);
    return (int)($stmt->fetchColumn() ?? 0);
}

function forumResolveCategoryId(array $categories, int $requested): int
{
    foreach ($categories as $cat) {
        if ((int)$cat['id'] === $requested) {
            return $requested;
        }
    }
    return !empty($categories) ? (int)$categories[0]['id'] : 0;
}

function forumResolveSubcategoryId(array $subcategories, int $requested): int
{
    foreach ($subcategories as $sub) {
        if ((int)$sub['id'] === $requested) {
            return $requested;
        }
    }
    return !empty($subcategories) ? (int)$subcategories[0]['id'] : 0;
}

function forumGetPosts(PDO $db, int $subcategoryId, int $page, int $perPage): array
{
    if ($subcategoryId <= 0) {
        return [[], 0];
    }

    $stmt = $db->prepare("SELECT COUNT(*) FROM forum_posts WHERE subcategory_id = ?");
    $stmt->execute([$subcategoryId]);
    $total = (int)$stmt->fetchColumn();

    $offset = max(0, ($page - 1) * $perPage);
    $stmt = $db->prepare("
        SELECT p.*, u.display_name, u.nickname, u.profile_photo,
            COALESCE(c.comment_count, 0) AS comment_count
        FROM forum_posts p
        LEFT JOIN site_users u ON u.id = p.user_id
        LEFT JOIN (
            SELECT post_id, COUNT(*) AS comment_count
            FROM forum_comments
            GROUP BY post_id
        ) c ON c.post_id = p.id
        WHERE p.subcategory_id = ?
        ORDER BY p.created_at DESC
        LIMIT ? OFFSET ?
    ");
    $stmt->execute([$subcategoryId, $perPage, $offset]);
    $posts = $stmt->fetchAll();

    if (empty($posts)) {
        return [$posts, $total];
    }

    return [forumAttachPostImages($db, $posts), $total];
}

/**
 * Attach images to each post by fetching them in a single batch query.
 */
function forumAttachPostImages(PDO $db, array $posts): array
{
    $postIds = array_values(array_filter(
        array_map(static fn ($p) => (int)($p['id'] ?? 0), $posts)
    ));

    if (empty($postIds)) {
        return $posts;
    }

    $placeholders = implode(',', array_fill(0, count($postIds), '?'));
    $stmt = $db->prepare(
        "SELECT post_id, image_path FROM forum_post_images WHERE post_id IN ($placeholders) ORDER BY sort_order ASC, id ASC"
    );
    $stmt->execute($postIds);

    $imagesByPost = [];
    foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) as $row) {
        $postId = (int)($row['post_id'] ?? 0);
        if ($postId > 0 && !empty($row['image_path'])) {
            $imagesByPost[$postId][] = $row['image_path'];
        }
    }

    foreach ($posts as $index => $post) {
        $posts[$index]['images'] = $imagesByPost[(int)($post['id'] ?? 0)] ?? [];
    }

    return $posts;
}

function renderForumCategories(array $categories, int $activeCategoryId): string
{
    ob_start();
    ?>
    <div class="card">
        <div class="card-header">
            <h3>Categories</h3>
        </div>
        <div class="card-body">
            <?php if (empty($categories)): ?>
                <div class="text-muted">No categories yet.</div>
            <?php else: ?>
                <div class="forum-list">
                    <?php foreach ($categories as $cat):
                        $isActive = (int)$cat['id'] === $activeCategoryId;
                        ?>
                        <div class="forum-list-item<?php echo $isActive ? ' is-active' : ''; ?>" data-forum-category="<?php echo (int)$cat['id']; ?>">
                            <div class="forum-list-main">
                                <div class="forum-list-title"><?php echo e($cat['name']); ?></div>
                                <div class="forum-list-meta"><?php echo !empty($cat['is_active']) ? 'Enabled' : 'Disabled'; ?></div>
                            </div>
                            <div class="forum-list-actions">
                                <button type="button" class="btn btn-secondary btn-sm" data-category-edit
                                    data-category-id="<?php echo (int)$cat['id']; ?>"
                                    data-category-name="<?php echo e($cat['name']); ?>"
                                    data-category-active="<?php echo (int)($cat['is_active'] ?? 0); ?>"
                                    data-category-sort="<?php echo (int)($cat['sort_order'] ?? 0); ?>">Edit</button>
                                <button type="button" class="btn btn-danger btn-sm" data-category-delete="<?php echo (int)$cat['id']; ?>">Delete</button>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>

            <form class="forum-form" data-forum-form="category">
                <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                <input type="hidden" name="action" value="add_category" data-category-action>
                <input type="hidden" name="category_id" value="" data-category-id>
                <div class="form-group">
                    <label for="forum-category-name">Category Name</label>
                    <input type="text" class="form-control" id="forum-category-name" name="name" required data-category-name>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="forum-category-sort">Sort Order</label>
                        <input type="number" class="form-control" id="forum-category-sort" name="sort_order" value="0" data-category-sort>
                    </div>
                    <div class="form-group">
                        <label for="forum-category-active">Enabled</label>
                        <label class="form-check">
                            <input type="checkbox" id="forum-category-active" name="is_active" value="1" checked data-category-active>
                            <span>Active</span>
                        </label>
                    </div>
                </div>
                <div class="form-actions">
                    <button type="button" class="btn btn-secondary btn-sm" data-category-reset>New</button>
                    <button type="submit" class="btn btn-primary btn-sm">Save Category</button>
                </div>
            </form>
        </div>
    </div>
    <?php
    return ob_get_clean();
}

function renderSubcategoryTreeItems(array $subcategories, int $activeSubcategoryId, int $depth = 0): string
{
    $html = '';
    foreach ($subcategories as $sub) {
        $isActive = (int)$sub['id'] === $activeSubcategoryId;
        $icon = $depth > 0 ? '└─ ' : '📁 ';
        $depthClass = 'forum-sub-depth-' . min($depth, 8);

        $html .= '<div class="forum-list-item' . ($isActive ? ' is-active' : '') . ' ' . $depthClass . '" data-forum-subcategory="' . (int)$sub['id'] . '">';
        $html .= '<div class="forum-list-main">';
        $html .= '<div class="forum-list-title"><span class="forum-tree-icon">' . $icon . '</span>' . e($sub['name']) . '</div>';
        $html .= '<div class="forum-list-meta">' . (!empty($sub['is_active']) ? 'Enabled' : 'Disabled') . '</div>';
        $html .= '</div>';
        $html .= '<div class="forum-list-actions">';
        $html .= '<button type="button" class="btn btn-success btn-sm" data-subcategory-add-child="' . (int)$sub['id'] . '" title="Add child">+</button>';
        $html .= '<button type="button" class="btn btn-secondary btn-sm" data-subcategory-edit ';
        $html .= 'data-subcategory-id="' . (int)$sub['id'] . '" ';
        $html .= 'data-subcategory-name="' . e($sub['name']) . '" ';
        $html .= 'data-subcategory-description="' . e($sub['description'] ?? '') . '" ';
        $html .= 'data-subcategory-parent="' . (int)($sub['parent_id'] ?? 0) . '" ';
        $html .= 'data-subcategory-active="' . (int)($sub['is_active'] ?? 0) . '" ';
        $html .= 'data-subcategory-sort="' . (int)($sub['sort_order'] ?? 0) . '">Edit</button>';
        $html .= '<button type="button" class="btn btn-danger btn-sm" data-subcategory-delete="' . (int)$sub['id'] . '">Delete</button>';
        $html .= '</div></div>';

        if (!empty($sub['children'])) {
            $html .= renderSubcategoryTreeItems($sub['children'], $activeSubcategoryId, $depth + 1);
        }
    }
    return $html;
}

function renderSubcategoryOptions(array $subcategories, int $depth = 0): string
{
    $html = '';
    foreach ($subcategories as $sub) {
        $indent = str_repeat('&nbsp;&nbsp;&nbsp;', $depth);
        $prefix = $depth > 0 ? '└─ ' : '';
        $html .= '<option value="' . (int)$sub['id'] . '">' . $indent . $prefix . e($sub['name']) . '</option>';
        if (!empty($sub['children'])) {
            $html .= renderSubcategoryOptions($sub['children'], $depth + 1);
        }
    }
    return $html;
}

function renderForumSubcategories(array $subcategories, int $activeCategoryId, int $activeSubcategoryId): string
{
    ob_start();
    ?>
    <div class="card">
        <div class="card-header">
            <h3>Subcategories</h3>
            <small class="text-muted">Tree structure: subcategories can have child subcategories</small>
        </div>
        <div class="card-body">
            <?php if ($activeCategoryId <= 0): ?>
                <div class="text-muted">Select a category to see subcategories.</div>
            <?php elseif (empty($subcategories)): ?>
                <div class="text-muted">No subcategories yet. Create one below.</div>
            <?php else: ?>
                <div class="forum-list">
                    <?php echo renderSubcategoryTreeItems($subcategories, $activeSubcategoryId, 0); ?>
                </div>
            <?php endif; ?>

            <form class="forum-form" data-forum-form="subcategory">
                <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                <input type="hidden" name="action" value="add_subcategory" data-subcategory-action>
                <input type="hidden" name="subcategory_id" value="" data-subcategory-id>
                <input type="hidden" name="category_id" value="<?php echo (int)$activeCategoryId; ?>" data-subcategory-category>
                <input type="hidden" name="parent_id" value="" data-subcategory-parent>
                <div class="form-group">
                    <label for="forum-subcategory-name">Subcategory Name</label>
                    <input type="text" class="form-control" id="forum-subcategory-name" name="name" required data-subcategory-name>
                </div>
                <div class="form-group">
                    <label for="forum-subcategory-desc">Description</label>
                    <textarea class="form-control" id="forum-subcategory-desc" name="description" rows="2" data-subcategory-description></textarea>
                </div>
                <div class="form-group">
                    <label for="forum-subcategory-parent">Parent Subcategory (optional - leave empty for top-level)</label>
                    <select class="form-control" id="forum-subcategory-parent" name="parent_id" data-subcategory-parent-select>
                        <option value="">None (Top Level)</option>
                        <?php echo renderSubcategoryOptions($subcategories, 0); ?>
                    </select>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="forum-subcategory-sort">Sort Order</label>
                        <input type="number" class="form-control" id="forum-subcategory-sort" name="sort_order" value="0" data-subcategory-sort>
                    </div>
                    <div class="form-group">
                        <label for="forum-subcategory-active">Enabled</label>
                        <label class="form-check">
                            <input type="checkbox" id="forum-subcategory-active" name="is_active" value="1" checked data-subcategory-active>
                            <span>Active</span>
                        </label>
                    </div>
                </div>
                <div class="form-actions">
                    <button type="button" class="btn btn-secondary btn-sm" data-subcategory-reset>New</button>
                    <button type="submit" class="btn btn-primary btn-sm">Save Subcategory</button>
                </div>
            </form>
        </div>
    </div>
    <?php
    return ob_get_clean();
}

function forumResolvePostAvatar(array $post, string $defaultAvatar): string
{
    $profilePhoto = trim($post['profile_photo'] ?? '');
    return $profilePhoto !== '' ? toPublicFilePath($profilePhoto) : $defaultAvatar;
}

function forumResolvePostDisplayName(array $post): string
{
    $displayName = trim($post['nickname'] ?? '');
    if ($displayName === '') {
        $displayName = trim($post['display_name'] ?? '');
    }
    if ($displayName === '') {
        $displayName = 'User';
    }
    return $displayName;
}

function forumTruncateText(string $text, int $maxLength = 600): string
{
    if ($text === '') {
        return '';
    }
    $strlen = function_exists('mb_strlen') ? 'mb_strlen' : 'strlen';
    $substr = function_exists('mb_substr') ? 'mb_substr' : 'substr';
    return $strlen($text) > $maxLength ? $substr($text, 0, $maxLength) . '…' : $text;
}

function forumRenderPostImages(array $images, string $title): string
{
    if (empty($images)) {
        return '';
    }
    ob_start();
    ?>
                                    <div class="forum-post-images">
                                        <?php foreach ($images as $imagePath):
                                            $imageUrl = ImageUrlHelper::getForumImageUrl($imagePath);
                                            if (!$imageUrl) {
                                                continue;
                                            }
                                            ?>
                                            <img src="<?php echo e($imageUrl); ?>" alt="<?php echo e($title); ?>">
                                        <?php endforeach; ?>
                                    </div>
    <?php
    return ob_get_clean();
}

function forumRenderPostActions(string $status, int $postId): string
{
    $statusKey = strtolower(preg_replace('/[^a-z]/', '', $status));
    ob_start();
    ?>
                            <div class="forum-post-actions">
                                <span class="status-pill status-pill--<?php echo e($statusKey ?: 'pending'); ?>"><?php echo e($status); ?></span>
                                <?php if ($status === 'approved'): ?>
                                    <button type="button" class="btn btn-danger btn-sm" data-post-delete="<?php echo $postId; ?>">Remove Post</button>
                                <?php else: ?>
                                    <button type="button" class="btn btn-secondary btn-sm" data-post-status="approved" data-post-id="<?php echo $postId; ?>">Accept</button>
                                    <button type="button" class="btn btn-danger btn-sm" data-post-status="declined" data-post-id="<?php echo $postId; ?>">Decline</button>
                                <?php endif; ?>
                            </div>
    <?php
    return ob_get_clean();
}

function forumRenderSinglePost(array $post, string $defaultAvatar): string
{
    $avatarUrl = forumResolvePostAvatar($post, $defaultAvatar);
    $displayName = forumResolvePostDisplayName($post);
    $createdAt = !empty($post['created_at']) ? date('M j, Y', strtotime($post['created_at'])) : '';
    $status = $post['status'] ?? 'pending';
    $commentCount = (int)($post['comment_count'] ?? 0);
    $bodyPreview = forumTruncateText(trim((string)($post['body'] ?? '')));
    $postId = (int)$post['id'];
    ob_start();
    ?>
                        <div class="forum-post">
                            <div class="forum-post-avatar">
                                <img src="<?php echo e($avatarUrl); ?>" alt="<?php echo e($displayName); ?>">
                            </div>
                            <div class="forum-post-main">
                                <div class="forum-post-title"><?php echo e($post['title']); ?></div>
                                <div class="forum-post-meta">
                                    <span>By <?php echo e($displayName); ?></span>
                                    <?php if ($createdAt !== ''): ?>
                                        <span>· <?php echo e($createdAt); ?></span>
                                    <?php endif; ?>
                                    <span>· <?php echo $commentCount; ?> comments</span>
                                </div>
                                <?php if ($bodyPreview !== ''): ?>
                                    <div class="forum-post-body">
                                        <?php echo nl2br(e($bodyPreview)); ?>
                                    </div>
                                <?php endif; ?>
                                <?php echo forumRenderPostImages($post['images'] ?? [], trim($post['title'] ?? 'Post attachment')); ?>
                            </div>
                            <?php echo forumRenderPostActions($status, $postId); ?>
                        </div>
    <?php
    return ob_get_clean();
}

function forumRenderPagination(int $page, int $totalPages): string
{
    $prevDisabled = $page <= 1 ? 'disabled' : '';
    $nextDisabled = $page >= $totalPages ? 'disabled' : '';
    ob_start();
    ?>
                <div class="forum-pagination">
                    <button type="button" class="btn btn-secondary btn-sm" data-posts-page="<?php echo max(1, $page - 1); ?>" <?php echo $prevDisabled; ?>>Prev</button>
                    <span class="forum-page-label">Page <?php echo $page; ?> of <?php echo $totalPages; ?></span>
                    <button type="button" class="btn btn-secondary btn-sm" data-posts-page="<?php echo min($totalPages, $page + 1); ?>" <?php echo $nextDisabled; ?>>Next</button>
                </div>
    <?php
    return ob_get_clean();
}

function renderForumPosts(array $posts, int $total, int $page, int $perPage, int $activeSubcategoryId, string $defaultAvatar): string
{
    $totalPages = max(1, (int)ceil($total / $perPage));
    $page = min(max(1, $page), $totalPages);
    ob_start();
    ?>
    <div class="card">
        <div class="card-header">
            <h3>Posts</h3>
        </div>
        <div class="card-body">
            <?php if ($activeSubcategoryId <= 0): ?>
                <div class="text-muted">Select a subcategory to view posts.</div>
            <?php elseif (empty($posts)): ?>
                <div class="text-muted">No posts yet.</div>
            <?php else: ?>
                <div class="forum-posts">
                    <?php foreach ($posts as $post): ?>
                        <?php echo forumRenderSinglePost($post, $defaultAvatar); ?>
                    <?php endforeach; ?>
                </div>
                <?php echo forumRenderPagination($page, $totalPages); ?>
            <?php endif; ?>
        </div>
    </div>
    <?php
    return ob_get_clean();
}

function forumBuildPayload(PDO $db, int $activeCategoryId, int $activeSubcategoryId, int $page, int $perPage, string $defaultAvatar): array
{
    $categories = forumGetCategories($db);
    $activeCategoryId = forumResolveCategoryId($categories, $activeCategoryId);
    $subcategories = forumGetSubcategories($db, $activeCategoryId);
    $activeSubcategoryId = forumResolveSubcategoryId($subcategories, $activeSubcategoryId);

    [$posts, $total] = forumGetPosts($db, $activeSubcategoryId, $page, $perPage);

    return [
        'categories_html' => renderForumCategories($categories, $activeCategoryId),
        'subcategories_html' => renderForumSubcategories($subcategories, $activeCategoryId, $activeSubcategoryId),
        'posts_html' => renderForumPosts($posts, $total, $page, $perPage, $activeSubcategoryId, $defaultAvatar),
        'active_category_id' => $activeCategoryId,
        'active_subcategory_id' => $activeSubcategoryId
    ];
}

const FORUM_DEFAULT_DESCRIPTION = 'Join the conversation and explore the latest discussions.';

$forumEnabled = SettingsService::get('forum_enabled', '0') === '1';
$forumAutoApprove = SettingsService::get('forum_auto_approve', '0') === '1';
$forumTitleSetting = trim((string)SettingsService::get('forum_title', 'Forum'));
if ($forumTitleSetting === '') {
    $forumTitleSetting = 'Forum';
}
$forumDescriptionSetting = trim((string)SettingsService::get('forum_description', FORUM_DEFAULT_DESCRIPTION));
if ($forumDescriptionSetting === '') {
    $forumDescriptionSetting = FORUM_DEFAULT_DESCRIPTION;
}

$perPage = 50;

if (!empty($_POST['ajax']) || !empty($_GET['ajax'])) {
    if ($_SERVER['REQUEST_METHOD'] === 'GET') {
        $action = $_GET['action'] ?? '';
        $page = max(1, (int)($_GET['page'] ?? 1));
        if ($action === 'subcategories') {
            $categoryId = (int)($_GET['category_id'] ?? 0);
            $payload = forumBuildPayload($db, $categoryId, 0, 1, $perPage, $defaultAvatar);
            forumJson(['success' => true] + $payload);
        }
        if ($action === 'posts') {
            $subcategoryId = (int)($_GET['subcategory_id'] ?? 0);
            $categoryId = forumGetCategoryIdForSubcategory($db, $subcategoryId);
            $payload = forumBuildPayload($db, $categoryId, $subcategoryId, $page, $perPage, $defaultAvatar);
            forumJson(['success' => true] + $payload);
        }
        forumJson(['success' => false, 'error' => 'Invalid action.'], 400);
    }

    $action = $_POST['action'] ?? '';
    verifyCSRF();

    if ($action === 'save_settings') {
        $enabled = !empty($_POST['forum_enabled']) ? '1' : '0';
        $autoApprove = !empty($_POST['forum_auto_approve']) ? '1' : '0';
        $title = trim((string)($_POST['forum_title'] ?? ''));
        $description = trim((string)($_POST['forum_description'] ?? ''));
        if ($title === '') {
            $title = 'Forum';
        }
        if ($description === '') {
            $description = 'Join the conversation and explore the latest discussions.';
        }
        if (function_exists('mb_substr')) {
            $title = mb_substr($title, 0, 120);
            $description = mb_substr($description, 0, 200);
        } else {
            $title = substr($title, 0, 120);
            $description = substr($description, 0, 200);
        }
        SettingsService::set('forum_enabled', $enabled);
        SettingsService::set('forum_auto_approve', $autoApprove);
        SettingsService::set('forum_title', $title);
        SettingsService::set('forum_description', $description);
        forumJson(['success' => true]);
    }

    if ($action === 'add_category' || $action === 'edit_category') {
        $name = trim($_POST['name'] ?? '');
        $sortOrder = (int)($_POST['sort_order'] ?? 0);
        $isActive = !empty($_POST['is_active']) ? 1 : 0;
        if ($name === '') {
            forumJson(['success' => false, 'error' => 'Category name is required.'], 422);
        }

        $slug = forumSlug($name);
        $now = DbHelper::nowString();
        if ($action === 'add_category') {
            $stmt = $db->prepare("INSERT INTO forum_categories (name, slug, is_active, sort_order, created_at, updated_at) VALUES (?, ?, ?, ?, ?, ?)");
            $stmt->execute([$name, $slug, $isActive, $sortOrder, $now, $now]);
            $categoryId = (int)$db->lastInsertId();
        } else {
            $categoryId = (int)($_POST['category_id'] ?? 0);
            $stmt = $db->prepare("UPDATE forum_categories SET name = ?, slug = ?, is_active = ?, sort_order = ?, updated_at = ? WHERE id = ?");
            $stmt->execute([$name, $slug, $isActive, $sortOrder, $now, $categoryId]);
        }

        $payload = forumBuildPayload($db, $categoryId, 0, 1, $perPage, $defaultAvatar);
        forumJson(['success' => true] + $payload);
    }

    if ($action === 'delete_category') {
        $categoryId = (int)($_POST['category_id'] ?? 0);
        if ($categoryId > 0) {
            $stmt = $db->prepare("DELETE FROM forum_categories WHERE id = ?");
            $stmt->execute([$categoryId]);
        }
        $payload = forumBuildPayload($db, 0, 0, 1, $perPage, $defaultAvatar);
        forumJson(['success' => true] + $payload);
    }

    if ($action === 'add_subcategory' || $action === 'edit_subcategory') {
        $categoryId = (int)($_POST['category_id'] ?? 0);
        $parentId = isset($_POST['parent_id']) && $_POST['parent_id'] !== '' ? (int)$_POST['parent_id'] : null;
        $name = trim($_POST['name'] ?? '');
        $description = trim($_POST['description'] ?? '');
        $sortOrder = (int)($_POST['sort_order'] ?? 0);
        $isActive = !empty($_POST['is_active']) ? 1 : 0;
        if ($categoryId <= 0 || $name === '') {
            forumJson(['success' => false, 'error' => 'Subcategory name is required.'], 422);
        }

        $slug = forumSlug($name);
        $now = DbHelper::nowString();
        if ($action === 'add_subcategory') {
            $stmt = $db->prepare("INSERT INTO forum_subcategories (category_id, parent_id, name, slug, description, is_active, sort_order, created_at, updated_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
            $stmt->execute([$categoryId, $parentId, $name, $slug, $description, $isActive, $sortOrder, $now, $now]);
            $subcategoryId = (int)$db->lastInsertId();
        } else {
            $subcategoryId = (int)($_POST['subcategory_id'] ?? 0);
            $stmt = $db->prepare("UPDATE forum_subcategories SET parent_id = ?, name = ?, slug = ?, description = ?, is_active = ?, sort_order = ?, updated_at = ? WHERE id = ?");
            $stmt->execute([$parentId, $name, $slug, $description, $isActive, $sortOrder, $now, $subcategoryId]);
        }

        $payload = forumBuildPayload($db, $categoryId, $subcategoryId, 1, $perPage, $defaultAvatar);
        forumJson(['success' => true] + $payload);
    }

    if ($action === 'delete_subcategory') {
        $subcategoryId = (int)($_POST['subcategory_id'] ?? 0);
        $categoryId = forumGetCategoryIdForSubcategory($db, $subcategoryId);
        if ($subcategoryId > 0) {
            $stmt = $db->prepare("DELETE FROM forum_subcategories WHERE id = ?");
            $stmt->execute([$subcategoryId]);
        }
        $payload = forumBuildPayload($db, $categoryId, 0, 1, $perPage, $defaultAvatar);
        forumJson(['success' => true] + $payload);
    }

    if ($action === 'update_post_status') {
        $postId = (int)($_POST['post_id'] ?? 0);
        $status = trim($_POST['status'] ?? '');
        $status = in_array($status, ['approved', 'declined', 'pending'], true) ? $status : 'pending';
        $subcategoryId = (int)($_POST['subcategory_id'] ?? 0);
        if ($postId > 0) {
            $stmt = $db->prepare("UPDATE forum_posts SET status = ?, updated_at = ? WHERE id = ?");
            $stmt->execute([$status, DbHelper::nowString(), $postId]);
        }
        $payload = forumBuildPayload($db, forumGetCategoryIdForSubcategory($db, $subcategoryId), $subcategoryId, 1, $perPage, $defaultAvatar);
        forumJson(['success' => true] + $payload);
    }

    if ($action === 'delete_post') {
        $postId = (int)($_POST['post_id'] ?? 0);
        $subcategoryId = (int)($_POST['subcategory_id'] ?? 0);
        if ($postId > 0) {
            $stmt = $db->prepare("DELETE FROM forum_posts WHERE id = ?");
            $stmt->execute([$postId]);
        }
        $payload = forumBuildPayload($db, forumGetCategoryIdForSubcategory($db, $subcategoryId), $subcategoryId, 1, $perPage, $defaultAvatar);
        forumJson(['success' => true] + $payload);
    }

    forumJson(['success' => false, 'error' => 'Unknown action.'], 400);
}

$categories = forumGetCategories($db);
$activeCategoryId = forumResolveCategoryId($categories, (int)($_GET['category'] ?? 0));
$subcategories = forumGetSubcategories($db, $activeCategoryId);
$activeSubcategoryId = forumResolveSubcategoryId($subcategories, (int)($_GET['subcategory'] ?? 0));
$page = max(1, (int)($_GET['page'] ?? 1));
[$posts, $totalPosts] = forumGetPosts($db, $activeSubcategoryId, $page, $perPage);

AdminLayout::renderHeader();
?>

<div class="admin-content" data-forum-root>
    <div class="content-header content-header-actions">
        <div>
            <h1>Forum</h1>
            <p>Manage forum settings, categories, and posts</p>
        </div>
    </div>

    <div class="forum-settings card">
        <div class="card-header">
            <h3>Forum Settings</h3>
        </div>
        <div class="card-body">
            <form class="forum-form" data-forum-form="settings">
                <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                <input type="hidden" name="action" value="save_settings">
                <div class="form-row">
                    <label class="form-check">
                        <input type="checkbox" name="forum_enabled" value="1" <?php echo $forumEnabled ? 'checked' : ''; ?>>
                        <span>Enable forum</span>
                    </label>
                    <label class="form-check">
                        <input type="checkbox" name="forum_auto_approve" value="1" <?php echo $forumAutoApprove ? 'checked' : ''; ?>>
                        <span>Auto-approve new posts</span>
                    </label>
                </div>
                <div class="form-group">
                    <label for="forum-title-setting">Forum title</label>
                    <input id="forum-title-setting" type="text" class="form-control" name="forum_title" maxlength="120" required value="<?php echo e($forumTitleSetting); ?>">
                </div>
                <div class="form-group">
                    <label for="forum-description-setting">Forum description</label>
                    <textarea id="forum-description-setting" class="form-control" name="forum_description" rows="2" maxlength="200" required><?php echo e($forumDescriptionSetting); ?></textarea>
                </div>
                <div class="form-actions">
                    <button type="submit" class="btn btn-primary btn-sm">Save Settings</button>
                </div>
            </form>
        </div>
    </div>

    <div class="forum-layout">
        <div class="forum-column" data-forum-categories>
            <?php echo renderForumCategories($categories, $activeCategoryId); ?>
        </div>
        <div class="forum-column" data-forum-subcategories>
            <?php echo renderForumSubcategories($subcategories, $activeCategoryId, $activeSubcategoryId); ?>
        </div>
        <div class="forum-column forum-column--posts" data-forum-posts>
            <?php echo renderForumPosts($posts, $totalPosts, $page, $perPage, $activeSubcategoryId, $defaultAvatar); ?>
        </div>
    </div>
</div>

<script nonce="<?php echo e(getCspNonce()); ?>">
(function() {
    const root = document.querySelector('[data-forum-root]');
    if (!root) return;

    const csrfToken = document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '';

    const updateDom = (payload) => {
        if (payload.categories_html) {
            const node = document.querySelector('[data-forum-categories]');
            if (node) node.innerHTML = payload.categories_html;
        }
        if (payload.subcategories_html) {
            const node = document.querySelector('[data-forum-subcategories]');
            if (node) node.innerHTML = payload.subcategories_html;
        }
        if (payload.posts_html) {
            const node = document.querySelector('[data-forum-posts]');
            if (node) node.innerHTML = payload.posts_html;
        }
        bindForum();
    };

    const postForm = (form) => {
        const formData = new FormData(form);
        formData.append('ajax', '1');
        fetch('/admin/forum.php', {
            method: 'POST',
            headers: {
                'X-CSRF-Token': csrfToken
            },
            body: formData
        })
            .then(r => r.json())
            .then(data => {
                if (data.success) {
                    updateDom(data);
                }
            });
    };

    const loadSubcategories = (categoryId) => {
        fetch('/admin/forum.php?ajax=1&action=subcategories&category_id=' + categoryId)
            .then(r => r.json())
            .then(data => {
                if (data.success) {
                    updateDom(data);
                }
            });
    };

    const loadPosts = (subcategoryId, page) => {
        fetch('/admin/forum.php?ajax=1&action=posts&subcategory_id=' + subcategoryId + '&page=' + page)
            .then(r => r.json())
            .then(data => {
                if (data.success) {
                    updateDom(data);
                }
            });
    };

    const bindForum = () => {
        document.querySelectorAll('[data-forum-form]').forEach(form => {
            form.addEventListener('submit', (event) => {
                event.preventDefault();
                postForm(form);
            });
        });

        document.querySelectorAll('[data-forum-category]').forEach(item => {
            item.addEventListener('click', () => {
                const id = item.getAttribute('data-forum-category');
                if (id) {
                    loadSubcategories(id);
                }
            });
        });

        document.querySelectorAll('[data-forum-subcategory]').forEach(item => {
            item.addEventListener('click', () => {
                const id = item.getAttribute('data-forum-subcategory');
                if (id) {
                    loadPosts(id, 1);
                }
            });
        });

        document.querySelectorAll('[data-posts-page]').forEach(btn => {
            btn.addEventListener('click', () => {
                const page = btn.getAttribute('data-posts-page');
                const activeSub = document.querySelector('[data-forum-subcategory].is-active')?.getAttribute('data-forum-subcategory');
                if (activeSub && page) {
                    loadPosts(activeSub, page);
                }
            });
        });

        document.querySelectorAll('[data-category-edit]').forEach(btn => {
            btn.addEventListener('click', (event) => {
                event.stopPropagation();
                const form = document.querySelector('[data-forum-form="category"]');
                if (!form) return;
                form.querySelector('[data-category-action]').value = 'edit_category';
                form.querySelector('[data-category-id]').value = btn.getAttribute('data-category-id') || '';
                form.querySelector('[data-category-name]').value = btn.getAttribute('data-category-name') || '';
                form.querySelector('[data-category-sort]').value = btn.getAttribute('data-category-sort') || '0';
                form.querySelector('[data-category-active]').checked = btn.getAttribute('data-category-active') === '1';
            });
        });

        document.querySelectorAll('[data-category-delete]').forEach(btn => {
            btn.addEventListener('click', (event) => {
                event.stopPropagation();
                if (!confirm('Delete this category and all subcategories?')) return;
                const formData = new FormData();
                formData.append('ajax', '1');
                formData.append('action', 'delete_category');
                formData.append('category_id', btn.getAttribute('data-category-delete') || '');
                fetch('/admin/forum.php', {
                    method: 'POST',
                    headers: { 'X-CSRF-Token': csrfToken },
                    body: formData
                }).then(r => r.json()).then(data => {
                    if (data.success) {
                        updateDom(data);
                    }
                });
            });
        });

        document.querySelectorAll('[data-category-reset]').forEach(btn => {
            btn.addEventListener('click', () => {
                const form = document.querySelector('[data-forum-form="category"]');
                if (!form) return;
                form.querySelector('[data-category-action]').value = 'add_category';
                form.querySelector('[data-category-id]').value = '';
                form.querySelector('[data-category-name]').value = '';
                form.querySelector('[data-category-sort]').value = '0';
                form.querySelector('[data-category-active]').checked = true;
            });
        });

        document.querySelectorAll('[data-subcategory-edit]').forEach(btn => {
            btn.addEventListener('click', (event) => {
                event.stopPropagation();
                const form = document.querySelector('[data-forum-form="subcategory"]');
                if (!form) return;
                form.querySelector('[data-subcategory-action]').value = 'edit_subcategory';
                form.querySelector('[data-subcategory-id]').value = btn.getAttribute('data-subcategory-id') || '';
                form.querySelector('[data-subcategory-name]').value = btn.getAttribute('data-subcategory-name') || '';
                form.querySelector('[data-subcategory-description]').value = btn.getAttribute('data-subcategory-description') || '';
                form.querySelector('[data-subcategory-sort]').value = btn.getAttribute('data-subcategory-sort') || '0';
                form.querySelector('[data-subcategory-active]').checked = btn.getAttribute('data-subcategory-active') === '1';
                const parentSelect = form.querySelector('[data-subcategory-parent-select]');
                if (parentSelect) {
                    parentSelect.value = btn.getAttribute('data-subcategory-parent') || '';
                }
            });
        });

        document.querySelectorAll('[data-subcategory-add-child]').forEach(btn => {
            btn.addEventListener('click', (event) => {
                event.stopPropagation();
                const form = document.querySelector('[data-forum-form="subcategory"]');
                if (!form) return;
                form.querySelector('[data-subcategory-action]').value = 'add_subcategory';
                form.querySelector('[data-subcategory-id]').value = '';
                form.querySelector('[data-subcategory-name]').value = '';
                form.querySelector('[data-subcategory-description]').value = '';
                form.querySelector('[data-subcategory-sort]').value = '0';
                form.querySelector('[data-subcategory-active]').checked = true;
                const parentSelect = form.querySelector('[data-subcategory-parent-select]');
                if (parentSelect) {
                    parentSelect.value = btn.getAttribute('data-subcategory-add-child') || '';
                }
                form.querySelector('[data-subcategory-name]').focus();
            });
        });

        document.querySelectorAll('[data-subcategory-delete]').forEach(btn => {
            btn.addEventListener('click', (event) => {
                event.stopPropagation();
                if (!confirm('Delete this subcategory and all posts?')) return;
                const formData = new FormData();
                formData.append('ajax', '1');
                formData.append('action', 'delete_subcategory');
                formData.append('subcategory_id', btn.getAttribute('data-subcategory-delete') || '');
                fetch('/admin/forum.php', {
                    method: 'POST',
                    headers: { 'X-CSRF-Token': csrfToken },
                    body: formData
                }).then(r => r.json()).then(data => {
                    if (data.success) {
                        updateDom(data);
                    }
                });
            });
        });

        document.querySelectorAll('[data-subcategory-reset]').forEach(btn => {
            btn.addEventListener('click', () => {
                const form = document.querySelector('[data-forum-form="subcategory"]');
                if (!form) return;
                form.querySelector('[data-subcategory-action]').value = 'add_subcategory';
                form.querySelector('[data-subcategory-id]').value = '';
                form.querySelector('[data-subcategory-name]').value = '';
                form.querySelector('[data-subcategory-description]').value = '';
                form.querySelector('[data-subcategory-sort]').value = '0';
                form.querySelector('[data-subcategory-active]').checked = true;
                const parentSelect = form.querySelector('[data-subcategory-parent-select]');
                if (parentSelect) {
                    parentSelect.value = '';
                }
            });
        });

        document.querySelectorAll('[data-post-status]').forEach(btn => {
            btn.addEventListener('click', () => {
                const postId = btn.getAttribute('data-post-id');
                const status = btn.getAttribute('data-post-status');
                const activeSub = document.querySelector('[data-forum-subcategory].is-active')?.getAttribute('data-forum-subcategory');
                if (!postId || !status || !activeSub) return;
                const formData = new FormData();
                formData.append('ajax', '1');
                formData.append('action', 'update_post_status');
                formData.append('post_id', postId);
                formData.append('status', status);
                formData.append('subcategory_id', activeSub);
                fetch('/admin/forum.php', {
                    method: 'POST',
                    headers: { 'X-CSRF-Token': csrfToken },
                    body: formData
                }).then(r => r.json()).then(data => {
                    if (data.success) {
                        updateDom(data);
                    }
                });
            });
        });

        document.querySelectorAll('[data-post-delete]').forEach(btn => {
            btn.addEventListener('click', () => {
                if (!confirm('Remove this post?')) return;
                const postId = btn.getAttribute('data-post-delete');
                const activeSub = document.querySelector('[data-forum-subcategory].is-active')?.getAttribute('data-forum-subcategory');
                if (!postId || !activeSub) return;
                const formData = new FormData();
                formData.append('ajax', '1');
                formData.append('action', 'delete_post');
                formData.append('post_id', postId);
                formData.append('subcategory_id', activeSub);
                fetch('/admin/forum.php', {
                    method: 'POST',
                    headers: { 'X-CSRF-Token': csrfToken },
                    body: formData
                }).then(r => r.json()).then(data => {
                    if (data.success) {
                        updateDom(data);
                    }
                });
            });
        });
    };

    bindForum();
})();
</script>

<?php AdminLayout::renderFooter(); ?>
