<?php
/**
 * Admin Panel - File Manager
 */

if (!defined('ROOT_PATH')) {
    http_response_code(403);
    exit;
}

use NewSite\Admin\AdminLayout;
use NewSite\Auth\AdminAuth;
use NewSite\Database\DatabaseManager;
use NewSite\Database\DbHelper;
use NewSite\Logging\LogService;
use NewSite\Settings\SettingsService;
use NewSite\Upload\UploadService;

AdminAuth::requireLogin();

$db = DatabaseManager::getWriteConnection();
$message = '';
$error = '';

// Path prefixes for URL routing (served by front controller)
const ADMIN_FILE_PATH_PREFIX = '/admin-file/';
const ADMIN_FILE_THUMBNAILS_PREFIX = '/admin-file/thumbnails/';
const SITE_FILE_PATH_PREFIX = '/site-file/';
const FILES_REDIRECT_MESSAGE = 'Location: /admin/files.php?message=';
const SQL_DELETE_FILE_BY_ID = 'DELETE FROM files WHERE id = ?';
const MIME_JPEG = 'image/jpeg';
const MIME_PNG = 'image/png';
const MIME_GIF = 'image/gif';
const MIME_WEBP = 'image/webp';
const RASTER_IMAGE_MIMES = [MIME_JPEG, MIME_PNG, MIME_GIF, MIME_WEBP];

// Get PHP upload limits
$uploadMaxSize = ini_get('upload_max_filesize');
$postMaxSize = ini_get('post_max_size');
$memoryLimit = ini_get('memory_limit');
$videoThumbnailMode = SettingsService::get('video_thumbnail_mode', 'auto');
if (!in_array($videoThumbnailMode, ['auto', 'client', 'server', 'off'], true)) {
    $videoThumbnailMode = 'auto';
}
$allowClientThumb = in_array($videoThumbnailMode, ['auto', 'client'], true);
$allowServerThumb = in_array($videoThumbnailMode, ['auto', 'server'], true);
$productsForFilter = $db->query("SELECT id, name FROM products ORDER BY name ASC")->fetchAll();

// Upload directory — realpath() resolves symlinks and ../../ to canonical absolute path
$uploadDir = realpath(ROOT_PATH . '/data/admin_uploads');
if ($uploadDir === false) {
    // Directory missing — create it and resolve again
    $uploadDir = ROOT_PATH . '/data/admin_uploads';
    mkdir($uploadDir, 0755, true);
    file_put_contents($uploadDir . '/.htaccess', "Options -Indexes\n<FilesMatch \"\\.php$\">\n    Deny from all\n</FilesMatch>\nAddType application/octet-stream .php .php5 .phtml");
    $uploadDir = realpath($uploadDir) ?: $uploadDir;
}
$uploadDir .= '/';

// Thumbnail directory (child of resolved upload dir)
$thumbnailDir = $uploadDir . 'thumbnails/';
if (!is_dir($thumbnailDir)) {
    mkdir($thumbnailDir, 0755, true);
}

/**
 * Generate video thumbnail using FFmpeg.
 *
 * @param string $videoPath     Absolute path to the source video
 * @param string $videoFilename Original filename (used to derive thumb name)
 * @param string $thumbDir      Absolute path to thumbnail output directory (trailing slash)
 */
function generateVideoThumbnail(string $videoPath, string $videoFilename, string $thumbDir): ?string
{
    // Check if FFmpeg is available
    $ffmpegPath = trim(shell_exec('which ffmpeg 2>/dev/null') ?: '');
    if (empty($ffmpegPath)) {
        return null; // FFmpeg not available
    }

    $thumbnailFilename = pathinfo($videoFilename, PATHINFO_FILENAME) . '_thumb.jpg';
    $thumbnailPath = $thumbDir . $thumbnailFilename;

    // Extract frame at 1 second
    $command = sprintf(
        '%s -i %s -ss 00:00:01 -vframes 1 -vf scale=320:-1 %s 2>&1',
        escapeshellarg($ffmpegPath),
        escapeshellarg($videoPath),
        escapeshellarg($thumbnailPath)
    );

    exec($command, $output, $returnCode);

    if ($returnCode === 0 && file_exists($thumbnailPath)) {
        return ADMIN_FILE_THUMBNAILS_PREFIX . $thumbnailFilename;
    }

    return null;
}

/**
 * Generate image thumbnail using GD (max 320px wide, JPEG output).
 * Supports JPEG, PNG, GIF, WebP. Returns thumbnail path or null on failure.
 *
 * @param string $imagePath     Absolute path to the source image
 * @param string $imageFilename Original filename (used to derive thumb name)
 * @param string $mimeType      MIME type of the source image
 * @param string $thumbDir      Absolute path to thumbnail output directory (trailing slash)
 */
function generateImageThumbnail(string $imagePath, string $imageFilename, string $mimeType, string $thumbDir): ?string
{

    if (!extension_loaded('gd')) {
        return null;
    }

    $sourceImage = match ($mimeType) {
        MIME_JPEG => @imagecreatefromjpeg($imagePath),
        MIME_PNG  => @imagecreatefrompng($imagePath),
        MIME_GIF  => @imagecreatefromgif($imagePath),
        MIME_WEBP => function_exists('imagecreatefromwebp') ? @imagecreatefromwebp($imagePath) : false,
        default   => false,
    };

    if (!$sourceImage) {
        return null;
    }

    $origWidth = imagesx($sourceImage);
    $origHeight = imagesy($sourceImage);
    $maxThumbWidth = 320;
    $result = null;

    // Only generate thumbnail if image is wider than the max
    if ($origWidth > $maxThumbWidth) {
        $thumbWidth = $maxThumbWidth;
        $thumbHeight = max(1, (int)round(($origHeight / $origWidth) * $thumbWidth));

        $thumbImage = imagecreatetruecolor($thumbWidth, $thumbHeight);
        if ($thumbImage) {
            // Preserve transparency for PNG/GIF before resampling
            if ($mimeType === MIME_PNG || $mimeType === MIME_GIF) {
                imagealphablending($thumbImage, false);
                imagesavealpha($thumbImage, true);
                $transparent = imagecolorallocatealpha($thumbImage, 0, 0, 0, 127);
                imagefill($thumbImage, 0, 0, $transparent);
            }

            imagecopyresampled($thumbImage, $sourceImage, 0, 0, 0, 0, $thumbWidth, $thumbHeight, $origWidth, $origHeight);

            $thumbnailFilename = pathinfo($imageFilename, PATHINFO_FILENAME) . '_thumb.jpg';
            $thumbnailPath = $thumbDir . $thumbnailFilename;

            if (imagejpeg($thumbImage, $thumbnailPath, 80) && file_exists($thumbnailPath)) {
                $result = ADMIN_FILE_THUMBNAILS_PREFIX . $thumbnailFilename;
            }
        }
    }

    return $result;
}

// Handle file upload
if ($_SERVER['REQUEST_METHOD'] === 'POST' && empty($_FILES)) {
    $acceptHeader = $_SERVER['HTTP_ACCEPT'] ?? '';
    $acceptsJson = $acceptHeader !== '' && strpos($acceptHeader, 'application/json') !== false;
    $isAjax = isset($_POST['ajax']) || $acceptsJson;
    if ($isAjax) {
        $contentLength = (int)($_SERVER['CONTENT_LENGTH'] ?? 0);
        if ($contentLength > 0) {
            $error = 'Upload failed: file exceeds server limits (upload_max_filesize: ' . ini_get('upload_max_filesize')
                . ', post_max_size: ' . ini_get('post_max_size') . ').';
        } else {
            $error = 'Upload failed: no file received.';
        }
        header('Content-Type: application/json');
        echo json_encode([
            'success' => false,
            'message' => $error,
            'file' => null
        ]);
        exit;
    }
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['file'])) {
    // Log upload attempt
    LogService::add('info', 'File upload attempt', json_encode([
        'filename' => $_FILES['file']['name'] ?? 'unknown',
        'size' => $_FILES['file']['size'] ?? 0,
        'type' => $_FILES['file']['type'] ?? 'unknown',
        'error_code' => $_FILES['file']['error'] ?? -1
    ]));

    $file = $_FILES['file'];

    if ($file['error'] === UPLOAD_ERR_OK) {
        $allowedMap = [
            'jpg' => [MIME_JPEG],
            'jpeg' => [MIME_JPEG],
            'png' => [MIME_PNG],
            'gif' => [MIME_GIF],
            'webp' => [MIME_WEBP],
            'svg' => ['image/svg+xml'],
            'pdf' => ['application/pdf'],
            'txt' => ['text/plain'],
            'csv' => ['text/csv', 'application/vnd.ms-excel'],
            'json' => ['application/json', 'text/plain'],
            'xml' => ['application/xml', 'text/xml'],
            'zip' => ['application/zip', 'application/x-zip-compressed'],
            'rar' => ['application/vnd.rar', 'application/x-rar-compressed'],
            '7z' => ['application/x-7z-compressed'],
            'mp4' => ['video/mp4'],
            'webm' => ['video/webm'],
            'mp3' => ['audio/mpeg'],
            'ogg' => ['audio/ogg'],
            'wav' => ['audio/wav'],
            'woff' => ['font/woff'],
            'woff2' => ['font/woff2'],
            'ttf' => ['font/ttf', 'application/font-sfnt', 'application/x-font-ttf'],
            'otf' => ['font/otf', 'application/font-sfnt', 'application/x-font-opentype'],
            'css' => ['text/css'],
            'js' => ['application/javascript', 'text/javascript']
        ];

        $validation = UploadService::validateFile($file, $allowedMap, true);
        if (!$validation['success']) {
            $error = $validation['error'] ?: 'This file type is not allowed.';
            LogService::add('warning', 'File upload blocked', json_encode(['name' => $file['name'], 'mime' => $validation['mime'], 'ext' => $validation['ext']]));
        } else {
            try {
                // Generate unique filename
                $ext = $validation['ext'] ?: pathinfo($file['name'], PATHINFO_EXTENSION);
                $filename = uniqid() . '_' . time() . '.' . $ext;
                $filepath = $uploadDir . $filename;

                if (move_uploaded_file($file['tmp_name'], $filepath)) {
                    $thumbnailPath = null;

                    // Generate image thumbnail using GD
                    if (in_array($validation['mime'], RASTER_IMAGE_MIMES, true)) {
                        $thumbnailPath = generateImageThumbnail($filepath, $filename, $validation['mime'], $thumbnailDir);
                    }

                    // Generate video thumbnail
                    if (strpos($validation['mime'], 'video/') === 0) {
                        // First try client-side uploaded thumbnail (if enabled)
                        if ($allowClientThumb && isset($_FILES['thumbnail']) && $_FILES['thumbnail']['error'] === UPLOAD_ERR_OK) {
                            $thumbFilename = pathinfo($filename, PATHINFO_FILENAME) . '_thumb.jpg';
                            $thumbPath = $thumbnailDir . $thumbFilename;
                            if (move_uploaded_file($_FILES['thumbnail']['tmp_name'], $thumbPath)) {
                                $thumbnailPath = ADMIN_FILE_THUMBNAILS_PREFIX . $thumbFilename;
                            }
                        }

                        // Fallback to server-side FFmpeg if client-side failed
                        if (!$thumbnailPath && $allowServerThumb) {
                            $thumbnailPath = generateVideoThumbnail($filepath, $filename, $thumbnailDir);
                        }
                    }

                    // Save to database
                    $stmt = $db->prepare("INSERT INTO files (filename, original_name, file_path, file_type, file_size, thumbnail_path) VALUES (?, ?, ?, ?, ?, ?)");
                    $stmt->execute([$filename, $file['name'], ADMIN_FILE_PATH_PREFIX . $filename, $validation['mime'] ?: $file['type'], $file['size'], $thumbnailPath]);
                    $message = 'File uploaded successfully.';
                } else {
                    $error = 'Failed to upload file.';
                    LogService::add('error', 'File upload failed: move_uploaded_file error', json_encode(['name' => $file['name']]));
                }
            } catch (Exception $e) {
                $error = 'Failed to save file.';
                LogService::add('error', 'File upload failed: ' . $e->getMessage(), json_encode(['name' => $file['name']]));
            }
        }
    } else {
        // Map error codes to friendly messages
        $errorMessages = [
            UPLOAD_ERR_INI_SIZE => 'File exceeds upload_max_filesize (' . ini_get('upload_max_filesize') . ')',
            UPLOAD_ERR_FORM_SIZE => 'File exceeds MAX_FILE_SIZE from form',
            UPLOAD_ERR_PARTIAL => 'File was only partially uploaded',
            UPLOAD_ERR_NO_FILE => 'No file was uploaded',
            UPLOAD_ERR_NO_TMP_DIR => 'Missing temporary folder',
            UPLOAD_ERR_CANT_WRITE => 'Failed to write file to disk',
            UPLOAD_ERR_EXTENSION => 'A PHP extension stopped the file upload'
        ];

        $error = 'Upload error: ' . ($errorMessages[$file['error']] ?? 'Unknown error code ' . $file['error']);
        LogService::add('error', 'File upload error code: ' . $file['error'], json_encode(['name' => $file['name'] ?? 'unknown', 'error_message' => $error]));
    }

    // AJAX response
    if (isset($_POST['ajax'])) {
        header('Content-Type: application/json');
        echo json_encode([
            'success' => empty($error),
            'message' => $message ?: $error,
            'file' => empty($error) ? [
                'id' => DbHelper::lastInsertId($db, 'files'),
                'filename' => $filename,
                'original_name' => $file['name'],
                'path' => ADMIN_FILE_PATH_PREFIX . $filename,
                'file_path' => ADMIN_FILE_PATH_PREFIX . $filename,
                'type' => $file['type'],
                'file_type' => $file['type']
            ] : null
        ]);
        exit;
    }
}

// Handle bulk file delete
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_GET['action']) && $_GET['action'] === 'bulk-delete') {
    verifyCSRF();
    $rawIds = $_POST['bulk_ids'] ?? '';
    $ids = [];
    if (is_array($rawIds)) {
        $ids = $rawIds;
    } elseif (is_string($rawIds) && $rawIds !== '') {
        $ids = explode(',', $rawIds);
    }
    $ids = array_values(array_unique(array_filter(array_map(function ($id) {
        return (int)trim((string)$id);
    }, $ids), function ($id) {
        return $id > 0;
    })));

    if (empty($ids)) {
        header(FILES_REDIRECT_MESSAGE . urlencode('No files selected.'));
        exit;
    }

    $deletedCount = 0;
    $stmtSelect = $db->prepare("SELECT * FROM files WHERE id = ?");
    $stmtDelete = $db->prepare(SQL_DELETE_FILE_BY_ID);

    foreach ($ids as $fileId) {
        $stmtSelect->execute([$fileId]);
        $file = $stmtSelect->fetch();
        if (!$file) {
            continue;
        }

        $filepath = $uploadDir . $file['filename'];
        if (file_exists($filepath)) {
            unlink($filepath);
        }

        if (!empty($file['thumbnail_path'])) {
            $thumbnailFile = basename($file['thumbnail_path']);
            $thumbnailPath = $thumbnailDir . $thumbnailFile;
            if (file_exists($thumbnailPath)) {
                unlink($thumbnailPath);
            }
        }

        $stmtDelete->execute([$fileId]);
        $deletedCount++;
    }

    $message = 'Deleted ' . $deletedCount . ' file' . ($deletedCount === 1 ? '' : 's') . '.';
    header(FILES_REDIRECT_MESSAGE . urlencode($message));
    exit;
}

// Handle file delete
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_GET['action']) && $_GET['action'] === 'delete') {
    verifyCSRF();
    $fileId = (int)($_POST['id'] ?? 0);

    $stmt = $db->prepare("SELECT * FROM files WHERE id = ?");
    $stmt->execute([$fileId]);
    $file = $stmt->fetch();

    if ($file) {
        $filepath = $uploadDir . $file['filename'];
        if (file_exists($filepath)) {
            unlink($filepath);
        }

        // Delete thumbnail if exists
        if (!empty($file['thumbnail_path'])) {
            $thumbnailFile = basename($file['thumbnail_path']);
            $thumbnailPath = $thumbnailDir . $thumbnailFile;
            if (file_exists($thumbnailPath)) {
                unlink($thumbnailPath);
            }
        }

        $stmt = $db->prepare(SQL_DELETE_FILE_BY_ID);
        $stmt->execute([$fileId]);
        $message = 'File deleted.';
    }

    header(FILES_REDIRECT_MESSAGE . urlencode($message));
    exit;
}

// Handle mark-as-used / clear usage note
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_GET['action']) && $_GET['action'] === 'mark-usage') {
    verifyCSRF();
    $fileId = (int)($_POST['id'] ?? 0);
    $usageNote = trim((string)($_POST['usage_note'] ?? ''));

    if ($fileId > 0) {
        $noteValue = $usageNote !== '' ? $usageNote : null;
        $stmt = $db->prepare("UPDATE files SET manual_usage_note = ? WHERE id = ?");
        $stmt->execute([$noteValue, $fileId]);
        $message = $usageNote !== '' ? 'File marked as used.' : 'Usage note cleared.';
    }

    header(FILES_REDIRECT_MESSAGE . urlencode($message));
    exit;
}

// Handle bulk thumbnail generation for existing images without thumbnails
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_GET['action']) && $_GET['action'] === 'generate-thumbnails') {
    verifyCSRF();

    $mimeList = implode(',', array_map(function (string $m) use ($db) {
        return $db->quote($m);
    }, RASTER_IMAGE_MIMES));

    // Find images missing thumbnail in DB
    $missingStmt = $db->query(
        "SELECT id, filename, file_type, thumbnail_path FROM files
         WHERE file_type IN ({$mimeList})
         ORDER BY id ASC
         LIMIT 500"
    );
    $allImageRows = $missingStmt->fetchAll();

    // Filter to rows that need a thumbnail: path is empty OR file missing on disk
    $needsThumb = [];
    $clearStmt = $db->prepare("UPDATE files SET thumbnail_path = NULL WHERE id = ?");
    foreach ($allImageRows as $row) {
        $thumbPath = trim((string)($row['thumbnail_path'] ?? ''));
        if ($thumbPath === '') {
            $needsThumb[] = $row;
            continue;
        }
        // Check if thumbnail file actually exists on disk
        $thumbFile = basename($thumbPath);
        $thumbDisk = $thumbnailDir . $thumbFile;
        if (!is_file($thumbDisk)) {
            // DB says thumbnail exists but file is missing — clear and regenerate
            $clearStmt->execute([(int)$row['id']]);
            $needsThumb[] = $row;
        }
    }

    $generated = 0;
    $updateStmt = $db->prepare("UPDATE files SET thumbnail_path = ? WHERE id = ?");

    foreach ($needsThumb as $row) {
        $filePath = $uploadDir . $row['filename'];
        if (!is_file($filePath)) {
            continue;
        }
        $thumb = generateImageThumbnail($filePath, $row['filename'], $row['file_type'], $thumbnailDir);
        if ($thumb !== null) {
            $updateStmt->execute([$thumb, (int)$row['id']]);
            $generated++;
        }
    }

    $remaining = count($needsThumb) - $generated;
    $message = 'Generated ' . $generated . ' thumbnail' . ($generated !== 1 ? 's' : '') . '.';
    if ($remaining > 0) {
        $suffix = $remaining !== 1 ? 's' : '';
        $message .= ' ' . $remaining . ' image' . $suffix . ' skipped.';
    }

    $_SESSION['files_flash_message'] = $message;
    header('Location: /admin/files.php');
    exit;
}

// File filters and sorting
$fileTypeFilter = strtolower(trim((string)($_GET['type'] ?? 'all')));
$allowedTypeFilters = ['all', 'image', 'video', 'audio', 'document', 'archive', 'font', 'code', 'other'];
if (!in_array($fileTypeFilter, $allowedTypeFilters, true)) {
    $fileTypeFilter = 'all';
}

$usageScope = strtolower(trim((string)($_GET['scope'] ?? 'all')));
$allowedScopes = ['all', 'used_any', 'unused_any', 'manual_marked', 'used_by_product'];
if (!in_array($usageScope, $allowedScopes, true)) {
    $usageScope = 'all';
}

$productFilterId = max(0, (int)($_GET['product_id'] ?? 0));

$sort = strtolower(trim((string)($_GET['sort'] ?? 'newest')));
$sortMap = [
    'newest' => 'f.uploaded_at DESC, f.id DESC',
    'oldest' => 'f.uploaded_at ASC, f.id ASC',
    'name_asc' => 'f.original_name ASC, f.id ASC',
    'name_desc' => 'f.original_name DESC, f.id DESC',
    'type_asc' => "file_category ASC, f.original_name ASC",
    'type_desc' => "file_category DESC, f.original_name ASC",
    'size_desc' => 'f.file_size DESC NULLS LAST, f.id DESC',
    'size_asc' => 'f.file_size ASC NULLS LAST, f.id ASC',
];
if (!isset($sortMap[$sort])) {
    $sort = 'newest';
}

$fileCategoryCaseSql = "CASE
    WHEN f.file_type ILIKE 'image/%' THEN 'image'
    WHEN f.file_type ILIKE 'video/%' THEN 'video'
    WHEN f.file_type ILIKE 'audio/%' THEN 'audio'
    WHEN f.file_type ILIKE 'font/%' OR f.file_type IN ('application/font-sfnt', 'application/x-font-ttf', 'application/x-font-opentype') THEN 'font'
    WHEN f.file_type IN ('application/zip', 'application/x-zip-compressed', 'application/vnd.rar', 'application/x-rar-compressed', 'application/x-7z-compressed') THEN 'archive'
    WHEN f.file_type IN ('text/css', 'application/javascript', 'text/javascript') THEN 'code'
    WHEN f.file_type ILIKE 'text/%' OR f.file_type IN ('application/pdf', 'application/json', 'application/xml', 'text/xml', 'text/csv', 'application/vnd.ms-excel') THEN 'document'
    ELSE 'other'
END";

$siteFileReplaceExpr = "REPLACE(f.file_path, '" . ADMIN_FILE_PATH_PREFIX . "', '" . SITE_FILE_PATH_PREFIX . "')";

$pathUsageSql = "(
    p.media_url IN (f.file_path, {$siteFileReplaceExpr})
    OR p.download_url IN (f.file_path, {$siteFileReplaceExpr})
    OR pm.media_url IN (f.file_path, {$siteFileReplaceExpr})
    OR pv.download_url IN (f.file_path, {$siteFileReplaceExpr})
)";

$usedInProductsSql = "EXISTS (
    SELECT 1
    FROM products p
    LEFT JOIN product_media pm ON pm.product_id = p.id
    LEFT JOIN product_variants pv ON pv.product_id = p.id
    WHERE {$pathUsageSql}
)";

$usedInCollectionsSql = "EXISTS (
    SELECT 1
    FROM collections c
    WHERE c.image_url IN (f.file_path, {$siteFileReplaceExpr})
)";

$usedInThemeSettingsSql = "EXISTS (
    SELECT 1
    FROM theme_settings ts
    WHERE ts.setting_value IN (f.file_path, {$siteFileReplaceExpr})
)";

$usedAnySql = "({$usedInProductsSql} OR {$usedInCollectionsSql} OR {$usedInThemeSettingsSql} OR (f.manual_usage_note IS NOT NULL AND f.manual_usage_note <> ''))";

$usedByProductSql = "EXISTS (
    SELECT 1
    FROM products p
    LEFT JOIN product_media pm ON pm.product_id = p.id
    LEFT JOIN product_variants pv ON pv.product_id = p.id
    WHERE p.id = :product_filter_id
      AND {$pathUsageSql}
)";

$whereParts = [];
$params = [];

if ($fileTypeFilter !== 'all') {
    $whereParts[] = "({$fileCategoryCaseSql}) = :file_type";
    $params[':file_type'] = $fileTypeFilter;
}

if ($usageScope === 'used_any') {
    $whereParts[] = $usedAnySql;
} elseif ($usageScope === 'unused_any') {
    $whereParts[] = "NOT ({$usedAnySql})";
} elseif ($usageScope === 'manual_marked') {
    $whereParts[] = "(f.manual_usage_note IS NOT NULL AND f.manual_usage_note <> '')";
} elseif ($usageScope === 'used_by_product') {
    if ($productFilterId > 0) {
        $whereParts[] = $usedByProductSql;
        $params[':product_filter_id'] = $productFilterId;
    } else {
        $whereParts[] = '1 = 0';
        $error = 'Select a product to use the "Used by product" filter.';
    }
}

$whereSql = '';
if (!empty($whereParts)) {
    $whereSql = ' WHERE ' . implode(' AND ', $whereParts);
}

// Get all files with pagination
$perPage = 50;
$page = max(1, (int)($_GET['page'] ?? 1));
$offset = ($page - 1) * $perPage;

$countSql = "SELECT COUNT(*) FROM files f{$whereSql}";
$countStmt = $db->prepare($countSql);
foreach ($params as $paramName => $paramValue) {
    if ($paramName === ':product_filter_id') {
        $countStmt->bindValue($paramName, (int)$paramValue, PDO::PARAM_INT);
    } else {
        $countStmt->bindValue($paramName, $paramValue, PDO::PARAM_STR);
    }
}
$countStmt->execute();
$totalFiles = (int)$countStmt->fetchColumn();
$totalPages = max(1, (int)ceil($totalFiles / $perPage));
if ($page > $totalPages) {
    $page = $totalPages;
    $offset = ($page - 1) * $perPage;
}

$listSql = "SELECT
    f.*,
    {$fileCategoryCaseSql} AS file_category,
    CASE WHEN {$usedAnySql} THEN 1 ELSE 0 END AS is_used_any
FROM files f
{$whereSql}
ORDER BY {$sortMap[$sort]}
LIMIT :limit OFFSET :offset";

$listStmt = $db->prepare($listSql);
foreach ($params as $paramName => $paramValue) {
    if ($paramName === ':product_filter_id') {
        $listStmt->bindValue($paramName, (int)$paramValue, PDO::PARAM_INT);
    } else {
        $listStmt->bindValue($paramName, $paramValue, PDO::PARAM_STR);
    }
}
$listStmt->bindValue(':limit', $perPage, PDO::PARAM_INT);
$listStmt->bindValue(':offset', $offset, PDO::PARAM_INT);
$listStmt->execute();
$files = $listStmt->fetchAll();

$hasActiveFileFilters = ($fileTypeFilter !== 'all')
    || ($usageScope !== 'all')
    || ($sort !== 'newest')
    || ($productFilterId > 0);

// Remove entries for missing files
if (!empty($files)) {
    $filtered = [];
    $stmtDelete = $db->prepare(SQL_DELETE_FILE_BY_ID);
    foreach ($files as $file) {
        $filepath = $uploadDir . $file['filename'];
        if (is_file($filepath)) {
            $file['file_category'] = (string)($file['file_category'] ?? 'other');
            $filtered[] = $file;
        } else {
            $stmtDelete->execute([$file['id']]);
        }
    }
    $files = $filtered;
}

// Check for message (query string or session flash)
if (isset($_GET['message'])) {
    $message = $_GET['message'];
}
if (isset($_SESSION['files_flash_message'])) {
    $message = $_SESSION['files_flash_message'];
    unset($_SESSION['files_flash_message']);
}

AdminLayout::renderHeader();
?>

<div class="admin-content">
    <div class="content-header content-header-actions">
        <div>
            <h1>File Manager</h1>
            <p>Manage uploaded files and media</p>
        </div>
        <form method="POST" action="/admin/files.php?action=generate-thumbnails" class="inline-block">
            <input type="hidden" name="csrf_token" value="<?php echo getCSRFToken(); ?>">
            <button type="submit" class="btn btn-secondary btn-sm">Generate Missing Thumbnails</button>
        </form>
    </div>

    <?php if ($message): ?>
    <div class="alert alert-success"><?php echo e($message); ?></div>
    <?php endif; ?>

    <?php if ($error): ?>
    <div class="alert alert-error"><?php echo e($error); ?></div>
    <?php endif; ?>

    <!-- Upload Zone -->
    <div class="card">
        <div class="card-body">
            <form id="upload-form" method="POST" enctype="multipart/form-data">
                <input type="hidden" name="csrf_token" value="<?php echo getCSRFToken(); ?>">
                <div class="upload-zone" id="upload-zone">
                    <input type="file" name="file" id="file-input" hidden multiple>
                    <div class="upload-zone-icon">📁</div>
                    <p>Drag & drop files here or click to browse</p>
                    <p class="text-muted upload-hint">Supports: Most file types (dangerous types are blocked)</p>
                    <p class="text-muted upload-hint mt-8">
                        <strong>Admin Unlimited Uploads</strong> | Server limits: <?php echo e($uploadMaxSize); ?> / <?php echo e($postMaxSize); ?>
                    </p>
                </div>
            </form>
            <div id="upload-progress" class="upload-progress is-hidden">
                <progress id="progress-bar" class="upload-progress-bar" value="0" max="100"></progress>
                <p class="text-muted progress-text" id="progress-text">Uploading...</p>
            </div>
        </div>
    </div>

    <!-- Files Grid -->
    <div class="card">
        <div class="card-header file-manager-header">
            <h3>Uploaded Files (<?php echo $totalFiles; ?>)</h3>
            <form method="POST" action="/admin/files.php?action=bulk-delete" class="file-bulk-actions" id="bulk-delete-form" data-confirm="Delete selected files?">
                <input type="hidden" name="csrf_token" value="<?php echo getCSRFToken(); ?>">
                <input type="hidden" name="bulk_ids" id="bulk-delete-ids" value="">
                <label class="form-check file-select-all">
                    <input type="checkbox" id="file-select-all">
                    <span>Select all</span>
                </label>
                <span class="file-selection-count" id="bulk-delete-count">0 selected</span>
                <button type="submit" class="btn btn-danger btn-sm" id="bulk-delete-btn" disabled>Delete Selected</button>
            </form>
        </div>
        <div class="card-body">
            <form method="GET" action="/admin/files.php" class="file-filter-bar" aria-label="File filters">
                <div class="file-filter-field">
                    <label for="files-sort">Sort</label>
                    <select id="files-sort" name="sort" class="form-control form-control-sm">
                        <option value="newest" <?php echo $sort === 'newest' ? 'selected' : ''; ?>>Newest first</option>
                        <option value="oldest" <?php echo $sort === 'oldest' ? 'selected' : ''; ?>>Oldest first</option>
                        <option value="name_asc" <?php echo $sort === 'name_asc' ? 'selected' : ''; ?>>Name A-Z</option>
                        <option value="name_desc" <?php echo $sort === 'name_desc' ? 'selected' : ''; ?>>Name Z-A</option>
                        <option value="type_asc" <?php echo $sort === 'type_asc' ? 'selected' : ''; ?>>Type A-Z</option>
                        <option value="type_desc" <?php echo $sort === 'type_desc' ? 'selected' : ''; ?>>Type Z-A</option>
                        <option value="size_desc" <?php echo $sort === 'size_desc' ? 'selected' : ''; ?>>Largest first</option>
                        <option value="size_asc" <?php echo $sort === 'size_asc' ? 'selected' : ''; ?>>Smallest first</option>
                    </select>
                </div>

                <div class="file-filter-field">
                    <label for="files-type">File type</label>
                    <select id="files-type" name="type" class="form-control form-control-sm">
                        <option value="all" <?php echo $fileTypeFilter === 'all' ? 'selected' : ''; ?>>All types</option>
                        <option value="image" <?php echo $fileTypeFilter === 'image' ? 'selected' : ''; ?>>Images</option>
                        <option value="video" <?php echo $fileTypeFilter === 'video' ? 'selected' : ''; ?>>Videos</option>
                        <option value="audio" <?php echo $fileTypeFilter === 'audio' ? 'selected' : ''; ?>>Audio</option>
                        <option value="document" <?php echo $fileTypeFilter === 'document' ? 'selected' : ''; ?>>Documents</option>
                        <option value="archive" <?php echo $fileTypeFilter === 'archive' ? 'selected' : ''; ?>>Archives</option>
                        <option value="font" <?php echo $fileTypeFilter === 'font' ? 'selected' : ''; ?>>Fonts</option>
                        <option value="code" <?php echo $fileTypeFilter === 'code' ? 'selected' : ''; ?>>Code</option>
                        <option value="other" <?php echo $fileTypeFilter === 'other' ? 'selected' : ''; ?>>Other</option>
                    </select>
                </div>

                <div class="file-filter-field">
                    <label for="files-scope">Usage</label>
                    <select id="files-scope" name="scope" class="form-control form-control-sm">
                        <option value="all" <?php echo $usageScope === 'all' ? 'selected' : ''; ?>>All files</option>
                        <option value="used_any" <?php echo $usageScope === 'used_any' ? 'selected' : ''; ?>>Used files (any product)</option>
                        <option value="unused_any" <?php echo $usageScope === 'unused_any' ? 'selected' : ''; ?>>Unused files</option>
                        <option value="manual_marked" <?php echo $usageScope === 'manual_marked' ? 'selected' : ''; ?>>Manually marked</option>
                        <option value="used_by_product" <?php echo $usageScope === 'used_by_product' ? 'selected' : ''; ?>>Used by product</option>
                    </select>
                </div>

                <div class="file-filter-field file-filter-field--product">
                    <label for="files-product">Product</label>
                    <select id="files-product" name="product_id" class="form-control form-control-sm">
                        <option value="0">All products</option>
                        <?php foreach ($productsForFilter as $filterProduct): ?>
                        <option value="<?php echo (int)$filterProduct['id']; ?>" <?php echo $productFilterId === (int)$filterProduct['id'] ? 'selected' : ''; ?>>
                            <?php echo e($filterProduct['name'] ?? ('Product #' . (int)$filterProduct['id'])); ?>
                        </option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div class="file-filter-actions">
                    <button type="submit" class="btn btn-primary btn-sm">Apply</button>
                    <a href="/admin/files.php" class="btn btn-secondary btn-sm">Reset</a>
                </div>
            </form>

            <?php if (empty($files)): ?>
            <p class="text-muted text-center empty-state"><?php echo $hasActiveFileFilters ? 'No files match the current filters.' : 'No files uploaded yet.'; ?></p>
            <?php else: ?>
            <div class="file-grid" id="file-grid">
                <?php foreach ($files as $file): ?>
                <?php $publicFilePath = str_replace(ADMIN_FILE_PATH_PREFIX, SITE_FILE_PATH_PREFIX, (string)($file['file_path'] ?? '')); ?>
                <div class="file-item" data-id="<?php echo $file['id']; ?>">
                    <label class="file-select">
                        <input type="checkbox" class="file-select-checkbox" data-file-id="<?php echo $file['id']; ?>" aria-label="Select file <?php echo e($file['original_name']); ?>">
                    </label>
                    <?php if (strpos($file['file_type'], 'image/') === 0): ?>
                    <?php
                    $imgThumbSrc = !empty($file['thumbnail_path']) ? $file['thumbnail_path'] : $file['file_path'];
                        ?>
                    <img data-src="<?php echo e($imgThumbSrc); ?>" alt="<?php echo e($file['original_name']); ?>" class="file-preview lazy-image" src="data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 1 1'%3E%3C/svg%3E">
                    <?php elseif (strpos($file['file_type'], 'video/') === 0): ?>
                        <?php if (!empty($file['thumbnail_path'])): ?>
                        <img data-src="<?php echo e($file['thumbnail_path']); ?>" alt="<?php echo e($file['original_name']); ?>" class="file-preview lazy-image" src="data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 1 1'%3E%3C/svg%3E">
                        <div class="video-play-icon">▶</div>
                        <?php else: ?>
                        <div class="file-icon">🎥</div>
                        <?php endif; ?>
                    <?php else: ?>
                    <div class="file-icon">📄</div>
                    <?php endif; ?>
                    <div class="file-name"><?php echo e($file['original_name']); ?></div>
                    <div class="file-meta"><?php echo e(ucfirst((string)($file['file_category'] ?? 'other'))); ?><?php if (!empty($file['is_used_any'])): ?> &bull; Used<?php else: ?> &bull; Unused<?php endif; ?></div>
                    <?php $manualNote = trim((string)($file['manual_usage_note'] ?? '')); ?>
                    <div class="file-usage-note">
                        <?php if ($manualNote !== ''): ?>
                        <span class="file-usage-badge">Used by: <?php echo e($manualNote); ?></span>
                        <?php endif; ?>
                        <form method="POST" action="/admin/files.php?action=mark-usage" class="file-usage-form">
                            <input type="hidden" name="csrf_token" value="<?php echo getCSRFToken(); ?>">
                            <input type="hidden" name="id" value="<?php echo (int)$file['id']; ?>">
                            <input type="text" name="usage_note" class="form-control form-control-sm" value="<?php echo e($manualNote); ?>" placeholder="Mark used by&hellip;" maxlength="250">
                            <button type="submit" class="btn btn-secondary btn-sm"><?php echo $manualNote !== '' ? 'Update' : 'Save'; ?></button>
                            <?php if ($manualNote !== ''): ?>
                            <button type="submit" class="btn btn-danger btn-sm" name="usage_note" value="">Clear</button>
                            <?php endif; ?>
                        </form>
                    </div>
                    <div class="file-actions">
                        <a class="btn btn-primary btn-sm" href="<?php echo e($file['file_path']); ?>" download>Download</a>
                        <button class="btn btn-secondary btn-sm" data-action="copy-file-url" data-file-public-path="<?php echo e($publicFilePath); ?>">Create Link</button>
                        <form method="POST" action="/admin/files.php?action=delete" class="inline-block" data-confirm="Delete this file?">
                            <input type="hidden" name="id" value="<?php echo $file['id']; ?>">
                            <input type="hidden" name="csrf_token" value="<?php echo getCSRFToken(); ?>">
                            <button type="submit" class="btn btn-danger btn-sm has-tooltip" data-tooltip="Permanently delete this file.">Delete</button>
                        </form>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>

            <?php if ($totalPages > 1): ?>
            <div class="pagination logs-pagination">
                <?php
                $paginationParams = $_GET;
                unset($paginationParams['page']);
                $buildPageUrl = function (int $targetPage) use ($paginationParams): string {
                    $params = $paginationParams;
                    $params['page'] = $targetPage;
                    return '/admin/files.php?' . http_build_query($params);
                };
                ?>
                <?php if ($page > 1): ?>
                <a href="<?php echo e($buildPageUrl($page - 1)); ?>" class="btn btn-secondary btn-sm">&laquo; Prev</a>
                <?php endif; ?>

                <?php
                $start = max(1, $page - 2);
                $end = min($totalPages, $page + 2);
                if ($start > 1): ?>
                <a href="<?php echo e($buildPageUrl(1)); ?>" class="btn btn-secondary btn-sm">1</a>
                <?php if ($start > 2): ?><span class="pagination-dots">...</span><?php endif; ?>
                <?php endif; ?>

                <?php for ($i = $start; $i <= $end; $i++): ?>
                <a href="<?php echo e($buildPageUrl($i)); ?>" class="btn <?php echo $i === $page ? 'btn-primary' : 'btn-secondary'; ?> btn-sm"><?php echo $i; ?></a>
                <?php endfor; ?>

                <?php if ($end < $totalPages): ?>
                <?php if ($end < $totalPages - 1): ?><span class="pagination-dots">...</span><?php endif; ?>
                <a href="<?php echo e($buildPageUrl($totalPages)); ?>" class="btn btn-secondary btn-sm"><?php echo $totalPages; ?></a>
                <?php endif; ?>

                <?php if ($page < $totalPages): ?>
                <a href="<?php echo e($buildPageUrl($page + 1)); ?>" class="btn btn-secondary btn-sm">Next &raquo;</a>
                <?php endif; ?>
            </div>
            <?php endif; ?>

            <?php endif; ?>
        </div>
    </div>
</div>

<script nonce="<?php echo e(getCspNonce()); ?>">
const uploadZone = document.getElementById('upload-zone');
const fileInput = document.getElementById('file-input');
const uploadProgress = document.getElementById('upload-progress');
const progressBar = document.getElementById('progress-bar');
const progressText = document.getElementById('progress-text');
const fileGrid = document.getElementById('file-grid');
const bulkForm = document.getElementById('bulk-delete-form');
const bulkIdsInput = document.getElementById('bulk-delete-ids');
const bulkDeleteBtn = document.getElementById('bulk-delete-btn');
const bulkCount = document.getElementById('bulk-delete-count');
const selectAllToggle = document.getElementById('file-select-all');
const filesVideoThumbnailMode = <?php echo json_encode($videoThumbnailMode); ?>;
const filesAllowClientThumbnail = filesVideoThumbnailMode === 'auto' || filesVideoThumbnailMode === 'client';

if (uploadZone && fileInput) {
    if (!uploadZone.dataset.bound) {
        uploadZone.dataset.bound = '1';

        uploadZone.addEventListener('click', (e) => {
            if (e.target === fileInput) return;
            if (fileInput.dataset.dialogOpen === '1') return;
            fileInput.dataset.dialogOpen = '1';
            fileInput.click();
            setTimeout(() => { fileInput.dataset.dialogOpen = '0'; }, 300);
        });

        uploadZone.addEventListener('dragover', (e) => {
            e.preventDefault();
            uploadZone.classList.add('dragover');
        });

        uploadZone.addEventListener('dragleave', () => {
            uploadZone.classList.remove('dragover');
        });

        uploadZone.addEventListener('drop', (e) => {
            e.preventDefault();
            uploadZone.classList.remove('dragover');
            if (e.dataTransfer.files.length) {
                handleFiles(e.dataTransfer.files);
            }
        });
    }
}

fileInput.addEventListener('change', () => {
    if (fileInput.files.length) {
        handleFiles(fileInput.files);
    }
});

function getSelectedFileIds() {
    const selected = Array.from(document.querySelectorAll('.file-select-checkbox:checked'));
    return selected.map((checkbox) => checkbox.getAttribute('data-file-id')).filter(Boolean);
}

function updateBulkSelection() {
    const selectedIds = getSelectedFileIds();
    if (bulkIdsInput) {
        bulkIdsInput.value = selectedIds.join(',');
    }
    if (bulkCount) {
        bulkCount.textContent = selectedIds.length + ' selected';
    }
    if (bulkDeleteBtn) {
        bulkDeleteBtn.disabled = selectedIds.length === 0;
    }

    const allCheckboxes = Array.from(document.querySelectorAll('.file-select-checkbox'));
    const allChecked = allCheckboxes.length > 0 && allCheckboxes.every((checkbox) => checkbox.checked);
    if (selectAllToggle) {
        selectAllToggle.checked = allChecked;
    }

    if (fileGrid) {
        fileGrid.querySelectorAll('.file-item').forEach((item) => {
            const checkbox = item.querySelector('.file-select-checkbox');
            item.classList.toggle('selected', !!(checkbox && checkbox.checked));
        });
    }
}

if (fileGrid) {
    fileGrid.addEventListener('click', (e) => {
        const copyBtn = e.target.closest('[data-action="copy-file-url"]');
        if (copyBtn) {
            e.preventDefault();
            const path = copyBtn.getAttribute('data-file-public-path') || copyBtn.getAttribute('data-file-path');
            if (path) {
                copyToClipboard(path, copyBtn);
            }
            return;
        }

        if (e.target.closest('button, a, form, input, label')) {
            return;
        }

        const item = e.target.closest('.file-item');
        if (!item) return;
        const checkbox = item.querySelector('.file-select-checkbox');
        if (!checkbox) return;
        checkbox.checked = !checkbox.checked;
        updateBulkSelection();
    });
}

document.querySelectorAll('.file-select-checkbox').forEach((checkbox) => {
    checkbox.addEventListener('change', updateBulkSelection);
});

if (selectAllToggle) {
    selectAllToggle.addEventListener('change', () => {
        const shouldCheck = selectAllToggle.checked;
        document.querySelectorAll('.file-select-checkbox').forEach((checkbox) => {
            checkbox.checked = shouldCheck;
        });
        updateBulkSelection();
    });
}

if (bulkForm) {
    bulkForm.addEventListener('submit', (e) => {
        const selectedIds = getSelectedFileIds();
        if (!selectedIds.length) {
            e.preventDefault();
            alert('Select at least one file.');
        }
    });
}

updateBulkSelection();

function handleFiles(files) {
    Array.from(files).forEach(file => {
        // Check if it's a video and generate client-side thumbnail
        if (file.type.startsWith('video/') && filesAllowClientThumbnail) {
            generateClientSideThumbnail(file, (thumbnailBlob) => {
                uploadFile(file, thumbnailBlob);
            });
        } else {
            uploadFile(file, null);
        }
    });
}

function generateClientSideThumbnail(videoFile, callback) {
    const video = document.createElement('video');
    const canvas = document.createElement('canvas');
    const ctx = canvas.getContext('2d');

    video.preload = 'metadata';
    video.muted = true;

    video.onloadeddata = () => {
        // Seek to 1 second or 10% of duration
        video.currentTime = Math.min(1, video.duration * 0.1);
    };

    video.onseeked = () => {
        // Set canvas size
        canvas.width = 320;
        canvas.height = (video.videoHeight / video.videoWidth) * 320;

        // Draw video frame to canvas
        ctx.drawImage(video, 0, 0, canvas.width, canvas.height);

        // Convert to blob
        canvas.toBlob((blob) => {
            URL.revokeObjectURL(video.src);
            callback(blob);
        }, 'image/jpeg', 0.85);
    };

    video.onerror = () => {
        URL.revokeObjectURL(video.src);
        callback(null); // Fallback to no thumbnail
    };

    video.src = URL.createObjectURL(videoFile);
}

function uploadFile(file, thumbnailBlob) {
    const formData = new FormData();
    formData.append('file', file);
    formData.append('ajax', '1');

    // Add client-side generated thumbnail if available
    if (thumbnailBlob) {
        formData.append('thumbnail', thumbnailBlob, 'thumbnail.jpg');
    }

    // Add CSRF token from the form
    const csrfToken = document.querySelector('input[name="csrf_token"]');
    if (csrfToken) {
        formData.append('csrf_token', csrfToken.value);
    }

    const xhr = new XMLHttpRequest();

    uploadProgress.classList.remove('is-hidden');

    xhr.upload.addEventListener('progress', (e) => {
        if (e.lengthComputable) {
            const percent = (e.loaded / e.total) * 100;
            progressBar.value = percent;
            progressText.textContent = 'Uploading ' + file.name + '... ' + Math.round(percent) + '%';
        }
    });

    xhr.addEventListener('load', () => {
        uploadProgress.classList.add('is-hidden');
        progressBar.value = 0;

        if (xhr.status === 200) {
            try {
                const response = JSON.parse(xhr.responseText);
                if (response.success) {
                    location.reload();
                } else {
                    alert('Upload failed: ' + (response.message || 'Unknown error'));
                    console.error('Upload error:', response);
                }
            } catch (e) {
                alert('Upload failed: Invalid server response');
                console.error('Parse error:', e, 'Response:', xhr.responseText);
            }
        } else {
            alert('Upload failed: Server returned status ' + xhr.status);
            console.error('XHR error:', xhr.status, xhr.statusText, xhr.responseText);
        }
    });

    xhr.addEventListener('error', () => {
        uploadProgress.classList.add('is-hidden');
        progressBar.value = 0;
        alert('Upload failed: Network error');
        console.error('Network error during upload');
    });

    // Use current page URL to avoid routing issues
    xhr.open('POST', window.location.pathname);
    xhr.setRequestHeader('Accept', 'application/json');
    xhr.send(formData);
}

function showFilesNotification(message, type, durationMs) {
    if (typeof showNotification === 'function') {
        showNotification(message, type, durationMs);
        return;
    }

    const fallback = document.createElement('div');
    fallback.className = 'notification notification-' + (type || 'success');
    fallback.innerHTML = '<span>' + message + '</span>';
    document.body.appendChild(fallback);
    setTimeout(() => {
        fallback.remove();
    }, Number.isFinite(durationMs) ? durationMs : 2000);
}

function setCreateLinkButtonState(button, success) {
    if (!button) {
        return;
    }

    const defaultText = (button.dataset.defaultText || button.textContent || 'Create Link').trim();
    button.dataset.defaultText = defaultText;

    const previousTimer = Number(button.dataset.resetTimerId || 0);
    if (previousTimer) {
        window.clearTimeout(previousTimer);
    }

    if (success) {
        button.classList.add('btn-copy-success');
        button.classList.remove('btn-copy-error');
        button.textContent = 'Copied ✓';
    } else {
        button.classList.add('btn-copy-error');
        button.classList.remove('btn-copy-success');
        button.textContent = 'Copy Failed';
    }

    const resetTimerId = window.setTimeout(() => {
        button.classList.remove('btn-copy-success', 'btn-copy-error');
        button.textContent = button.dataset.defaultText || 'Create Link';
        delete button.dataset.resetTimerId;
    }, 2000);

    button.dataset.resetTimerId = String(resetTimerId);
}

function fallbackCopyText(text) {
    const helper = document.createElement('textarea');
    helper.value = text;
    helper.setAttribute('readonly', 'readonly');
    helper.style.position = 'fixed';
    helper.style.top = '-1000px';
    helper.style.opacity = '0';
    document.body.appendChild(helper);

    helper.focus();
    helper.select();
    helper.setSelectionRange(0, helper.value.length);

    let copied = false;
    try {
        copied = document.execCommand('copy');
    } catch (err) {
        copied = false;
    }

    document.body.removeChild(helper);
    return copied;
}

async function copyToClipboard(text, sourceButton) {
    const normalizedText = (text || '').trim();
    if (!normalizedText) {
        return;
    }

    const isAbsoluteUrl = /^https?:\/\//i.test(normalizedText);
    const fullUrl = isAbsoluteUrl ? normalizedText : (window.location.origin + normalizedText);
    let copied = false;

    if (navigator.clipboard && window.isSecureContext) {
        try {
            await navigator.clipboard.writeText(fullUrl);
            copied = true;
        } catch (err) {
            copied = false;
        }
    }

    if (!copied) {
        copied = fallbackCopyText(fullUrl);
    }

    if (copied) {
        showFilesNotification('Share link copied to clipboard!', 'success', 2000);
        setCreateLinkButtonState(sourceButton, true);
        return;
    }

    setCreateLinkButtonState(sourceButton, false);
    showFilesNotification('Copy failed. A manual copy prompt will open.', 'error', 3500);
    window.prompt('Copy this share link:', fullUrl);
}
</script>

<?php AdminLayout::renderFooter(); ?>
