<?php
/**
 * Admin Panel - DMCA Requests
 */

if (!defined('ROOT_PATH')) {
    http_response_code(403);
    exit;
}

use NewSite\Auth\AdminAuth;
use NewSite\Settings\SettingsService;
use NewSite\Contact\ContactService;
use NewSite\Template\ContentRenderer;
use NewSite\Email\SmtpMailer;
use NewSite\Database\DbHelper;
use NewSite\Logging\LogService;
use NewSite\Admin\AdminLayout;
use NewSite\Database\DatabaseManager;

AdminAuth::requireLogin();

$db = DatabaseManager::getWriteConnection();
$message = $_GET['message'] ?? '';
$error = $_GET['error'] ?? '';

$allowedStatuses = ['pending', 'verified', 'in_review', 'actioned', 'rejected', 'closed'];

function dmcaStatusBadgeClass(string $status): string
{
    switch ($status) {
        case 'actioned':
            return 'status-active';
        case 'verified':
        case 'in_review':
            return 'status-info';
        case 'pending':
        case 'rejected':
        case 'closed':
        default:
            return 'status-inactive';
    }
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    verifyCSRF();
    $action = $_POST['action'] ?? '';
    if ($action === 'update_status') {
        $id = (int)($_POST['id'] ?? 0);
        $status = trim($_POST['status'] ?? '');
        if ($id > 0 && in_array($status, $allowedStatuses, true)) {
            if ($status === 'verified') {
                $now = DbHelper::nowString();
                $stmt = $db->prepare("UPDATE dmca_requests SET status = ?, verified_at = COALESCE(verified_at, ?) WHERE id = ?");
                $stmt->execute([$status, $now, $id]);
            } else {
                $stmt = $db->prepare("UPDATE dmca_requests SET status = ? WHERE id = ?");
                $stmt->execute([$status, $id]);
            }
            LogService::add('info', 'DMCA status updated', json_encode(['id' => $id, 'status' => $status]));

            $notifyStatuses = ['in_review', 'rejected', 'actioned', 'closed'];
            if (in_array($status, $notifyStatuses, true)) {
                $stmt = $db->prepare("SELECT name, email FROM dmca_requests WHERE id = ? LIMIT 1");
                $stmt->execute([$id]);
                $requestRow = $stmt->fetch(PDO::FETCH_ASSOC);
                $toEmail = trim((string)($requestRow['email'] ?? ''));
                if ($toEmail !== '' && filter_var($toEmail, FILTER_VALIDATE_EMAIL)) {
                    $smtp = [
                        'host' => SettingsService::get('smtp_host', ''),
                        'port' => SettingsService::get('smtp_port', '587'),
                        'user' => SettingsService::get('smtp_user', ''),
                        'pass' => SettingsService::get('smtp_pass', ''),
                        'secure' => SettingsService::get('smtp_secure', 'tls')
                    ];
                    if ($smtp['host'] !== '' && $smtp['user'] !== '' && $smtp['pass'] !== '') {
                        $contactEmail = trim((string)ContactService::getEmail());
                        $replyTo = '';
                        $replyName = '';
                        if ($contactEmail !== '' && filter_var($contactEmail, FILTER_VALIDATE_EMAIL)) {
                            $replyTo = $contactEmail;
                            $replyName = trim((string)SettingsService::get('site_name', ''));
                        }

                        $statusLabels = [
                            'in_review' => 'In Review',
                            'rejected' => 'Rejected',
                            'actioned' => 'Actioned',
                            'closed' => 'Closed'
                        ];
                        $statusLabel = $statusLabels[$status] ?? ucfirst(str_replace('_', ' ', $status));
                        $siteName = trim((string)SettingsService::get('site_name', ''));
                        $recipientName = trim((string)($requestRow['name'] ?? ''));
                        $vars = [
                            '{name}' => $recipientName,
                            '{request_id}' => (string)$id,
                            '{status}' => $status,
                            '{status_label}' => $statusLabel,
                            '{contact_email}' => $contactEmail,
                            '{site_name}' => $siteName
                        ];

                        $templates = [
                            'in_review' => [
                                'subject_key' => 'dmca_status_in_review_subject',
                                'body_key' => 'dmca_status_in_review_body',
                                'subject_default' => 'DMCA request in review',
                                'body_default' => "Hello {name},\n\nYour DMCA request (ID {request_id}) is now in review.\n\nWe will contact you if we need more information.\n\nThank you,\n{site_name}"
                            ],
                            'rejected' => [
                                'subject_key' => 'dmca_status_rejected_subject',
                                'body_key' => 'dmca_status_rejected_body',
                                'subject_default' => 'DMCA request rejected',
                                'body_default' => "Hello {name},\n\nYour DMCA request (ID {request_id}) has been rejected.\n\nIf you believe this is a mistake, contact {contact_email}.\n\nThank you,\n{site_name}"
                            ],
                            'actioned' => [
                                'subject_key' => 'dmca_status_actioned_subject',
                                'body_key' => 'dmca_status_actioned_body',
                                'subject_default' => 'DMCA request actioned',
                                'body_default' => "Hello {name},\n\nYour DMCA request (ID {request_id}) has been actioned.\n\nThank you,\n{site_name}"
                            ],
                            'closed' => [
                                'subject_key' => 'dmca_status_closed_subject',
                                'body_key' => 'dmca_status_closed_body',
                                'subject_default' => 'DMCA request closed',
                                'body_default' => "Hello {name},\n\nYour DMCA request (ID {request_id}) has been closed.\n\nThank you,\n{site_name}"
                            ]
                        ];

                        if (isset($templates[$status])) {
                            $subjectTemplate = SettingsService::get($templates[$status]['subject_key'], $templates[$status]['subject_default']);
                            $bodyTemplate = SettingsService::get($templates[$status]['body_key'], $templates[$status]['body_default']);
                            $finalSubject = ContentRenderer::renderEmailTemplate($subjectTemplate, $vars);
                            $finalBody = ContentRenderer::renderEmailTemplate($bodyTemplate, $vars);
                            $sent = SmtpMailer::send($toEmail, $finalSubject, $finalBody, $replyTo, $replyName, $smtp);
                            if ($sent) {
                                LogService::add('info', 'DMCA status email sent', json_encode(['id' => $id, 'status' => $status]));
                            } else {
                                LogService::add('error', 'DMCA status email failed', json_encode(['id' => $id, 'status' => $status]));
                            }
                        }
                    }
                }
            }

            header('Location: /admin/dmca.php?message=' . urlencode('Status updated.'));
            exit;
        }
        header('Location: /admin/dmca.php?error=' . urlencode('Invalid request.'));
        exit;
    }

    if ($action === 'delete_request') {
        $id = (int)($_POST['id'] ?? 0);
        if ($id > 0) {
            $stmt = $db->prepare("DELETE FROM dmca_requests WHERE id = ?");
            $stmt->execute([$id]);
            LogService::add('info', 'DMCA request deleted', json_encode(['id' => $id]));
            header('Location: /admin/dmca.php?message=' . urlencode('Request deleted.'));
            exit;
        }
        header('Location: /admin/dmca.php?error=' . urlencode('Invalid request.'));
        exit;
    }
}

$statusFilter = $_GET['status'] ?? '';
if (!in_array($statusFilter, $allowedStatuses, true)) {
    $statusFilter = '';
}

$page = max(1, (int)($_GET['page'] ?? 1));
$perPage = 50;
$offset = ($page - 1) * $perPage;

$whereSql = '';
$params = [];
if ($statusFilter !== '') {
    $whereSql = 'WHERE status = ?';
    $params[] = $statusFilter;
}

$requestsStmt = $db->prepare("SELECT * FROM dmca_requests $whereSql ORDER BY created_at DESC LIMIT ? OFFSET ?");
foreach ($params as $i => $value) {
    $requestsStmt->bindValue($i + 1, $value);
}
$requestsStmt->bindValue(count($params) + 1, $perPage, PDO::PARAM_INT);
$requestsStmt->bindValue(count($params) + 2, $offset, PDO::PARAM_INT);
$requestsStmt->execute();
$requests = $requestsStmt->fetchAll(PDO::FETCH_ASSOC);

$countStmt = $db->prepare("SELECT COUNT(*) FROM dmca_requests $whereSql");
foreach ($params as $i => $value) {
    $countStmt->bindValue($i + 1, $value);
}
$countStmt->execute();
$totalRequests = (int)$countStmt->fetchColumn();
$totalPages = (int)ceil($totalRequests / $perPage);

$statusCounts = [];
$statusStmt = $db->query("SELECT status, COUNT(*) AS total FROM dmca_requests GROUP BY status");
foreach ($statusStmt->fetchAll(PDO::FETCH_ASSOC) as $row) {
    $statusCounts[$row['status']] = (int)$row['total'];
}

$viewId = (int)($_GET['id'] ?? 0);
$viewRequest = null;
if ($viewId > 0) {
    $stmt = $db->prepare("SELECT * FROM dmca_requests WHERE id = ? LIMIT 1");
    $stmt->execute([$viewId]);
    $viewRequest = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$viewRequest) {
        $error = 'Request not found.';
    }
}

AdminLayout::renderHeader();
?>

<div class="admin-content">
    <div class="content-header content-header-actions">
        <div>
            <h1>DMCA Requests</h1>
            <p>Review and manage DMCA takedown requests</p>
        </div>
    </div>

    <?php if ($message): ?>
    <div class="alert alert-success"><?php echo e($message); ?></div>
    <?php endif; ?>

    <?php if ($error): ?>
    <div class="alert alert-error"><?php echo e($error); ?></div>
    <?php endif; ?>

    <div class="stats-grid stats-grid-spaced">
        <div class="stat-card">
            <div class="stat-icon">DM</div>
            <div class="stat-info">
                <h3><?php echo number_format($totalRequests); ?></h3>
                <p>Total Requests</p>
            </div>
        </div>
        <div class="stat-card">
            <div class="stat-icon">PN</div>
            <div class="stat-info">
                <h3><?php echo number_format($statusCounts['pending'] ?? 0); ?></h3>
                <p>Pending</p>
            </div>
        </div>
        <div class="stat-card">
            <div class="stat-icon">VR</div>
            <div class="stat-info">
                <h3><?php echo number_format($statusCounts['verified'] ?? 0); ?></h3>
                <p>Verified</p>
            </div>
        </div>
        <div class="stat-card">
            <div class="stat-icon">AC</div>
            <div class="stat-info">
                <h3><?php echo number_format($statusCounts['actioned'] ?? 0); ?></h3>
                <p>Actioned</p>
            </div>
        </div>
    </div>

    <div class="card">
        <div class="card-body">
            <form method="GET" action="" class="d-flex gap-1 align-center">
                <label class="mr-10" for="dmca-status-filter">Filter by status:</label>
                <select id="dmca-status-filter" name="status" class="form-control width-200">
                    <option value="">All</option>
                    <?php foreach ($allowedStatuses as $statusOption): ?>
                    <option value="<?php echo e($statusOption); ?>" <?php echo $statusFilter === $statusOption ? 'selected' : ''; ?>>
                        <?php echo e(ucwords(str_replace('_', ' ', $statusOption))); ?>
                    </option>
                    <?php endforeach; ?>
                </select>
                <button type="submit" class="btn btn-secondary btn-sm">Filter</button>
                <?php if ($statusFilter !== ''): ?>
                <a href="/admin/dmca.php" class="btn btn-secondary btn-sm">Clear</a>
                <?php endif; ?>
            </form>
        </div>
    </div>

    <?php if ($viewRequest): ?>
    <div class="card">
        <div class="card-header">
            <h3>Request #<?php echo (int)$viewRequest['id']; ?></h3>
        </div>
        <div class="card-body">
            <div class="table-container">
                <table class="data-table">
                    <tbody>
                        <tr>
                            <th>Status</th>
                            <td>
                                <span class="status-badge <?php echo e(dmcaStatusBadgeClass((string)$viewRequest['status'])); ?>">
                                    <?php echo e($viewRequest['status']); ?>
                                </span>
                            </td>
                        </tr>
                        <tr>
                            <th>Name</th>
                            <td><?php echo e($viewRequest['name'] ?? ''); ?></td>
                        </tr>
                        <tr>
                            <th>Email</th>
                            <td><?php echo e($viewRequest['email'] ?? ''); ?></td>
                        </tr>
                        <tr>
                            <th>Address</th>
                            <td><?php echo nl2br(e($viewRequest['address'] ?? ''), false); ?></td>
                        </tr>
                        <tr>
                            <th>Phone</th>
                            <td><?php echo e($viewRequest['phone'] ?? ''); ?></td>
                        </tr>
                        <tr>
                            <th>Work Description</th>
                            <td><?php echo nl2br(e($viewRequest['work_description'] ?? ''), false); ?></td>
                        </tr>
                        <tr>
                            <th>Infringing URLs</th>
                            <td>
                                <?php
                                $urlLines = preg_split('/\r\n|\r|\n/', (string)($viewRequest['infringing_urls'] ?? ''));
        $urlLines = array_filter(array_map('trim', $urlLines));
        if (empty($urlLines)) {
            echo '<span class="text-muted">No URLs provided.</span>';
        } else {
            echo '<ul>';
            foreach ($urlLines as $line) {
                $safeLine = e($line);
                if (filter_var($line, FILTER_VALIDATE_URL)) {
                    echo '<li><a href="' . $safeLine . '" target="_blank" rel="noopener noreferrer">' . $safeLine . '</a></li>';
                } else {
                    echo '<li>' . $safeLine . '</li>';
                }
            }
            echo '</ul>';
        }
?>
                            </td>
                        </tr>
                        <tr>
                            <th>Signature</th>
                            <td><?php echo e($viewRequest['signature'] ?? ''); ?></td>
                        </tr>
                        <tr>
                            <th>Statements</th>
                            <td>
                                <div>Good-faith: <?php echo !empty($viewRequest['good_faith']) ? 'Yes' : 'No'; ?></div>
                                <div>Perjury: <?php echo !empty($viewRequest['perjury']) ? 'Yes' : 'No'; ?></div>
                                <div>Authorized: <?php echo !empty($viewRequest['authorized']) ? 'Yes' : 'No'; ?></div>
                            </td>
                        </tr>
                        <tr>
                            <th>Submitted</th>
                            <td><?php echo e($viewRequest['created_at'] ?? ''); ?></td>
                        </tr>
                        <tr>
                            <th>Verified</th>
                            <td><?php echo e($viewRequest['verified_at'] ?? ''); ?></td>
                        </tr>
                        <tr>
                            <th>IP</th>
                            <td><?php echo e($viewRequest['ip_address'] ?? ''); ?></td>
                        </tr>
                        <tr>
                            <th>User Agent</th>
                            <td><?php echo e($viewRequest['user_agent'] ?? ''); ?></td>
                        </tr>
                    </tbody>
                </table>
            </div>

            <div class="d-flex gap-1 align-center mt-16">
                <form method="POST" action="/admin/dmca.php" class="d-flex gap-1 align-center">
                    <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                    <input type="hidden" name="action" value="update_status">
                    <input type="hidden" name="id" value="<?php echo (int)$viewRequest['id']; ?>">
                    <select name="status" class="form-control width-200">
                        <?php foreach ($allowedStatuses as $statusOption): ?>
                        <option value="<?php echo e($statusOption); ?>" <?php echo $viewRequest['status'] === $statusOption ? 'selected' : ''; ?>>
                            <?php echo e(ucwords(str_replace('_', ' ', $statusOption))); ?>
                        </option>
                        <?php endforeach; ?>
                    </select>
                    <button type="submit" class="btn btn-primary">Update Status</button>
                </form>
                <form method="POST" action="/admin/dmca.php" data-confirm="Delete this DMCA request?">
                    <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                    <input type="hidden" name="action" value="delete_request">
                    <input type="hidden" name="id" value="<?php echo (int)$viewRequest['id']; ?>">
                    <button type="submit" class="btn btn-danger">Delete</button>
                </form>
            </div>
        </div>
    </div>
    <?php endif; ?>

    <div class="table-container">
        <table class="data-table">
            <thead>
                <tr>
                    <th>ID</th>
                    <th>Status</th>
                    <th>Name</th>
                    <th>Email</th>
                    <th>Submitted</th>
                    <th>Verified</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php if (empty($requests)): ?>
                <tr>
                    <td colspan="7" class="text-center text-muted">No DMCA requests found.</td>
                </tr>
                <?php else: ?>
                <?php foreach ($requests as $request): ?>
                <tr>
                    <td><?php echo (int)$request['id']; ?></td>
                    <td>
                        <span class="status-badge <?php echo e(dmcaStatusBadgeClass((string)$request['status'])); ?>">
                            <?php echo e($request['status']); ?>
                        </span>
                    </td>
                    <td><?php echo e($request['name'] ?? ''); ?></td>
                    <td><?php echo e($request['email'] ?? ''); ?></td>
                    <td><?php echo e($request['created_at'] ?? ''); ?></td>
                    <td><?php echo e($request['verified_at'] ?? ''); ?></td>
                    <td class="table-actions">
                        <a href="/admin/dmca.php?id=<?php echo (int)$request['id']; ?>" class="btn btn-secondary btn-sm">View</a>
                    </td>
                </tr>
                <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
    </div>

    <?php if ($totalPages > 1): ?>
    <div class="pagination">
        <?php for ($i = 1; $i <= $totalPages; $i++): ?>
            <?php
            $query = ['page' => $i];
            if ($statusFilter !== '') {
                $query['status'] = $statusFilter;
            }
            $url = '/admin/dmca.php?' . http_build_query($query);
            ?>
            <a href="<?php echo e($url); ?>" class="pagination-link <?php echo $page === $i ? 'active' : ''; ?>"><?php echo $i; ?></a>
        <?php endfor; ?>
    </div>
    <?php endif; ?>
</div>

<?php AdminLayout::renderFooter(); ?>
