<?php
/**
 * Admin Panel - Currency Rates
 */

if (!defined('ROOT_PATH')) {
    http_response_code(403);
    exit;
}

use NewSite\Auth\AdminAuth;
use NewSite\Settings\SettingsService;
use NewSite\Shop\CurrencyService;
use NewSite\Util\CurrencyData;
use NewSite\Admin\AdminLayout;
use NewSite\Database\DatabaseManager;

AdminAuth::requireLogin();

$db = DatabaseManager::getWriteConnection();
$message = '';
$error = '';

$storeCurrency = SettingsService::get('store_currency', 'USD');
$currencyMap = CurrencyData::getMap();
$rates = CurrencyService::getRates();

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';

    if ($action === 'save_rates') {
        $submitted = $_POST['rates'] ?? [];
        $normalized = [];
        foreach ($currencyMap as $code => $symbol) {
            $inputValue = trim((string)($submitted[$code] ?? ''));
            if ($code === $storeCurrency) {
                $normalized[$code] = 1.0;
                continue;
            }
            if ($inputValue === '') {
                continue;
            }
            $rate = (float)str_replace(',', '.', $inputValue);
            if ($rate <= 0) {
                $error = 'Rates must be greater than 0.';
                break;
            }
            $normalized[$code] = $rate;
        }

        if ($error === '') {
            SettingsService::set('currency_rates', json_encode($normalized));
            $rates = CurrencyService::getRates();
            $message = 'Currency rates updated.';
        }
    }

    if ($action === 'reset_rates') {
        SettingsService::set('currency_rates', '');
        $rates = CurrencyService::getRates();
        $message = 'Currency rates reset to defaults.';
    }
}

AdminLayout::renderHeader();
?>

<div class="admin-content">
    <div class="content-header">
        <h1>Currency Rates</h1>
        <p>Set conversion rates relative to the store currency (<?php echo e($storeCurrency); ?>).</p>
    </div>

    <?php if ($message): ?>
        <div class="alert alert-success"><?php echo e($message); ?></div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="alert alert-error"><?php echo e($error); ?></div>
    <?php endif; ?>

    <div class="card">
        <div class="card-header">
            <h3>Rates</h3>
        </div>
        <div class="card-body">
            <p class="text-muted">Example: if store currency is USD and EUR rate is 0.92, then $10 shows as €9.20.</p>
            <form method="POST" action="/admin/currency-rates.php">
                <input type="hidden" name="action" value="save_rates">
                <div class="form-row">
                    <?php foreach ($currencyMap as $code => $symbol):
                        $currentRate = $rates[$code] ?? '';
                        $isBase = $code === $storeCurrency;
                        ?>
                        <div class="form-group currency-rate-field">
                            <label for="rate-<?php echo e($code); ?>"><?php echo e($code . ' (' . $symbol . ')'); ?></label>
                            <input
                                type="text"
                                id="rate-<?php echo e($code); ?>"
                                name="rates[<?php echo e($code); ?>]"
                                class="form-control"
                                value="<?php
                                    if ($isBase) {
                                        echo e('1.0');
                                    } elseif ($currentRate !== '') {
                                        echo e($currentRate);
                                    }
                        ?>"
                                placeholder="<?php echo e($isBase ? '1.0' : '0.00'); ?>"
                                <?php echo $isBase ? 'readonly' : ''; ?>
                            >
                        </div>
                    <?php endforeach; ?>
                </div>
                <div class="table-actions">
                    <button type="submit" class="btn btn-primary">Save Rates</button>
                    <button type="submit" name="action" value="reset_rates" class="btn btn-outline" data-confirm="Reset to default rates?">Reset Defaults</button>
                </div>
            </form>
        </div>
    </div>
</div>

<?php AdminLayout::renderFooter(); ?>
