<?php
/**
 * Admin Panel - Collections
 */

if (!defined('ROOT_PATH')) {
    http_response_code(403);
    exit;
}

use NewSite\Admin\AdminLayout;
use NewSite\Auth\AdminAuth;
use NewSite\Config\SetupService;
use NewSite\Database\DatabaseManager;
use NewSite\Database\DbHelper;
use NewSite\Logging\LogService;
use NewSite\Settings\SettingsService;
use NewSite\Util\SlugGenerator;

AdminAuth::requireLogin();

$db = DatabaseManager::getWriteConnection();
$action = $_GET['action'] ?? 'list';
$message = '';
$error = '';

if (!function_exists('buildUniqueCollectionSlug')) {
    function buildUniqueCollectionSlug(PDO $db, string $slugInput, string $title, int $excludeId = 0): string
    {
        $baseSlug = SlugGenerator::generate($slugInput);
        if ($baseSlug === '') {
            $baseSlug = SlugGenerator::generate($title);
        }
        if ($baseSlug === '') {
            $baseSlug = 'collection';
        }

        $candidate = $baseSlug;
        $suffix = 2;
        $existsStmt = $db->prepare("SELECT id FROM collections WHERE collection_slug = ? AND id <> ? LIMIT 1");

        while (true) {
            $existsStmt->execute([$candidate, $excludeId]);
            if (!$existsStmt->fetch()) {
                return $candidate;
            }
            $candidate = $baseSlug . '-' . $suffix;
            $suffix++;
        }
    }
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_GET['action'] ?? 'save';

    if ($action === 'delete') {
        verifyCSRF();
        $id = $_POST['id'] ?? null;
        if ($id) {
            $stmt = $db->prepare("DELETE FROM collections WHERE id = ?");
            $stmt->execute([$id]);
            header('Location: /admin/collections.php?message=' . urlencode('Collection deleted.'));
            exit;
        }
    }

    $collectionId = $_POST['id'] ?? null;
    $title = trim($_POST['title'] ?? '');
    $collectionSlugInput = trim((string)($_POST['collection_slug'] ?? ''));
    $description = trim($_POST['description'] ?? '');
    $metaTitle = trim(strip_tags((string)($_POST['meta_title'] ?? '')));
    $metaDescription = trim(strip_tags((string)($_POST['meta_description'] ?? '')));
    $imageUrl = trim($_POST['image_url'] ?? '');
    $targetPageId = isset($_POST['target_page_id']) && $_POST['target_page_id'] !== ''
        ? (int)$_POST['target_page_id']
        : null;

    if ($title === '') {
        $error = 'Collection title is required.';
    } else {
        try {
            verifyCSRF();
            $existingSlug = '';
            if ($collectionId) {
                $existingStmt = $db->prepare("SELECT collection_slug FROM collections WHERE id = ? LIMIT 1");
                $existingStmt->execute([$collectionId]);
                $existingSlug = trim((string)$existingStmt->fetchColumn());
            }
            $slugSeed = $collectionSlugInput !== '' ? $collectionSlugInput : $existingSlug;
            $collectionSlug = buildUniqueCollectionSlug($db, $slugSeed, $title, (int)$collectionId);

            if ($collectionId) {
                $now = DbHelper::nowString();
                $stmt = $db->prepare("UPDATE collections SET title = ?, collection_slug = ?, description = ?, meta_title = ?, meta_description = ?, image_url = ?, target_page_id = ?, updated_at = ? WHERE id = ?");
                $stmt->execute([$title, $collectionSlug, $description, $metaTitle, $metaDescription, $imageUrl, $targetPageId, $now, $collectionId]);
                $message = 'Collection updated successfully.';
            } else {
                $stmt = $db->prepare("INSERT INTO collections (title, collection_slug, description, meta_title, meta_description, image_url, target_page_id) VALUES (?, ?, ?, ?, ?, ?, ?)");
                $stmt->execute([$title, $collectionSlug, $description, $metaTitle, $metaDescription, $imageUrl, $targetPageId]);
                $message = 'Collection created successfully.';
            }

            if (!$error) {
                header('Location: /admin/collections.php?message=' . urlencode($message));
                exit;
            }
        } catch (PDOException $e) {
            $error = 'Database error occurred while saving the collection.';
            LogService::add('error', 'Collection save failed: ' . $e->getMessage(), json_encode(['collection_id' => $collectionId, 'title' => $title]));
        }
    }
}

if ($action === 'delete' && isset($_GET['id'])) {
    // Moved to POST for CSRF protection
    header('Location: /admin/collections.php');
    exit;
}

$collection = null;
if ($action === 'edit' && isset($_GET['id'])) {
    $stmt = $db->prepare("SELECT * FROM collections WHERE id = ?");
    $stmt->execute([$_GET['id']]);
    $collection = $stmt->fetch();
    if (!$collection) {
        header('Location: /admin/collections.php');
        exit;
    }
}

$collections = $db->query("SELECT * FROM collections ORDER BY created_at DESC")->fetchAll();

$collectionPages = $db->query("SELECT DISTINCT p.id, p.title, p.slug FROM pages p JOIN sections s ON s.page_id = p.id WHERE s.section_type = 'products_list' AND p.is_published = 1 ORDER BY p.title ASC")->fetchAll();
$collectionPageMap = [];
foreach ($collectionPages as $page) {
    $collectionPageMap[(int)$page['id']] = $page;
}
$defaultProductsPageId = null;
foreach ($collectionPages as $page) {
    if (($page['slug'] ?? '') === 'products') {
        $defaultProductsPageId = (int)$page['id'];
        break;
    }
}
if ($defaultProductsPageId === null && !empty($collectionPages)) {
    $defaultProductsPageId = (int)$collectionPages[0]['id'];
}

$themeSettings = SettingsService::getAllTheme();
$siteName = is_array($themeSettings) ? trim((string)($themeSettings['site_name'] ?? '')) : '';

$formTitle = trim((string)($collection['title'] ?? ''));
$formCollectionSlug = trim((string)($collection['collection_slug'] ?? ''));
$formDescription = trim((string)($collection['description'] ?? ''));
$formMetaTitle = trim((string)($collection['meta_title'] ?? ''));
$formMetaDescription = trim((string)($collection['meta_description'] ?? ''));
$formImageUrl = trim((string)($collection['image_url'] ?? ''));
$formTargetPageId = (int)($collection['target_page_id'] ?? 0);

if (!$collection && $formTargetPageId === 0 && $defaultProductsPageId) {
    $formTargetPageId = $defaultProductsPageId;
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && $error !== '') {
    $formTitle = trim((string)($_POST['title'] ?? $formTitle));
    $formCollectionSlug = trim((string)($_POST['collection_slug'] ?? $formCollectionSlug));
    $formDescription = trim((string)($_POST['description'] ?? $formDescription));
    $formMetaTitle = trim(strip_tags((string)($_POST['meta_title'] ?? $formMetaTitle)));
    $formMetaDescription = trim(strip_tags((string)($_POST['meta_description'] ?? $formMetaDescription)));
    $formImageUrl = trim((string)($_POST['image_url'] ?? $formImageUrl));

    if (array_key_exists('target_page_id', $_POST)) {
        $postedTargetPageId = (string)($_POST['target_page_id'] ?? '');
        $formTargetPageId = $postedTargetPageId !== '' ? (int)$postedTargetPageId : 0;
    }
}

$previewSlug = '';
if ($formTargetPageId > 0 && isset($collectionPageMap[$formTargetPageId])) {
    $previewSlug = trim((string)($collectionPageMap[$formTargetPageId]['slug'] ?? ''));
}
if ($previewSlug === '' && $defaultProductsPageId && isset($collectionPageMap[$defaultProductsPageId])) {
    $previewSlug = trim((string)($collectionPageMap[$defaultProductsPageId]['slug'] ?? ''));
}
if ($previewSlug === '') {
    $previewSlug = 'products';
}

$previewCollectionToken = $formCollectionSlug !== ''
    ? SlugGenerator::generate($formCollectionSlug)
    : SlugGenerator::generate($formTitle);
if ($previewCollectionToken === '') {
    $previewCollectionToken = 'collection-slug';
}
$previewPath = '/' . ltrim($previewSlug, '/') . '/collection/' . $previewCollectionToken;
$previewUrl = rtrim(SetupService::getBaseUrl(), '/') . $previewPath;

$previewTitleBase = $formMetaTitle;
if ($previewTitleBase === '') {
    $previewTitleBase = $formTitle !== '' ? $formTitle : 'Collection';
}
$previewTitle = $siteName !== '' ? ($previewTitleBase . ' | ' . $siteName) : $previewTitleBase;

$previewDescription = $formMetaDescription !== '' ? $formMetaDescription : $formDescription;
if ($previewDescription === '' && $formTitle !== '') {
    $previewDescription = 'Browse products in ' . $formTitle . '.';
}
if ($previewDescription === '') {
    $previewDescription = 'Browse this collection.';
}
$previewDescription = preg_replace('/\s+/', ' ', strip_tags($previewDescription));
$previewDescription = trim((string)$previewDescription);
if (function_exists('mb_substr')) {
    $previewDescription = mb_substr($previewDescription, 0, 180);
} else {
    $previewDescription = substr($previewDescription, 0, 180);
}

if (isset($_GET['message'])) {
    $message = $_GET['message'];
}

AdminLayout::renderHeader();
?>

<div class="admin-content">
    <?php if ($action === 'list'): ?>
    <div class="content-header content-header-actions">
        <div>
            <h1>Collections</h1>
            <p>Manage product collections</p>
        </div>
        <a href="/admin/collections.php?action=new" class="btn btn-primary">+ New Collection</a>
    </div>

    <?php if ($message): ?>
    <div class="alert alert-success"><?php echo e($message); ?></div>
    <?php endif; ?>

    <div class="table-container">
        <table class="data-table">
            <thead>
                <tr>
                    <th>Title</th>
                    <th>Description</th>
                    <th>Updated</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($collections as $c): ?>
                <tr>
                    <td><?php echo e($c['title']); ?></td>
                    <td><?php echo e($c['description']); ?></td>
                    <td><?php echo date('M j, Y', strtotime($c['updated_at'] ?? $c['created_at'])); ?></td>
                    <td class="table-actions">
                        <a href="/admin/collections.php?action=edit&id=<?php echo $c['id']; ?>" class="btn btn-secondary btn-sm">Edit</a>
                        <form method="POST" action="/admin/collections.php?action=delete" class="inline-block" data-confirm="Delete this collection?">
                            <input type="hidden" name="id" value="<?php echo $c['id']; ?>">
                            <input type="hidden" name="csrf_token" value="<?php echo getCSRFToken(); ?>">
                            <button type="submit" class="btn btn-danger btn-sm">Delete</button>
                        </form>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>

    <?php else: ?>
    <div class="content-header">
        <h1><?php echo $collection ? 'Edit Collection' : 'New Collection'; ?></h1>
        <p><a href="/admin/collections.php">← Back to Collections</a></p>
    </div>

    <?php if ($error): ?>
    <div class="alert alert-error"><?php echo e($error); ?></div>
    <?php endif; ?>

    <form method="POST" action="/admin/collections.php?action=<?php echo $collection ? 'edit' : 'new'; ?><?php echo $collection ? '&id=' . $collection['id'] : ''; ?>">
        <input type="hidden" name="csrf_token" value="<?php echo getCSRFToken(); ?>">
        <?php if ($collection): ?>
        <input type="hidden" name="id" value="<?php echo $collection['id']; ?>">
        <?php endif; ?>

        <div class="card">
            <div class="card-body">
                <div class="form-group">
                    <label for="title">Collection Title</label>
                    <input type="text" id="title" name="title" class="form-control" value="<?php echo e($formTitle); ?>" required>
                </div>
                <div class="form-group">
                    <label for="collection_slug">Collection URL Slug</label>
                    <input type="text" id="collection_slug" name="collection_slug" class="form-control" value="<?php echo e($formCollectionSlug); ?>" placeholder="community-imageboard-php-sqlite">
                    <small class="text-muted">Optional. Used for clean URLs: <code>/products-page/collection/slug</code>.</small>
                </div>
                <div class="form-group">
                    <label for="image_url">Collection Image</label>
                    <input type="text" id="image_url" name="image_url" class="form-control" value="<?php echo e($formImageUrl); ?>" placeholder="/admin-file/collection.jpg" data-media-picker="image">
                </div>
                <div class="form-group">
                    <label for="target_page_id">Products List Page</label>
                    <select id="target_page_id" name="target_page_id" class="form-control">
                        <option value="">Select a page</option>
                        <?php foreach ($collectionPages as $page): ?>
                            <option value="<?php echo (int)$page['id']; ?>" <?php echo ($formTargetPageId === (int)$page['id']) ? 'selected' : ''; ?>>
                                <?php echo e($page['title']); ?>
                            </option>
                        <?php endforeach; ?>
                        <?php if ($formTargetPageId > 0 && !isset($collectionPageMap[$formTargetPageId])): ?>
                            <option value="<?php echo (int)$formTargetPageId; ?>" selected>
                                Current page (no products list)
                            </option>
                        <?php endif; ?>
                    </select>
                    <small class="text-muted">Only pages with a Products List section are available.</small>
                </div>
                <div class="form-group">
                    <label for="description">Description</label>
                    <textarea id="description" name="description" class="form-control" rows="6"><?php echo e($formDescription); ?></textarea>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="meta_title">Meta Title (SEO)</label>
                        <input type="text" id="meta_title" name="meta_title" class="form-control" maxlength="255" value="<?php echo e($formMetaTitle); ?>" placeholder="Optional SEO title">
                    </div>
                    <div class="form-group">
                        <label for="meta_description">Meta Description (SEO)</label>
                        <input type="text" id="meta_description" name="meta_description" class="form-control" maxlength="320" value="<?php echo e($formMetaDescription); ?>" placeholder="Optional SEO description">
                    </div>
                </div>
                <div class="seo-preview-box" aria-live="polite">
                    <div class="seo-preview-label">Google result preview (server-rendered)</div>
                    <a class="seo-preview-url" href="<?php echo e($previewPath); ?>" target="_blank" rel="noopener"><?php echo e($previewUrl); ?></a>
                    <div class="seo-preview-title"><?php echo e($previewTitle); ?></div>
                    <div class="seo-preview-description"><?php echo e($previewDescription); ?></div>
                    <small class="text-muted seo-preview-note">Preview updates when the form is rendered (after save/refresh). New collections use a URL placeholder until first save.</small>
                </div>
            </div>
            <div class="card-footer">
                <a href="/admin/collections.php" class="btn btn-secondary">Cancel</a>
                <button type="submit" class="btn btn-primary">Save Collection</button>
            </div>
        </div>
    </form>
    <?php endif; ?>
</div>

<?php AdminLayout::renderFooter(); ?>
