<?php
/**
 * Admin Panel - Backups
 *
 * Web: Routed through admin front controller (index.php).
 * CLI: Run directly with `php backup.php` for scheduled/cron tasks.
 */

use NewSite\Auth\AdminAuth;
use NewSite\Settings\SettingsService;
use NewSite\Logging\LogService;
use NewSite\Database\DatabaseManager;

// Web access: verified by front-controller bootstrap
if (!defined('ROOT_PATH') && php_sapi_name() !== 'cli') {
    http_response_code(403);
    exit;
}

// CLI access: bootstrap directly
if (!defined('ROOT_PATH')) {
    require_once __DIR__ . '/../../vendor/autoload.php';
}

use NewSite\Admin\AdminLayout;

$isCli = php_sapi_name() === 'cli';
if (!$isCli) {
    AdminAuth::requireLogin();
}

$message = '';
$error = '';
$requestMethod = strtoupper((string)($_SERVER['REQUEST_METHOD'] ?? 'GET'));

$backupModeOptions = [
    'full' => 'Full backup (includes uploads)',
    'site_only' => 'Site files only (no uploads)'
];

$backupSchedulerOptions = [
    'web' => 'Web trigger (runs when this page is opened and backup is due)',
    'cron' => 'Cron/CLI trigger (runs only from Scheduled Jobs or SSH cron)'
];

$rootPath = rtrim((string)ROOT_PATH, DIRECTORY_SEPARATOR);
$backupBaseDir = rtrim(dirname($rootPath) . DIRECTORY_SEPARATOR . 'backups', DIRECTORY_SEPARATOR);

function normalizePath(string $path): string
{
    return str_replace('\\', '/', $path);
}

function relativePath(string $path, string $root): string
{
    $rootNormalized = rtrim(normalizePath($root), '/');
    $pathNormalized = normalizePath($path);
    if (strpos($pathNormalized, $rootNormalized . '/') === 0) {
        return substr($pathNormalized, strlen($rootNormalized) + 1);
    }
    return ltrim($pathNormalized, '/');
}

function ensureBackupDir(string $path, string &$error): bool
{
    $freshlyCreated = false;
    if (!is_dir($path)) {
        if (!@mkdir($path, 0755, true)) {
            $error = 'Unable to create the backups folder. Check filesystem permissions.';
            return false;
        }
        $freshlyCreated = true;
    }

    if ($freshlyCreated) {
        $htaccessPath = $path . DIRECTORY_SEPARATOR . '.htaccess';
        if (!is_file($htaccessPath)) {
            @file_put_contents($htaccessPath, "Deny from all\n", LOCK_EX);
        }
    }

    $writable = is_writable($path);
    if (!$writable) {
        $error = 'Backups folder is not writable. Check filesystem permissions.';
    }
    return $writable;
}

function getUploadDirectories(): array
{
    return [
        'data/admin_uploads',
        'data/chat_images',
        'data/contact_uploads',
        'data/forum_images',
        'data/profile_photos',
        'data/push_queue'
    ];
}

function addUploadPlaceholders($archive, array $uploadDirs, string $rootPath, string $mode): void
{
    if ($mode !== 'site_only') {
        return;
    }
    foreach ($uploadDirs as $uploadDir) {
        $uploadPath = $rootPath . DIRECTORY_SEPARATOR . str_replace('/', DIRECTORY_SEPARATOR, $uploadDir);
        if (is_dir($uploadPath)) {
            $archive->addEmptyDir($uploadDir);
            $htaccess = $uploadPath . DIRECTORY_SEPARATOR . '.htaccess';
            if (is_file($htaccess)) {
                $archive->addFile($htaccess, $uploadDir . '/.htaccess');
            }
        }
    }
}

function isPathInsideUploadDirs(string $pathname, string $rootPath, array $uploadDirs): bool
{
    $relative = relativePath($pathname, $rootPath);
    foreach ($uploadDirs as $uploadDir) {
        if ($relative === $uploadDir || strpos($relative, $uploadDir . '/') === 0) {
            return true;
        }
    }
    return false;
}

function isValidIdentifier(string $name): bool
{
    return (bool)preg_match('/^\w+$/', $name);
}

function quoteIdentifier(string $name): string
{
    return '"' . str_replace('"', '""', $name) . '"';
}

function fetchDatabaseTables(PDO $db): array
{
    $stmt = $db->prepare("SELECT table_name FROM information_schema.tables WHERE table_schema = 'public' AND table_type = 'BASE TABLE' ORDER BY table_name ASC");
    $stmt->execute();
    return array_values(array_filter($stmt->fetchAll(PDO::FETCH_COLUMN), 'is_string'));
}

function fetchTableColumns(PDO $db, string $table): array
{
    $stmt = $db->prepare("SELECT column_name FROM information_schema.columns WHERE table_schema = 'public' AND table_name = ? ORDER BY ordinal_position ASC");
    $stmt->execute([$table]);
    return array_values(array_filter($stmt->fetchAll(PDO::FETCH_COLUMN), 'is_string'));
}

function fetchTableDependencies(PDO $db): array
{
    $deps = [];
    $stmt = $db->prepare(
        "SELECT tc.table_name, ccu.table_name AS foreign_table_name
         FROM information_schema.table_constraints tc
         JOIN information_schema.key_column_usage kcu
           ON tc.constraint_name = kcu.constraint_name AND tc.table_schema = kcu.table_schema
         JOIN information_schema.constraint_column_usage ccu
           ON ccu.constraint_name = tc.constraint_name AND ccu.table_schema = tc.table_schema
         WHERE tc.constraint_type = 'FOREIGN KEY' AND tc.table_schema = 'public'"
    );
    $stmt->execute();
    foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) as $row) {
        $table = $row['table_name'] ?? '';
        $foreign = $row['foreign_table_name'] ?? '';
        if (!isValidIdentifier($table) || !isValidIdentifier($foreign)) {
            continue;
        }
        if (!isset($deps[$table])) {
            $deps[$table] = [];
        }
        $deps[$table][$foreign] = true;
    }
    foreach ($deps as $table => $tables) {
        $deps[$table] = array_keys($tables);
    }
    return $deps;
}

/**
 * Build adjacency list and in-degree map from a table list and its foreign-key dependencies.
 *
 * @param  list<string>              $tables
 * @param  array<string,list<string>> $dependencies  table → tables it depends on
 * @return array{0: array<string,list<string>>, 1: array<string,int>}  [$graph, $inDegree]
 */
function buildDependencyGraph(array $tables, array $dependencies): array
{
    $inDegree = [];
    $graph = [];
    foreach ($tables as $table) {
        $inDegree[$table] = 0;
        $graph[$table] = [];
    }
    foreach ($dependencies as $table => $dependsOn) {
        foreach ($dependsOn as $depTable) {
            if (!isset($graph[$depTable])) {
                continue;
            }
            $graph[$depTable][] = $table;
            $inDegree[$table] = ($inDegree[$table] ?? 0) + 1;
        }
    }
    return [$graph, $inDegree];
}

/**
 * Run Kahn's BFS topological sort over the given graph.
 *
 * @param  array<string,list<string>> $graph
 * @param  array<string,int>          $inDegree
 * @return list<string>               Topologically ordered table names
 */
function processTopoQueue(array $graph, array $inDegree): array
{
    $queue = [];
    foreach ($inDegree as $table => $degree) {
        if ($degree === 0) {
            $queue[] = $table;
        }
    }

    $ordered = [];
    while (!empty($queue)) {
        $table = array_shift($queue);
        $ordered[] = $table;
        foreach ($graph[$table] as $neighbor) {
            $inDegree[$neighbor] -= 1;
            if ($inDegree[$neighbor] === 0) {
                $queue[] = $neighbor;
            }
        }
    }

    return $ordered;
}

/**
 * Sort tables in dependency-safe order (topological sort with cycle fallback).
 *
 * @param  list<string>              $tables
 * @param  array<string,list<string>> $dependencies
 * @return list<string>
 */
function sortTablesByDependencies(array $tables, array $dependencies): array
{
    [$graph, $inDegree] = buildDependencyGraph($tables, $dependencies);
    $ordered = processTopoQueue($graph, $inDegree);

    // Append any tables caught in dependency cycles so nothing is silently dropped.
    if (count($ordered) !== count($tables)) {
        foreach ($tables as $table) {
            if (!in_array($table, $ordered, true)) {
                $ordered[] = $table;
            }
        }
    }

    return $ordered;
}

function writeJsonLine($handle, array $payload, string &$error): bool
{
    $json = json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_INVALID_UTF8_SUBSTITUTE);
    if ($json === false) {
        $error = 'Failed to encode database backup data.';
        return false;
    }
    if (@fwrite($handle, $json . "\n") === false) {
        $error = 'Failed to write database backup data.';
        return false;
    }
    return true;
}

/**
 * Write the JSON‑Lines meta header to the backup file.
 *
 * @param  resource $handle  Open file handle
 * @param  string   &$error  Populated on failure
 * @return bool
 */
function exportDbMeta($handle, string &$error): bool
{
    return writeJsonLine($handle, [
        'type'       => 'meta',
        'version'    => 1,
        'created_at' => date('c'),
    ], $error);
}

/**
 * Export a single table (header + chunked rows) to the backup file.
 *
 * Tables with an invalid identifier or no columns are silently skipped.
 *
 * @param  PDO      $db
 * @param  resource $handle
 * @param  string   $table
 * @param  int      $chunkSize
 * @param  string   &$error
 * @return bool     true on success (or skip), false on write failure
 */
function exportDbTable(PDO $db, $handle, string $table, int $chunkSize, string &$error): bool
{
    $columns = isValidIdentifier($table) ? fetchTableColumns($db, $table) : [];
    if (empty($columns)) {
        return true; // nothing to export for this table
    }

    if (!writeJsonLine($handle, [
        'type'    => 'table',
        'name'    => $table,
        'columns' => $columns,
    ], $error)) {
        return false;
    }

    $sql  = 'SELECT * FROM ' . quoteIdentifier($table);
    $stmt = $db->prepare($sql);
    $stmt->execute();

    $success = true;
    $rows    = [];
    while ($success && ($row = $stmt->fetch(PDO::FETCH_ASSOC))) {
        $rowValues = [];
        foreach ($columns as $column) {
            $rowValues[] = array_key_exists($column, $row) ? $row[$column] : null;
        }
        $rows[] = $rowValues;
        if (count($rows) >= $chunkSize) {
            $success = writeJsonLine($handle, [
                'type'  => 'rows',
                'table' => $table,
                'rows'  => $rows,
            ], $error);
            $rows = [];
        }
    }

    if ($success && !empty($rows)) {
        $success = writeJsonLine($handle, [
            'type'  => 'rows',
            'table' => $table,
            'rows'  => $rows,
        ], $error);
    }

    return $success;
}

/**
 * Export the full database to a JSON‑Lines backup file.
 *
 * @param  PDO    $db
 * @param  string $filePath
 * @param  string &$error
 * @return bool
 */
function exportDatabaseToFile(PDO $db, string $filePath, string &$error): bool
{
    $handle = @fopen($filePath, 'wb');
    if (!$handle) {
        $error = 'Unable to create database backup file.';
        return false;
    }

    $tables        = fetchDatabaseTables($db);
    $deps          = fetchTableDependencies($db);
    $orderedTables = sortTablesByDependencies($tables, $deps);

    $success   = exportDbMeta($handle, $error);
    $chunkSize = 200;

    foreach ($orderedTables as $table) {
        if (!$success) {
            break;
        }
        $success = exportDbTable($db, $handle, $table, $chunkSize, $error);
    }

    fclose($handle);
    return $success;
}

function resetSequences(PDO $db): void
{
    $stmt = $db->prepare("SELECT table_name, column_name FROM information_schema.columns WHERE table_schema = 'public' AND column_default LIKE 'nextval%'");
    $stmt->execute();
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
    foreach ($rows as $row) {
        $table = $row['table_name'] ?? '';
        $column = $row['column_name'] ?? '';
        if (!isValidIdentifier($table) || !isValidIdentifier($column)) {
            continue;
        }
        $seqStmt = $db->prepare('SELECT pg_get_serial_sequence(?, ?)');
        $seqStmt->execute([$table, $column]);
        $sequence = $seqStmt->fetchColumn();
        if (!$sequence) {
            continue;
        }
        $sql = 'SELECT setval(?, COALESCE((SELECT MAX(' . quoteIdentifier($column) . ') FROM ' . quoteIdentifier($table) . '), 1), true)';
        $setStmt = $db->prepare($sql);
        $setStmt->execute([$sequence]);
    }
}

/**
 * Fetch tables, begin transaction, defer constraints, and truncate all tables.
 *
 * @param  PDO    $db
 * @param  string &$error
 * @return array  Quoted table names; empty array on failure
 */
function prepareRestoreTransaction(PDO $db, string &$error): array
{
    $tables = fetchDatabaseTables($db);
    if (empty($tables)) {
        $error = 'No tables found for restore.';
        return [];
    }

    $quotedTables = array_map('quoteIdentifier', $tables);
    $db->beginTransaction();
    $db->exec('SET CONSTRAINTS ALL DEFERRED');
    $db->exec('TRUNCATE TABLE ' . implode(', ', $quotedTables) . ' RESTART IDENTITY CASCADE');

    return $quotedTables;
}

/**
 * Insert rows for a single 'rows' payload into the target table.
 *
 * @param  PDO    $db
 * @param  string $table
 * @param  array  $rows
 * @param  array  $columns
 * @param  array  &$statementCache
 * @param  string &$error
 * @return bool   False when a row fails validation (column count mismatch)
 */
function processRestoreRows(
    PDO $db,
    string $table,
    array $rows,
    array $columns,
    array &$statementCache,
    string &$error
): bool {
    if (!isset($statementCache[$table])) {
        $placeholders = '(' . implode(', ', array_fill(0, count($columns), '?')) . ')';
        $sql          = 'INSERT INTO ' . quoteIdentifier($table)
                      . ' (' . implode(', ', array_map('quoteIdentifier', $columns)) . ')'
                      . ' VALUES ' . $placeholders;
        $statementCache[$table] = $db->prepare($sql);
    }

    $stmt        = $statementCache[$table];
    $columnCount = count($columns);

    foreach ($rows as $row) {
        if (!is_array($row) || count($row) !== $columnCount) {
            $error = 'Invalid database backup format.';
            return false;
        }
        $stmt->execute(array_values($row));
    }

    return true;
}

/**
 * Handle a single parsed JSON line from the backup stream.
 *
 * @param  PDO    $db
 * @param  array  $payload        Decoded JSON line
 * @param  array  &$columnsByTable
 * @param  array  &$statementCache
 * @param  string &$error
 * @return bool   False only when a row validation error occurs
 */
function processRestoreLine(
    PDO $db,
    array $payload,
    array &$columnsByTable,
    array &$statementCache,
    string &$error
): bool {
    $type = $payload['type'] ?? '';

    if ($type === 'table') {
        $table   = $payload['name'] ?? '';
        $columns = $payload['columns'] ?? [];
        if (isValidIdentifier($table) && is_array($columns) && !empty($columns)) {
            $columnsByTable[$table] = array_values(array_filter($columns, 'is_string'));
        }
        return true;
    }

    if ($type !== 'rows') {
        return true;
    }

    $table      = $payload['table'] ?? '';
    $rows       = $payload['rows'] ?? [];
    $canProcess = isValidIdentifier($table) && isset($columnsByTable[$table]) && is_array($rows);

    return !$canProcess || processRestoreRows($db, $table, $rows, $columnsByTable[$table], $statementCache, $error);
}

function restoreDatabaseFromStream(PDO $db, $stream, string &$error): bool
{
    set_time_limit(0);
    $statementCache = [];
    $columnsByTable = [];

    try {
        $quotedTables = prepareRestoreTransaction($db, $error);
        if ($quotedTables === []) {
            return false;
        }

        $valid = true;
        while ($valid && ($line = fgets($stream)) !== false) {
            $line = trim($line);
            if ($line === '') {
                continue;
            }
            $payload = json_decode($line, true);
            if (!is_array($payload)) {
                continue;
            }
            $valid = processRestoreLine($db, $payload, $columnsByTable, $statementCache, $error);
        }

        if ($valid) {
            resetSequences($db);
            $db->commit();
            return true;
        }

        $db->rollBack();
    } catch (Throwable $e) {
        if ($db->inTransaction()) {
            $db->rollBack();
        }
        $error = 'Database restore failed.';
    }

    return false;
}

/**
 * Validates that the backup environment is ready: writable directory,
 * no overlap with root, and ZipArchive availability.
 *
 * @return string Error message, or empty string on success.
 */
function validateBackupEnvironment(string $rootPath, string $backupDir): string
{
    $error = '';
    if (!is_dir($backupDir) || !is_writable($backupDir)) {
        $error = 'Backups folder is not writable.';
    } elseif (strpos(normalizePath($backupDir), normalizePath($rootPath) . '/') === 0) {
        $error = 'Backups folder must be outside the site root.';
    } elseif (!class_exists('ZipArchive')) {
        $error = 'ZipArchive is not available on this server.';
    }

    return $error;
}

/**
 * Acquires a file-based lock to prevent concurrent backups.
 * A stale lock older than 1 hour is ignored.
 *
 * @return string Error message, or empty string on success.
 */
function acquireBackupLock(string $backupDir): string
{
    $lockPath = $backupDir . DIRECTORY_SEPARATOR . '.backup_lock';
    if (is_file($lockPath) && (time() - filemtime($lockPath)) < 3600) {
        return 'A backup is already running. Please try again later.';
    }
    @file_put_contents($lockPath, (string)time(), LOCK_EX);

    return '';
}

/**
 * Exports the database to a temp NDJSON file and adds it to the archive.
 *
 * @param string &$tempDbPath Set to the temp file path so the caller can clean up.
 * @return string Error message, or empty string on success.
 */
function exportDatabaseToArchive(ZipArchive $zip, string $backupDir, string &$tempDbPath): string
{
    $tempName = tempnam($backupDir, 'db_');
    if ($tempName === false) {
        return 'Unable to create a temporary database backup file.';
    }

    $tempDbPath = $tempName . '.ndjson';
    @rename($tempName, $tempDbPath);

    $dbError = '';
    $db = DatabaseManager::getReadConnection();
    if (!exportDatabaseToFile($db, $tempDbPath, $dbError)) {
        return $dbError !== '' ? $dbError : 'Database export failed.';
    }

    $zip->addFile($tempDbPath, 'db_backup.ndjson');
    return '';
}

/**
 * Iterates the site filesystem and adds files/directories to the archive,
 * respecting the backup mode and skipping upload directories in site-only mode.
 */
function addFilesToArchive(ZipArchive $zip, string $rootPath, string $mode, array $uploadDirs): void
{
    $directoryIterator = new RecursiveDirectoryIterator(
        $rootPath,
        FilesystemIterator::SKIP_DOTS | FilesystemIterator::CURRENT_AS_FILEINFO
    );

    $filter = new RecursiveCallbackFilterIterator(
        $directoryIterator,
        static function (SplFileInfo $current) use ($mode, $uploadDirs, $rootPath): bool {
            if ($current->isLink()) {
                return false;
            }
            if ($mode !== 'site_only') {
                return true;
            }
            return !isPathInsideUploadDirs($current->getPathname(), $rootPath, $uploadDirs);
        }
    );

    $iterator = new RecursiveIteratorIterator($filter, RecursiveIteratorIterator::SELF_FIRST);

    foreach ($iterator as $fileInfo) {
        $fullPath = $fileInfo->getPathname();
        $relative = relativePath($fullPath, $rootPath);
        if ($relative === '') {
            continue;
        }
        if ($fileInfo->isDir()) {
            $zip->addEmptyDir($relative);
        } elseif ($fileInfo->isFile()) {
            $zip->addFile($fullPath, $relative);
        }
    }
}

/**
 * Orchestrates creation of a backup archive: validates environment,
 * acquires lock, builds the zip, then cleans up on all exit paths.
 */
function createBackupArchive(string $rootPath, string $backupDir, string $mode, bool $includeDb): array
{
    $result = [
        'success' => false,
        'file' => '',
        'error' => ''
    ];

    // --- Pre-lock validation (no cleanup needed) ---
    $result['error'] = validateBackupEnvironment($rootPath, $backupDir);
    if ($result['error'] !== '') {
        return $result;
    }

    $result['error'] = acquireBackupLock($backupDir);
    if ($result['error'] !== '') {
        return $result;
    }

    // --- From this point on, cleanup is required ---
    $lockPath = $backupDir . DIRECTORY_SEPARATOR . '.backup_lock';
    $tempDbPath = '';

    set_time_limit(0);

    $timestamp = date('Ymd_His');
    $modeSuffix = $mode === 'site_only' ? 'site' : 'full';
    $archiveName = 'backup_' . $modeSuffix . '_' . $timestamp . '.zip';
    $archivePath = $backupDir . DIRECTORY_SEPARATOR . $archiveName;

    $uploadDirs = getUploadDirectories();

    $zip = new ZipArchive();
    if ($zip->open($archivePath, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== true) {
        $result['error'] = 'Unable to create the backup archive.';
    }

    // Write manifest, placeholders, optional DB, and site files
    if ($result['error'] === '') {
        $manifest = json_encode([
            'mode' => $mode,
            'includes_db' => $includeDb,
            'created_at' => date('c')
        ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
        if ($manifest !== false) {
            $zip->addFromString('backup_manifest.json', $manifest);
        }

        addUploadPlaceholders($zip, $uploadDirs, $rootPath, $mode);

        if ($includeDb) {
            $result['error'] = exportDatabaseToArchive($zip, $backupDir, $tempDbPath);
        }
    }

    if ($result['error'] === '') {
        addFilesToArchive($zip, $rootPath, $mode, $uploadDirs);
        $zip->close();

        $result['success'] = true;
        $result['file'] = $archiveName;
    } else {
        // Failure path: discard partial archive
        $zip->close();
        @unlink($archivePath);
    }

    // --- Cleanup: always remove lock and temp DB file ---
    @unlink($lockPath);
    if ($tempDbPath !== '' && is_file($tempDbPath)) {
        @unlink($tempDbPath);
    }

    return $result;
}

function listBackups(string $backupDir): array
{
    if (!is_dir($backupDir)) {
        return [];
    }
    $files = [];
    foreach (new DirectoryIterator($backupDir) as $fileInfo) {
        if ($fileInfo->isDot() || !$fileInfo->isFile()) {
            continue;
        }
        $name = $fileInfo->getFilename();
        if (!preg_match('/\.(zip|tar)$/i', $name)) {
            continue;
        }
        $files[] = [
            'name' => $name,
            'size' => (int)$fileInfo->getSize(),
            'modified' => (int)$fileInfo->getMTime()
        ];
    }
    usort($files, function ($a, $b) {
        return $b['modified'] <=> $a['modified'];
    });
    return $files;
}

function isBackupArchiveName(string $name): bool
{
    return (bool)preg_match('/^[a-z0-9._-]+\.(zip|tar)$/i', $name);
}

if (!$isCli && isset($_GET['download'])) {
    $token = (string)($_GET['csrf_token'] ?? '');
    if (!hash_equals(getCsrfToken(), $token)) {
        http_response_code(403);
        echo 'Invalid CSRF token.';
        exit;
    }
    $downloadName = basename((string)$_GET['download']);
    $downloadPath = $backupBaseDir . DIRECTORY_SEPARATOR . $downloadName;
    if (!is_file($downloadPath)) {
        http_response_code(404);
        echo 'Backup not found.';
        exit;
    }
    header('Content-Type: application/octet-stream');
    header('Content-Disposition: attachment; filename="' . $downloadName . '"');
    header('Content-Length: ' . filesize($downloadPath));
    header('Cache-Control: private, no-store, no-cache, must-revalidate, max-age=0');
    readfile($downloadPath);
    exit;
}

$backupEnabled = SettingsService::get('backup_enabled', '0') === '1';
$backupIntervalHours = max(0, (int)SettingsService::get('backup_interval_hours', '24'));
$backupMode = SettingsService::get('backup_mode', 'full');
if (!array_key_exists($backupMode, $backupModeOptions)) {
    $backupMode = 'full';
}
$backupSchedulerMode = SettingsService::get('backup_scheduler_mode', 'web');
if (!array_key_exists($backupSchedulerMode, $backupSchedulerOptions)) {
    $backupSchedulerMode = 'web';
}
$backupIncludeDb = SettingsService::get('backup_include_db', '0') === '1';
$backupLastRun = (int)SettingsService::get('backup_last_run', '0');
$backupLastFile = (string)SettingsService::get('backup_last_file', '');
$cronLastRun = (int)SettingsService::get('backup_cron_last_run', '0');
$cronLastStatus = (string)SettingsService::get('backup_cron_last_status', 'never');
$cronLastMessage = (string)SettingsService::get('backup_cron_last_message', '');

if (!$isCli && $requestMethod === 'POST') {
    verifyCSRF();
    $action = $_POST['action'] ?? '';

    if ($action === 'save_settings') {
        $backupEnabled = ($_POST['backup_enabled'] ?? '0') === '1';
        $backupIntervalHours = max(0, (int)($_POST['backup_interval_hours'] ?? 24));
        $backupMode = $_POST['backup_mode'] ?? 'full';
        if (!array_key_exists($backupMode, $backupModeOptions)) {
            $backupMode = 'full';
        }
        $backupSchedulerMode = $_POST['backup_scheduler_mode'] ?? 'web';
        if (!array_key_exists($backupSchedulerMode, $backupSchedulerOptions)) {
            $backupSchedulerMode = 'web';
        }
        $backupIncludeDb = ($_POST['backup_include_db'] ?? '0') === '1';
        if ($backupEnabled && $backupIntervalHours < 1) {
            $error = 'Auto backup interval must be at least 1 hour.';
        } else {
            SettingsService::set('backup_enabled', $backupEnabled ? '1' : '0');
            SettingsService::set('backup_interval_hours', (string)$backupIntervalHours);
            SettingsService::set('backup_mode', $backupMode);
            SettingsService::set('backup_scheduler_mode', $backupSchedulerMode);
            SettingsService::set('backup_include_db', $backupIncludeDb ? '1' : '0');
            $message = 'Backup settings saved.';
        }
    }

    if ($action === 'run_backup' && $error === '' && ensureBackupDir($backupBaseDir, $error)) {
        $requestedMode = $_POST['backup_mode'] ?? $backupMode;
        if (!array_key_exists($requestedMode, $backupModeOptions)) {
            $requestedMode = 'full';
        }
        $requestedIncludeDb = ($_POST['backup_include_db'] ?? ($backupIncludeDb ? '1' : '0')) === '1';
        $result = createBackupArchive($rootPath, $backupBaseDir, $requestedMode, $requestedIncludeDb);
        if ($result['success']) {
            $backupLastRun = time();
            $backupLastFile = $result['file'];
            SettingsService::set('backup_last_run', (string)$backupLastRun);
            SettingsService::set('backup_last_file', $backupLastFile);
            $message = 'Backup created: ' . $backupLastFile;
            LogService::add('backup', 'Backup created', json_encode([
                'mode' => $requestedMode,
                'includes_db' => $requestedIncludeDb,
                'file' => $backupLastFile
            ]));
        } else {
            $error = $result['error'] !== '' ? $result['error'] : 'Backup failed.';
        }
    }

    if ($action === 'delete_backup' && $error === '') {
        if (!ensureBackupDir($backupBaseDir, $error)) {
            // $error already set.
        } else {
            $deleteName = basename((string)($_POST['backup_file'] ?? ''));
            if ($deleteName === '' || !isBackupArchiveName($deleteName)) {
                $error = 'Invalid backup file selected.';
            } else {
                $deletePath = $backupBaseDir . DIRECTORY_SEPARATOR . $deleteName;
                if (!is_file($deletePath)) {
                    $error = 'Backup file not found.';
                } elseif (!@unlink($deletePath)) {
                    $error = 'Unable to delete backup file. Check permissions.';
                } else {
                    if ($backupLastFile === $deleteName) {
                        $backupLastFile = '';
                        SettingsService::set('backup_last_file', '');
                    }
                    $message = 'Backup deleted: ' . $deleteName;
                    LogService::add('backup', 'Backup deleted', json_encode(['file' => $deleteName]));
                }
            }
        }
    }

    if ($action === 'delete_all_backups' && $error === '') {
        if (!ensureBackupDir($backupBaseDir, $error)) {
            // $error already set.
        } else {
            $allBackups = listBackups($backupBaseDir);
            if (empty($allBackups)) {
                $message = 'No backups to delete.';
            } else {
                $deletedCount = 0;
                $failedCount = 0;
                foreach ($allBackups as $backupFile) {
                    $name = basename((string)($backupFile['name'] ?? ''));
                    if ($name === '' || !isBackupArchiveName($name)) {
                        $failedCount++;
                        continue;
                    }
                    $path = $backupBaseDir . DIRECTORY_SEPARATOR . $name;
                    if (is_file($path) && @unlink($path)) {
                        $deletedCount++;
                    } else {
                        $failedCount++;
                    }
                }

                if ($deletedCount > 0) {
                    $backupLastFile = '';
                    SettingsService::set('backup_last_file', '');
                    $message = 'Deleted ' . $deletedCount . ' backup' . ($deletedCount === 1 ? '' : 's') . '.';
                    if ($failedCount > 0) {
                        $message .= ' ' . $failedCount . ' could not be deleted.';
                    }
                    LogService::add('backup', 'Bulk backup delete', json_encode([
                        'deleted' => $deletedCount,
                        'failed' => $failedCount
                    ]));
                } elseif ($failedCount > 0) {
                    $error = 'No backups were deleted. Check filesystem permissions.';
                } else {
                    $message = 'No backups to delete.';
                }
            }
        }
    }

    if ($action === 'restore_backup' && $error === '') {
        $restoreName = basename((string)($_POST['restore_backup_file'] ?? ''));
        $confirmText = trim((string)($_POST['restore_confirm'] ?? ''));
        if ($confirmText !== 'RESTORE') {
            $error = 'Type RESTORE to confirm database restore.';
        } elseif ($restoreName === '') {
            $error = 'Select a backup file to restore.';
        } else {
            $restorePath = $backupBaseDir . DIRECTORY_SEPARATOR . $restoreName;
            if (!is_file($restorePath)) {
                $error = 'Backup file not found.';
            } elseif (!class_exists('ZipArchive')) {
                $error = 'ZipArchive is not available on this server.';
            } else {
                $zip = new ZipArchive();
                if ($zip->open($restorePath) !== true) {
                    $error = 'Unable to open the backup archive.';
                } else {
                    $entryName = 'db_backup.ndjson';
                    $index = $zip->locateName($entryName, ZipArchive::FL_NOCASE);
                    if ($index === false) {
                        $error = 'This backup does not contain a database export.';
                    } else {
                        $stream = $zip->getStream($entryName);
                        if (!$stream) {
                            $error = 'Unable to read the database export from the archive.';
                        } else {
                            $restoreError = '';
                            $db = DatabaseManager::getWriteConnection();
                            if (restoreDatabaseFromStream($db, $stream, $restoreError)) {
                                $message = 'Database restored from ' . $restoreName;
                                LogService::add('backup', 'Database restored', json_encode([
                                    'file' => $restoreName
                                ]));
                            } else {
                                $error = $restoreError !== '' ? $restoreError : 'Database restore failed.';
                            }
                            fclose($stream);
                        }
                    }
                    $zip->close();
                }
            }
        }
    }
}

$shouldRunScheduledBackup = false;
if ($backupSchedulerMode === 'web' && !$isCli) {
    $shouldRunScheduledBackup = true;
}
if ($backupSchedulerMode === 'cron' && $isCli) {
    $shouldRunScheduledBackup = true;
}

if ($error === '' && $shouldRunScheduledBackup && $backupEnabled && $backupIntervalHours > 0) {
    $nextDue = $backupLastRun + ($backupIntervalHours * 3600);
    if (($backupLastRun === 0 || $nextDue <= time()) && ensureBackupDir($backupBaseDir, $error)) {
        $result = createBackupArchive($rootPath, $backupBaseDir, $backupMode, $backupIncludeDb);
        if ($result['success']) {
            $backupLastRun = time();
            $backupLastFile = $result['file'];
            SettingsService::set('backup_last_run', (string)$backupLastRun);
            SettingsService::set('backup_last_file', $backupLastFile);
            $message = 'Scheduled backup created: ' . $backupLastFile;
            LogService::add('backup', 'Scheduled backup created', json_encode([
                'mode' => $backupMode,
                'includes_db' => $backupIncludeDb,
                'file' => $backupLastFile
            ]));
        } else {
            $error = $result['error'] !== '' ? $result['error'] : 'Scheduled backup failed.';
        }
    }
}

if ($isCli) {
    $cronLastRun = time();
    $cronStatusToStore = 'idle';
    $cronMessageToStore = 'No backup due.';

    if ($backupSchedulerMode !== 'cron') {
        $cronMessageToStore = 'Cron run skipped (Auto backup trigger is Web trigger).';
    }

    if ($error !== '') {
        $cronStatusToStore = 'error';
        $cronMessageToStore = $error;
        SettingsService::set('backup_cron_last_run', (string)$cronLastRun);
        SettingsService::set('backup_cron_last_status', $cronStatusToStore);
        SettingsService::set('backup_cron_last_message', $cronMessageToStore);
        fwrite(STDERR, '[backup-cron] ERROR: ' . $error . PHP_EOL);
        exit(1);
    }

    if ($message !== '') {
        $cronStatusToStore = 'success';
        $cronMessageToStore = $message;
    }

    SettingsService::set('backup_cron_last_run', (string)$cronLastRun);
    SettingsService::set('backup_cron_last_status', $cronStatusToStore);
    SettingsService::set('backup_cron_last_message', $cronMessageToStore);

    if ($message !== '') {
        echo '[backup-cron] ' . $message . PHP_EOL;
    } else {
        echo '[backup-cron] No backup due.' . PHP_EOL;
    }

    exit(0);
}

$backupList = listBackups($backupBaseDir);
$backupDirReady = ensureBackupDir($backupBaseDir, $error);
$displayDateFormat = 'Y-m-d H:i:s';
$lastRunLabel = $backupLastRun > 0 ? date($displayDateFormat, $backupLastRun) : 'Never';
$cronLastRunLabel = $cronLastRun > 0 ? date($displayDateFormat, $cronLastRun) : 'Never';

$cronStatusLabel = 'Never executed';
if ($cronLastStatus === 'success') {
    $cronStatusLabel = 'Success';
} elseif ($cronLastStatus === 'idle') {
    $cronStatusLabel = 'No backup due';
} elseif ($cronLastStatus === 'error') {
    $cronStatusLabel = 'Error';
}

$nextRunLabel = 'Disabled';
if ($backupEnabled && $backupIntervalHours > 0) {
    $nextTimestamp = $backupLastRun > 0 ? $backupLastRun + ($backupIntervalHours * 3600) : time() + ($backupIntervalHours * 3600);
    $nextRunLabel = date($displayDateFormat, $nextTimestamp);
}

AdminLayout::renderHeader();
?>

<div class="admin-content">
    <div class="content-header">
        <h1>Backups</h1>
        <p>Create full or site-only backups and schedule automatic runs.</p>
    </div>

    <?php if ($message): ?>
        <div class="alert alert-success"><?php echo e($message); ?></div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="alert alert-error"><?php echo e($error); ?></div>
    <?php endif; ?>

    <div class="card">
        <div class="card-header">
            <h3>Backup Settings</h3>
        </div>
        <div class="card-body">
            <form method="POST" action="/admin/backup.php">
                <input type="hidden" name="action" value="save_settings">
                <div class="form-row">
                    <div class="form-group">
                        <label for="backup-enabled">Enable auto backups</label>
                        <select id="backup-enabled" name="backup_enabled" class="form-control">
                            <option value="0" <?php echo !$backupEnabled ? 'selected' : ''; ?>>Disabled</option>
                            <option value="1" <?php echo $backupEnabled ? 'selected' : ''; ?>>Enabled</option>
                        </select>
                    </div>
                    <div class="form-group">
                        <label for="backup-interval">Interval (hours)</label>
                        <input type="number" id="backup-interval" name="backup_interval_hours" class="form-control" min="1" max="720" value="<?php echo e((string)$backupIntervalHours); ?>">
                    </div>
                    <div class="form-group">
                        <label for="backup-mode">Backup mode</label>
                        <select id="backup-mode" name="backup_mode" class="form-control">
                            <?php foreach ($backupModeOptions as $key => $label): ?>
                                <option value="<?php echo e($key); ?>" <?php echo $backupMode === $key ? 'selected' : ''; ?>><?php echo e($label); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="form-group">
                        <label for="backup-scheduler-mode">Auto backup trigger</label>
                        <select id="backup-scheduler-mode" name="backup_scheduler_mode" class="form-control">
                            <?php foreach ($backupSchedulerOptions as $key => $label): ?>
                                <option value="<?php echo e($key); ?>" <?php echo $backupSchedulerMode === $key ? 'selected' : ''; ?>><?php echo e($label); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="form-group">
                        <label for="backup-include-db">Include database</label>
                        <select id="backup-include-db" name="backup_include_db" class="form-control">
                            <option value="0" <?php echo !$backupIncludeDb ? 'selected' : ''; ?>>No</option>
                            <option value="1" <?php echo $backupIncludeDb ? 'selected' : ''; ?>>Yes</option>
                        </select>
                    </div>
                </div>
                <div class="table-actions">
                    <button type="submit" class="btn btn-primary">Save Settings</button>
                </div>
            </form>
            <div class="text-muted mt-10">
                Enable database backups if you want a full recovery snapshot.
                Choose <strong>Web trigger</strong> when you have no cron available,
                or <strong>Cron/CLI trigger</strong> when using OVH Scheduled Jobs or SSH cron.
            </div>
        </div>
    </div>

    <div class="card">
        <div class="card-header">
            <h3>Run Backup</h3>
        </div>
        <div class="card-body">
            <p class="text-muted">Last backup: <?php echo e($lastRunLabel); ?>. Next scheduled: <?php echo e($nextRunLabel); ?>.</p>
            <form method="POST" action="/admin/backup.php">
                <input type="hidden" name="action" value="run_backup">
                <div class="form-row">
                    <div class="form-group">
                        <label for="backup-run-mode">Backup mode</label>
                        <select id="backup-run-mode" name="backup_mode" class="form-control">
                            <?php foreach ($backupModeOptions as $key => $label): ?>
                                <option value="<?php echo e($key); ?>" <?php echo $backupMode === $key ? 'selected' : ''; ?>><?php echo e($label); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="form-group">
                        <label for="backup-run-include-db">Include database</label>
                        <select id="backup-run-include-db" name="backup_include_db" class="form-control">
                            <option value="0" <?php echo !$backupIncludeDb ? 'selected' : ''; ?>>No</option>
                            <option value="1" <?php echo $backupIncludeDb ? 'selected' : ''; ?>>Yes</option>
                        </select>
                    </div>
                </div>
                <div class="table-actions">
                    <button type="submit" class="btn btn-primary" <?php echo !$backupDirReady ? 'disabled' : ''; ?>>Backup Now</button>
                </div>
            </form>
        </div>
    </div>

    <div class="card">
        <div class="card-header">
            <h3>Cron / SSH Setup</h3>
        </div>
        <div class="card-body">
            <p class="text-muted">
                Last cron execution: <strong><?php echo e($cronLastRunLabel); ?></strong><br>
                Status: <strong><?php echo e($cronStatusLabel); ?></strong>
                <?php if ($cronLastMessage !== ''): ?>
                    <br>Message: <?php echo e($cronLastMessage); ?>
                <?php endif; ?>
            </p>

            <p class="text-muted">Use one of these optional automation methods:</p>
            <ol class="text-muted backup-note-list">
                <li>
                    <strong>OVH Scheduled Jobs (no SSH required):</strong><br>
                    Language: PHP 8.5<br>
                    Command: <code>./public/admin/backup.php</code><br>
                    Frequency: every 15 minutes
                </li>
                <li>
                    <strong>SSH cron (if SSH is available):</strong><br>
                    1) Run <code>crontab -e</code><br>
                    2) Add (adjust paths):<br>
                    <code>*/15 * * * * /usr/bin/php /home/your-user/www/public/admin/backup.php >/dev/null 2&gt;&amp;1</code><br>
                    3) Save and verify with <code>crontab -l</code>
                </li>
            </ol>
            <p class="text-muted">When using cron, set <strong>Auto backup trigger</strong> to <strong>Cron/CLI trigger</strong>.</p>
        </div>
    </div>

    <div class="card">
        <div class="card-header">
            <h3>Restore Database</h3>
        </div>
        <div class="card-body">
            <p class="text-muted">Restoring replaces all current database data. Only backups created with database included can be restored. This action is destructive and requires typed confirmation plus CSRF protection.</p>
            <form method="POST" action="/admin/backup.php">
                <input type="hidden" name="action" value="restore_backup">
                <div class="form-row">
                    <div class="form-group">
                        <label for="restore-backup-file">Backup file</label>
                        <select id="restore-backup-file" name="restore_backup_file" class="form-control">
                            <?php if (empty($backupList)): ?>
                                <option value="">No backups available</option>
                            <?php else: ?>
                                <?php foreach ($backupList as $backupFile): ?>
                                    <option value="<?php echo e($backupFile['name']); ?>"><?php echo e($backupFile['name']); ?></option>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </select>
                    </div>
                    <div class="form-group">
                        <label for="restore-confirm">Type RESTORE to confirm</label>
                        <input type="text" id="restore-confirm" name="restore_confirm" class="form-control" placeholder="RESTORE">
                    </div>
                </div>
                <div class="table-actions">
                    <button type="submit" class="btn btn-danger" <?php echo empty($backupList) ? 'disabled' : ''; ?>>Restore Database</button>
                </div>
            </form>
        </div>
    </div>

    <div class="card">
        <div class="card-header">
            <h3>Available Backups</h3>
            <form method="POST" action="/admin/backup.php" class="table-actions" data-confirm="Delete all backup files? This cannot be undone.">
                <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                <input type="hidden" name="action" value="delete_all_backups">
                <button type="submit" class="btn btn-danger btn-sm" <?php echo empty($backupList) ? 'disabled' : ''; ?>>Delete All Backups</button>
            </form>
        </div>
        <div class="card-body">
            <?php if (empty($backupList)): ?>
                <p class="text-muted">No backups found.</p>
            <?php else: ?>
                <div class="table-responsive">
                    <table class="admin-table">
                        <thead>
                            <tr>
                                <th>File</th>
                                <th>Size</th>
                                <th>Modified</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($backupList as $backupFile): ?>
                                <tr>
                                    <td><?php echo e($backupFile['name']); ?></td>
                                    <td><?php echo e(number_format($backupFile['size'] / 1024 / 1024, 2)); ?> MB</td>
                                    <td><?php echo e(date($displayDateFormat, $backupFile['modified'])); ?></td>
                                    <td>
                                        <a class="btn btn-outline btn-sm" href="/admin/backup.php?download=<?php echo e(urlencode($backupFile['name'])); ?>&amp;csrf_token=<?php echo e(getCsrfToken()); ?>">Download</a>
                                        <form method="POST" action="/admin/backup.php" class="inline-block" data-confirm="Delete this backup file?">
                                            <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                                            <input type="hidden" name="action" value="delete_backup">
                                            <input type="hidden" name="backup_file" value="<?php echo e($backupFile['name']); ?>">
                                            <button type="submit" class="btn btn-danger btn-sm">Delete</button>
                                        </form>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <div class="card">
        <div class="card-header">
            <h3>Security Notes</h3>
        </div>
        <div class="card-body">
            <ul class="text-muted backup-note-list">
                <li>Backups can contain personal data. Store them securely and limit retention.</li>
                <li>Site-only backups exclude upload files, but include the upload folders themselves.</li>
                <li>Database exports are app-level logical backups, not hosting provider snapshots. Large databases can take time and may hit PHP time or memory limits.</li>
                <li>Keep backups outside the web root to prevent public access.</li>
            </ul>
        </div>
    </div>
</div>

<?php AdminLayout::renderFooter(); ?>
