<?php
/**
 * Admin Panel - Product Attribute Names
 *
 * Manage predefined attribute names (e.g. Color, Size, Material) that are
 * available as selectable options when editing product filter-attributes.
 * This avoids free-text typos and keeps attribute naming consistent.
 *
 * Security: POST-only mutations, CSRF-verified, prepared statements,
 *           output escaped via e().
 */

if (!defined('ROOT_PATH')) {
    http_response_code(403);
    exit;
}

use NewSite\Admin\AdminLayout;
use NewSite\Auth\AdminAuth;
use NewSite\Database\DatabaseManager;
use NewSite\Logging\LogService;

AdminAuth::requireLogin();

$db = DatabaseManager::getWriteConnection();
$message = '';
$error = '';

// ── Handle POST actions ──────────────────────────────────────────
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    verifyCSRF();
    $postAction = trim((string)($_POST['action'] ?? ''));

    // ── Create new attribute name ────────────────────────────────
    if ($postAction === 'create') {
        $name = trim((string)($_POST['name'] ?? ''));
        $sortOrder = (int)($_POST['sort_order'] ?? 0);

        if ($name === '') {
            $error = 'Attribute name is required.';
        } else {
            try {
                $stmt = $db->prepare(
                    "INSERT INTO product_attribute_names (name, sort_order) VALUES (?, ?)"
                );
                $stmt->execute([$name, $sortOrder]);
                $message = 'Attribute "' . $name . '" created.';
                LogService::add('info', 'Attribute name created: ' . $name);
            } catch (\PDOException $e) {
                if (str_contains($e->getMessage(), 'unique') || str_contains($e->getMessage(), 'duplicate')) {
                    $error = 'An attribute with that name already exists.';
                } else {
                    $error = 'Database error while creating attribute.';
                    LogService::add('error', 'Attribute create failed: ' . $e->getMessage());
                }
            }
        }
    }

    // ── Update existing attribute name ───────────────────────────
    if ($postAction === 'update') {
        $id = (int)($_POST['id'] ?? 0);
        $name = trim((string)($_POST['name'] ?? ''));
        $sortOrder = (int)($_POST['sort_order'] ?? 0);

        if ($id < 1 || $name === '') {
            $error = 'Attribute ID and name are required.';
        } else {
            // Fetch old name to update product_attributes references
            $oldStmt = $db->prepare("SELECT name FROM product_attribute_names WHERE id = ?");
            $oldStmt->execute([$id]);
            $oldRow = $oldStmt->fetch(\PDO::FETCH_ASSOC);
            $oldName = $oldRow ? trim((string)($oldRow['name'] ?? '')) : '';

            try {
                $db->beginTransaction();

                $stmt = $db->prepare(
                    "UPDATE product_attribute_names SET name = ?, sort_order = ? WHERE id = ?"
                );
                $stmt->execute([$name, $sortOrder, $id]);

                // Keep product_attributes in sync: rename existing usages
                if ($oldName !== '' && $oldName !== $name) {
                    $renameStmt = $db->prepare(
                        "UPDATE product_attributes SET name = ? WHERE name = ?"
                    );
                    $renameStmt->execute([$name, $oldName]);
                }

                $db->commit();
                $message = 'Attribute updated.';
                LogService::add('info', 'Attribute name updated: ' . $oldName . ' → ' . $name);
            } catch (\PDOException $e) {
                $db->rollBack();
                if (str_contains($e->getMessage(), 'unique') || str_contains($e->getMessage(), 'duplicate')) {
                    $error = 'An attribute with that name already exists.';
                } else {
                    $error = 'Database error while updating attribute.';
                    LogService::add('error', 'Attribute update failed: ' . $e->getMessage());
                }
            }
        }
    }

    // ── Delete attribute name ────────────────────────────────────
    if ($postAction === 'delete') {
        $id = (int)($_POST['id'] ?? 0);
        if ($id > 0) {
            // Check usage count before deleting
            $nameStmt = $db->prepare("SELECT name FROM product_attribute_names WHERE id = ?");
            $nameStmt->execute([$id]);
            $nameRow = $nameStmt->fetch(\PDO::FETCH_ASSOC);
            $deleteName = $nameRow ? trim((string)($nameRow['name'] ?? '')) : '';

            $usageCount = 0;
            if ($deleteName !== '') {
                $usageStmt = $db->prepare("SELECT COUNT(*) FROM product_attributes WHERE name = ?");
                $usageStmt->execute([$deleteName]);
                $usageCount = (int)$usageStmt->fetchColumn();
            }

            $stmt = $db->prepare("DELETE FROM product_attribute_names WHERE id = ?");
            $stmt->execute([$id]);

            $suffix = $usageCount !== 1 ? 's' : '';
            $note = $usageCount > 0
                ? ' (was used in ' . $usageCount . ' product attribute' . $suffix . ' — those values are kept)'
                : '';
            $message = 'Attribute "' . $deleteName . '" deleted.' . $note;
            LogService::add('info', 'Attribute name deleted: ' . $deleteName);
        }
    }

    // PRG redirect to avoid form re-submission
    if ($error === '') {
        header('Location: /admin/attributes.php?message=' . urlencode($message));
        exit;
    }
}

// ── Fetch all attribute names ────────────────────────────────────
$attributes = $db->query(
    "SELECT * FROM product_attribute_names ORDER BY sort_order ASC, name ASC"
)->fetchAll(\PDO::FETCH_ASSOC);

// Fetch usage counts per attribute name
$usageCounts = [];
$usageStmt = $db->query(
    "SELECT name, COUNT(*) AS cnt FROM product_attributes GROUP BY name"
);
foreach ($usageStmt->fetchAll(\PDO::FETCH_ASSOC) as $row) {
    $usageCounts[trim((string)$row['name'])] = (int)$row['cnt'];
}

// Check for message via GET (after PRG redirect)
if (isset($_GET['message']) && $message === '') {
    $message = trim((string)$_GET['message']);
}

AdminLayout::renderHeader();
?>

<div class="admin-content">

    <div class="content-header">
        <h1>Product Attributes</h1>
        <p>Manage predefined attribute names. These appear as selectable options when adding attributes to products.</p>
    </div>

    <?php if ($message !== ''): ?>
        <div class="admin-alert-success"><?php echo e($message); ?></div>
    <?php endif; ?>
    <?php if ($error !== ''): ?>
        <div class="admin-alert-danger"><?php echo e($error); ?></div>
    <?php endif; ?>

    <!-- Add New Attribute -->
    <div class="card mb-20">
        <div class="card-header"><h3>Add New Attribute Name</h3></div>
        <div class="card-body">
            <form method="POST">
                <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                <input type="hidden" name="action" value="create">
                <div class="attr-form-row">
                    <div class="form-group attr-name-col">
                        <label for="new-attr-name">Name</label>
                        <input type="text" id="new-attr-name" name="name" class="form-control" placeholder="e.g. Color, Size, Material" required maxlength="200">
                    </div>
                    <div class="form-group attr-sort-col">
                        <label for="new-attr-sort">Sort Order</label>
                        <input type="number" id="new-attr-sort" name="sort_order" class="form-control" value="0" min="0" max="9999">
                    </div>
                    <div class="form-group attr-action-col">
                        <button type="submit" class="btn btn-primary">Add Attribute</button>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <!-- Existing Attributes List -->
    <div class="card">
        <div class="card-header"><h3>Existing Attributes (<?php echo count($attributes); ?>)</h3></div>
        <div class="card-body">
            <?php if (empty($attributes)): ?>
                <p class="text-muted">No attribute names defined yet. Add one above.</p>
            <?php else: ?>
                <table class="admin-table">
                    <thead>
                        <tr>
                            <th scope="col">Name</th>
                            <th scope="col">Sort Order</th>
                            <th scope="col">Used In</th>
                            <th scope="col">Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($attributes as $attr): ?>
                            <tr>
                                <form method="POST">
                                    <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                                    <input type="hidden" name="action" value="update">
                                    <input type="hidden" name="id" value="<?php echo (int)$attr['id']; ?>">
                                    <td>
                                        <input type="text" name="name" class="form-control" value="<?php echo e($attr['name']); ?>" required maxlength="200">
                                    </td>
                                    <td>
                                        <input type="number" name="sort_order" class="form-control" value="<?php echo (int)$attr['sort_order']; ?>" min="0" max="9999">
                                    </td>
                                    <td>
                                        <?php
                                        $usage = $usageCounts[trim((string)$attr['name'])] ?? 0;
                            $usageSuffix = $usage !== 1 ? 's' : '';
                            echo $usage > 0
                                ? $usage . ' product' . $usageSuffix
                                : '<span class="text-muted">None</span>';
                            ?>
                                    </td>
                                    <td class="admin-actions-cell">
                                        <button type="submit" class="btn btn-secondary btn-sm">Save</button>
                                </form>
                                <form method="POST" class="inline-block">
                                    <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                                    <input type="hidden" name="action" value="delete">
                                    <input type="hidden" name="id" value="<?php echo (int)$attr['id']; ?>">
                                    <button type="submit" class="btn btn-danger btn-sm">Delete</button>
                                </form>
                                    </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>
    </div>

</div><!-- /.admin-content -->

<?php AdminLayout::renderFooter(); ?>
