/**
 * Universal Media Picker Component — external JS.
 *
 * Media library data and configuration are read from data-* attributes
 * on the #media-picker-modal element set by AdminLayout::renderFooter().
 * This eliminates inline scripts and is fully CSP-safe.
 */

/* global showNotification */

// ── Bootstrap data from modal data attributes ────────────────────
const mediaPickerModal = document.getElementById("media-picker-modal");
const videoThumbnailMode =
  mediaPickerModal?.dataset.videoThumbnailMode || "auto";
const allowClientThumbnail =
  videoThumbnailMode === "auto" || videoThumbnailMode === "client";

const mediaLibrary = (() => {
  if (!mediaPickerModal?.dataset.mediaLibrary) {
    return { images: [], videos: [], files: [] };
  }
  try {
    return JSON.parse(mediaPickerModal.dataset.mediaLibrary);
  } catch (e) {
    console.error("Failed to parse media library data", e);
    return { images: [], videos: [], files: [] };
  }
})();

// ── State variables ──────────────────────────────────────────────
let mediaPickerCallback = null;
let mediaPickerType = "image";
let mediaUploadInProgress = false;
let mediaPickerClickLock = false;
let mediaUploadQueue = [];
let mediaUploadQueueActive = false;
let mediaPickerMultiMode = false;
let mediaPickerSelectedPaths = new Set();
let mediaUploadQueueTotal = 0;
let mediaUploadCurrentIndex = 0;
let mediaUploadCurrentName = "";
const mediaUploadQueueSignatures = new Set();

// ── Core helpers ─────────────────────────────────────────────────

function getMediaTypeConfig(type) {
  if (type === "video") {
    return {
      title: "Select Video",
      showVideoTabs: true,
      accept: "video/*",
      urlLabel: "Enter video URL",
    };
  }
  if (type === "file") {
    return {
      title: "Select File",
      showVideoTabs: false,
      accept: "*/*",
      urlLabel: "Enter file URL",
    };
  }
  return {
    title: "Select Image",
    showVideoTabs: false,
    accept: "image/*,video/*",
    urlLabel: "Enter Image/Video URL",
  };
}

function applyMediaTypeConfig(config) {
  const title = document.getElementById("media-picker-title");
  if (title) {
    title.textContent = config.title;
  }

  document.querySelectorAll(".tab-video-only").forEach((el) => {
    el.classList.toggle("is-hidden", !config.showVideoTabs);
  });

  const fileInput = document.getElementById("media-picker-file");
  if (fileInput) {
    fileInput.accept = config.accept;
  }

  const urlLabel = document.getElementById("media-url-label");
  if (urlLabel) {
    urlLabel.textContent = config.urlLabel;
  }
}

// ── Core functions ───────────────────────────────────────────────

function openMediaPicker(inputElement, type = "image", options = {}) {
  mediaPickerType = type;
  mediaPickerCallback = inputElement;
  mediaPickerMultiMode = !!options?.multi;
  mediaPickerSelectedPaths = new Set();
  if (mediaPickerMultiMode && Array.isArray(options?.preselected)) {
    options.preselected.forEach((path) => {
      mediaPickerSelectedPaths.add(normalizeMediaPath(path));
    });
  }

  applyMediaTypeConfig(getMediaTypeConfig(type));

  const modal = document.getElementById("media-picker-modal");
  if (modal) {
    modal.classList.remove("is-hidden");
  }
  populateMediaLibrary();
  switchMediaTab("upload");
  updateMediaPickerSelectionUi();
  syncUploadZoneState();

  setTimeout(initMediaPickerHandlers, 50);
  setTimeout(bindMediaPickerUi, 50);
}

function closeMediaPicker() {
  const modal = document.getElementById("media-picker-modal");
  if (modal) {
    modal.classList.add("is-hidden");
  }
  mediaPickerCallback = null;
  resetUploadZone();
  mediaPickerMultiMode = false;
  mediaPickerSelectedPaths = new Set();
  updateMediaPickerSelectionUi();
}

function bindMediaPickerUi() {
  document.querySelectorAll("[data-action]").forEach((el) => {
    if (el._mediaPickerBound) {
      return;
    }
    el._mediaPickerBound = true;
    const action = el.dataset.action;
    if (action === "close-media-picker") {
      el.addEventListener("click", closeMediaPicker);
    }
    if (action === "select-external-url") {
      el.addEventListener("click", selectExternalUrl);
    }
    if (action === "select-youtube-video") {
      el.addEventListener("click", selectYouTubeVideo);
    }
    if (action === "clear-media-selection") {
      el.addEventListener("click", clearMediaSelection);
    }
    if (action === "confirm-media-selection") {
      el.addEventListener("click", confirmMediaSelection);
    }
  });

  document.querySelectorAll("[data-tab]").forEach((btn) => {
    if (btn._mediaPickerBound) {
      return;
    }
    btn._mediaPickerBound = true;
    btn.addEventListener("click", () => {
      switchMediaTab(btn.dataset.tab);
    });
  });

  const grid = document.getElementById("media-library-grid");
  if (grid && !grid._mediaPickerBound) {
    grid._mediaPickerBound = true;
    grid.addEventListener("click", (e) => {
      const item = e.target.closest(".media-library-item");
      if (!item) {
        return;
      }
      const path = item.dataset.path;
      if (mediaPickerMultiMode) {
        toggleMediaSelection(path, item);
      } else {
        selectMediaFile(path);
      }
    });
  }
}

function switchMediaTab(tabName) {
  document.querySelectorAll(".media-tab-content").forEach((t) => {
    t.classList.remove("active");
  });
  document.querySelectorAll(".tab-btn").forEach((b) => {
    b.classList.remove("active");
  });

  const target = document.getElementById("media-tab-" + tabName);
  if (target) {
    target.classList.add("active");
  }

  const btn = document.querySelector('[data-tab="' + tabName + '"]');
  if (btn) {
    btn.classList.add("active");
  }

  if (tabName === "library") {
    populateMediaLibrary();
  }
}

function syncUploadZoneState() {
  if (!mediaUploadInProgress) {
    return;
  }
  const zone = document.getElementById("media-upload-zone");
  const progress = document.getElementById("media-upload-progress");
  const status = document.getElementById("media-upload-status");
  if (!zone || !progress) {
    return;
  }
  const content = zone.querySelector(".upload-zone-content");
  if (content) {
    content.classList.add("is-hidden");
  }
  progress.classList.remove("is-hidden");
  if (status) {
    status.textContent = getUploadStatusText();
  }
}

// ── File signature & status helpers ──────────────────────────────

function getFileSignature(file) {
  if (!file) {
    return "";
  }
  const name = file.name || "";
  const size = typeof file.size === "number" ? file.size : 0;
  const modified =
    typeof file.lastModified === "number" ? file.lastModified : 0;
  return name + "|" + size + "|" + modified;
}

function getUploadStatusText(percent) {
  const total = mediaUploadQueueTotal || (mediaUploadInProgress ? 1 : 0);
  const index = mediaUploadCurrentIndex || 1;
  const name = mediaUploadCurrentName ? ": " + mediaUploadCurrentName : "";
  const countLabel = total > 1 ? " " + index + " of " + total : "";
  const percentLabel = typeof percent === "number" ? " " + percent + "%" : "";
  return "Uploading" + countLabel + name + percentLabel;
}

function getProcessingStatusText() {
  const total = mediaUploadQueueTotal || (mediaUploadInProgress ? 1 : 0);
  const index = mediaUploadCurrentIndex || 1;
  const countLabel = total > 1 ? " " + index + " of " + total : "";
  return "Processing on server" + countLabel + "...";
}

function normalizeMediaPath(path) {
  return typeof path === "string" ? path.trim() : "";
}

// ── Selection management ─────────────────────────────────────────

function updateMediaPickerSelectionUi() {
  const actions = document.getElementById("media-picker-actions");
  if (actions) {
    actions.classList.toggle("is-hidden", !mediaPickerMultiMode);
  }
  const count = document.getElementById("media-picker-selected-count");
  if (count) {
    const size = mediaPickerSelectedPaths.size;
    count.textContent = size + " selected";
  }
}

function toggleMediaSelection(path, item) {
  const cleanPath = normalizeMediaPath(path);
  if (!cleanPath) {
    return;
  }
  if (mediaPickerSelectedPaths.has(cleanPath)) {
    mediaPickerSelectedPaths.delete(cleanPath);
    if (item) {
      item.classList.remove("selected");
    }
  } else {
    mediaPickerSelectedPaths.add(cleanPath);
    if (item) {
      item.classList.add("selected");
    }
  }
  updateMediaPickerSelectionUi();
}

function clearMediaSelection() {
  mediaPickerSelectedPaths.clear();
  document
    .querySelectorAll("#media-library-grid .media-library-item.selected")
    .forEach((item) => {
      item.classList.remove("selected");
    });
  updateMediaPickerSelectionUi();
}

function confirmMediaSelection() {
  if (!mediaPickerMultiMode) {
    return;
  }
  const selectedPaths = Array.from(mediaPickerSelectedPaths);
  if (!selectedPaths.length) {
    showNotification("Select at least one file", "error");
    return;
  }
  if (typeof mediaPickerCallback === "function") {
    mediaPickerCallback(selectedPaths);
  } else if (mediaPickerCallback) {
    mediaPickerCallback.value = selectedPaths[0];
    mediaPickerCallback.dispatchEvent(new Event("change", { bubbles: true }));
  }
  closeMediaPicker();
  showNotification("Media added!", "success");
}

// ── Library rendering ────────────────────────────────────────────

function populateMediaLibrary() {
  const grid = document.getElementById("media-library-grid");
  const empty = document.getElementById("media-library-empty");
  let files;

  if (mediaPickerType === "video") {
    files = mediaLibrary.videos;
  } else if (mediaPickerType === "file") {
    files = mediaLibrary.files;
  } else {
    files = [].concat(mediaLibrary.images, mediaLibrary.videos);
  }

  if (files.length === 0) {
    grid.innerHTML = "";
    empty.classList.remove("is-hidden");
    return;
  }

  empty.classList.add("is-hidden");
  grid.innerHTML = files
    .map((file) => {
      const isVideo = file.file_type?.startsWith("video/");
      const isImage = file.file_type?.startsWith("image/");
      const mediaPath = normalizeMediaPath(file.file_path);
      const selectedClass =
        mediaPickerMultiMode &&
        mediaPath &&
        mediaPickerSelectedPaths.has(mediaPath)
          ? " selected"
          : "";

      if (mediaPickerType === "file" && !isImage && !isVideo) {
        return (
          '<div class="media-library-item is-file' +
          selectedClass +
          '" data-path="' +
          file.file_path +
          '">' +
          '<div class="file-icon">\uD83D\uDCC4</div>' +
          '<span class="file-name">' +
          file.original_name +
          "</span>" +
          "</div>"
        );
      }

      if (isVideo) {
        const displayElement = file.thumbnail_path
          ? '<img src="' +
            file.thumbnail_path +
            '" alt="' +
            file.original_name +
            '" loading="lazy">'
          : '<video src="' + file.file_path + '" muted></video>';
        return (
          '<div class="media-library-item' +
          selectedClass +
          '" data-path="' +
          file.file_path +
          '">' +
          displayElement +
          '<span class="video-indicator">\u25B6\uFE0F</span>' +
          '<span class="file-name">' +
          file.original_name +
          "</span>" +
          "</div>"
        );
      }

      const thumbSrc = file.thumbnail_path || file.file_path;
      return (
        '<div class="media-library-item' +
        selectedClass +
        '" data-path="' +
        file.file_path +
        '">' +
        '<img src="' +
        thumbSrc +
        '" alt="' +
        file.original_name +
        '" loading="lazy">' +
        '<span class="file-name">' +
        file.original_name +
        "</span>" +
        "</div>"
      );
    })
    .join("");
}

// ── File selection ───────────────────────────────────────────────

function selectMediaFile(path) {
  if (mediaPickerCallback) {
    if (typeof mediaPickerCallback === "function") {
      mediaPickerCallback(path);
    } else {
      mediaPickerCallback.value = path;
      mediaPickerCallback.dispatchEvent(new Event("change", { bubbles: true }));
    }
  }
  closeMediaPicker();
  showNotification("Media selected!", "success");
}

function selectExternalUrl() {
  const url = document.getElementById("media-external-url").value.trim();
  if (!url) {
    showNotification("Please enter a URL", "error");
    return;
  }
  selectMediaFile(url);
}

function selectYouTubeVideo() {
  const input = document.getElementById("media-youtube-url").value.trim();
  const videoId = extractYouTubeId(input);
  if (!videoId) {
    showNotification("Invalid YouTube URL", "error");
    return;
  }
  const embedUrl = "https://www.youtube.com/embed/" + videoId;
  selectMediaFile(embedUrl);
}

function extractYouTubeId(url) {
  const patterns = [
    /(?:youtube\.com\/watch\?v=|youtu\.be\/|youtube\.com\/embed\/)([a-zA-Z0-9_-]{11})/,
    /^([a-zA-Z0-9_-]{11})$/,
  ];
  for (const pattern of patterns) {
    const match = url.match(pattern);
    if (match) {
      return match[1];
    }
  }
  return null;
}

// ── YouTube / URL preview listeners ──────────────────────────────

function initYouTubePreview() {
  const ytInput = document.getElementById("media-youtube-url");
  if (ytInput) {
    ytInput.addEventListener("input", () => {
      const videoId = extractYouTubeId(ytInput.value);
      const preview = document.getElementById("youtube-preview");
      const frame = document.getElementById("youtube-preview-frame");
      if (videoId) {
        frame.src = "https://www.youtube.com/embed/" + videoId;
        preview.classList.remove("is-hidden");
      } else {
        preview.classList.add("is-hidden");
      }
    });
  }
}

function initUrlPreview() {
  const urlInput = document.getElementById("media-external-url");
  if (urlInput) {
    urlInput.addEventListener("input", () => {
      const url = urlInput.value.trim();
      const preview = document.getElementById("media-url-preview");
      if (
        url &&
        (url.match(/\.(jpg|jpeg|png|gif|webp|svg)$/i) ||
          url.match(/\.(mp4|webm|ogg)$/i))
      ) {
        const isVideo = url.match(/\.(mp4|webm|ogg)$/i);
        preview.innerHTML = "";
        if (isVideo) {
          const video = document.createElement("video");
          video.src = url;
          video.controls = true;
          video.className = "media-preview-video";
          preview.appendChild(video);
        } else {
          const img = document.createElement("img");
          img.src = url;
          img.alt = "Preview";
          img.className = "media-preview-image";
          img.onerror = () => {
            preview.classList.add("is-hidden");
          };
          preview.appendChild(img);
        }
        preview.classList.remove("is-hidden");
      } else {
        preview.classList.add("is-hidden");
      }
    });
  }
}

// ── CSRF helper ──────────────────────────────────────────────────

function getMediaPickerCsrfToken() {
  const meta = document.querySelector('meta[name="csrf-token"]');
  const token = meta?.getAttribute("content");
  if (token) {
    return token;
  }
  const input = document.querySelector('input[name="csrf_token"]');
  return input?.value ?? "";
}

// ── Upload queue ─────────────────────────────────────────────────

function enqueueMediaUploads(fileList) {
  if (!fileList?.length) {
    return;
  }
  const files = Array.from(fileList);
  const items = files
    .map((file) => ({
      file,
      selectOnSuccess: files.length === 1,
      signature: getFileSignature(file),
    }))
    .filter((item) => {
      if (!item.signature) {
        return true;
      }
      if (mediaUploadQueueSignatures.has(item.signature)) {
        return false;
      }
      mediaUploadQueueSignatures.add(item.signature);
      return true;
    });
  if (!items.length) {
    return;
  }
  if (!mediaUploadQueueActive && !mediaUploadInProgress) {
    mediaUploadCurrentIndex = 0;
  }
  const currentInProgress = mediaUploadInProgress ? 1 : 0;
  mediaUploadQueueTotal =
    mediaUploadQueue.length + items.length + currentInProgress;
  mediaUploadQueue = mediaUploadQueue.concat(items);
  processMediaUploadQueue();
}

async function processMediaUploadQueue() {
  if (mediaUploadQueueActive) {
    return;
  }
  mediaUploadQueueActive = true;
  while (mediaUploadQueue.length) {
    const nextItem = mediaUploadQueue.shift();
    mediaUploadCurrentIndex += 1;
    mediaUploadCurrentName = nextItem.file?.name || "";
    await queueMediaUpload(nextItem.file, nextItem.selectOnSuccess);
    if (nextItem.signature) {
      mediaUploadQueueSignatures.delete(nextItem.signature);
    }
  }
  mediaUploadQueueActive = false;
  mediaUploadQueueTotal = 0;
  mediaUploadCurrentIndex = 0;
  mediaUploadCurrentName = "";
  mediaUploadQueueSignatures.clear();
}

// ── Client-side video thumbnail ──────────────────────────────────

function generateClientSideThumbnail(videoFile, callback) {
  const video = document.createElement("video");
  const canvas = document.createElement("canvas");
  const ctx = canvas.getContext("2d");

  video.preload = "metadata";
  video.muted = true;

  video.onloadeddata = () => {
    video.currentTime = Math.min(1, video.duration * 0.1);
  };

  video.onseeked = () => {
    canvas.width = 320;
    canvas.height = (video.videoHeight / video.videoWidth) * 320;
    ctx.drawImage(video, 0, 0, canvas.width, canvas.height);
    canvas.toBlob(
      (blob) => {
        URL.revokeObjectURL(video.src);
        callback(blob);
      },
      "image/jpeg",
      0.85,
    );
  };

  video.onerror = () => {
    URL.revokeObjectURL(video.src);
    callback(null);
  };

  video.src = URL.createObjectURL(videoFile);
}

function queueMediaUpload(file, selectOnSuccess) {
  return new Promise((resolve) => {
    if (file?.type?.startsWith("video/") && allowClientThumbnail) {
      generateClientSideThumbnail(file, (thumbnailBlob) => {
        uploadMediaFile(file, selectOnSuccess, thumbnailBlob).then(resolve);
      });
    } else {
      uploadMediaFile(file, selectOnSuccess, null).then(resolve);
    }
  });
}

// ── Upload via XHR ───────────────────────────────────────────────

function handleUploadHttpError(xhrStatus, response) {
  let statusMessage = "Upload failed: Server returned status " + xhrStatus;
  if (xhrStatus === 413) {
    statusMessage = "Upload failed: File exceeds server limits.";
  }
  if (response?.message) {
    statusMessage = response.message;
  }
  showNotification(statusMessage, "error");
}

function addUploadedFileToLibrary(file) {
  if (file.type.startsWith("image/")) {
    mediaLibrary.images.unshift(file);
  } else if (file.type.startsWith("video/")) {
    mediaLibrary.videos.unshift(file);
  }
  if (Array.isArray(mediaLibrary.files)) {
    mediaLibrary.files.unshift(file);
  }
  populateMediaLibrary();
}

function handleUploadSuccess(responseFile, selectOnSuccess) {
  addUploadedFileToLibrary(responseFile);
  if (mediaPickerMultiMode) {
    const cleanPath = normalizeMediaPath(responseFile.path);
    if (cleanPath) {
      mediaPickerSelectedPaths.add(cleanPath);
    }
    updateMediaPickerSelectionUi();
    resetUploadZone();
    mediaUploadInProgress = false;
    showNotification(
      "Upload complete: " + responseFile.original_name,
      "success",
    );
    return;
  }
  resetUploadZone();
  mediaUploadInProgress = false;
  if (selectOnSuccess) {
    selectMediaFile(responseFile.path);
  } else {
    showNotification(
      "Upload complete: " + responseFile.original_name,
      "success",
    );
  }
}

function finishUploadWithError(message) {
  showNotification(message, "error");
  mediaUploadInProgress = false;
  resetUploadZone();
}

function handleXhrOnload(xhr, selectOnSuccess, resolve) {
  let response = null;
  try {
    response = JSON.parse(xhr.responseText);
  } catch (parseError) {
    finishUploadWithError("Upload failed: " + parseError.message);
    resolve();
    return;
  }
  if (xhr.status !== 200) {
    handleUploadHttpError(xhr.status, response);
    mediaUploadInProgress = false;
    resetUploadZone();
    resolve();
    return;
  }
  if (response.success && response.file) {
    handleUploadSuccess(response.file, selectOnSuccess);
  } else {
    finishUploadWithError(response.message || "Upload failed");
  }
  resolve();
}

function buildAndSendXhr(formData, fill, status, selectOnSuccess, resolve) {
  const xhr = new XMLHttpRequest();

  xhr.upload.addEventListener("progress", (e) => {
    if (e.lengthComputable) {
      const percent = Math.round((e.loaded / e.total) * 100);
      fill.value = percent;
      if (status) {
        status.textContent = getUploadStatusText(percent);
      }
    }
  });

  xhr.upload.addEventListener("loadend", () => {
    if (mediaUploadInProgress && status) {
      status.textContent = getProcessingStatusText();
    }
  });

  xhr.onload = () => handleXhrOnload(xhr, selectOnSuccess, resolve);

  xhr.onerror = () => {
    finishUploadWithError("Upload failed");
    resolve();
  };

  xhr.open("POST", "/admin/files.php");
  xhr.setRequestHeader("Accept", "application/json");
  xhr.send(formData);
}

function buildUploadFormData(file, thumbnailBlob) {
  const formData = new FormData();
  formData.append("file", file);
  formData.append("ajax", "1");
  if (thumbnailBlob) {
    formData.append("thumbnail", thumbnailBlob, "thumbnail.jpg");
  }
  const csrfToken = getMediaPickerCsrfToken();
  if (csrfToken) {
    formData.append("csrf_token", csrfToken);
  }
  return formData;
}

async function uploadMediaFile(
  file,
  selectOnSuccess = true,
  thumbnailBlob = null,
) {
  if (!file) {
    return;
  }

  mediaUploadInProgress = true;

  const progress = document.getElementById("media-upload-progress");
  const fill = document.getElementById("media-progress-fill");
  const status = document.getElementById("media-upload-status");
  const zone = document.getElementById("media-upload-zone");

  zone.querySelector(".upload-zone-content").classList.add("is-hidden");
  progress.classList.remove("is-hidden");
  if (status) {
    status.textContent = getUploadStatusText();
  }

  const formData = buildUploadFormData(file, thumbnailBlob);

  return new Promise((resolve) => {
    buildAndSendXhr(formData, fill, status, selectOnSuccess, resolve);
  });
}

// ── Reset upload UI ──────────────────────────────────────────────

function resetUploadZone() {
  mediaUploadInProgress = false;
  const zone = document.getElementById("media-upload-zone");
  if (zone) {
    const content = zone.querySelector(".upload-zone-content");
    if (content) {
      content.classList.remove("is-hidden");
    }
  }
  const progress = document.getElementById("media-upload-progress");
  if (progress) {
    progress.classList.add("is-hidden");
  }
  const fill = document.getElementById("media-progress-fill");
  if (fill) {
    fill.value = 0;
  }
  const statusEl = document.getElementById("media-upload-status");
  if (statusEl) {
    statusEl.textContent = "Uploading...";
  }
}

// ── DOM-ready initialisation ─────────────────────────────────────

function initMediaPickerHandlers() {
  const fileInput = document.getElementById("media-picker-file");
  const uploadZone = document.getElementById("media-upload-zone");

  if (fileInput && !fileInput._mediaPickerBound) {
    fileInput._mediaPickerBound = true;
    fileInput.addEventListener("click", (e) => {
      e.stopPropagation();
      if (mediaPickerClickLock) {
        e.preventDefault();
        return;
      }
      mediaPickerClickLock = true;
      setTimeout(() => {
        mediaPickerClickLock = false;
      }, 300);
    });
    fileInput.onchange = () => {
      if (fileInput.files?.length) {
        enqueueMediaUploads(fileInput.files);
        fileInput.value = "";
      }
    };
  }

  if (uploadZone && !uploadZone._mediaPickerBound) {
    uploadZone._mediaPickerBound = true;

    const fileButton = uploadZone.querySelector(".file-input-button");
    if (fileButton) {
      fileButton.addEventListener("click", (e) => {
        e.stopPropagation();
      });
    }

    uploadZone.onclick = (e) => {
      if (
        e.target.closest(".file-input-button") ||
        e.target.closest('input[type="file"]')
      ) {
        return;
      }
      e.preventDefault();
      if (mediaPickerClickLock) {
        return;
      }
      const localInput = document.getElementById("media-picker-file");
      if (localInput) {
        localInput.click();
      }
    };

    uploadZone.ondragover = (e) => {
      e.preventDefault();
      e.stopPropagation();
      uploadZone.classList.add("drag-over");
    };

    uploadZone.ondragleave = (e) => {
      e.preventDefault();
      e.stopPropagation();
      uploadZone.classList.remove("drag-over");
    };

    uploadZone.ondrop = (e) => {
      e.preventDefault();
      e.stopPropagation();
      uploadZone.classList.remove("drag-over");
      if (e.dataTransfer.files?.length) {
        enqueueMediaUploads(e.dataTransfer.files);
      }
    };
  }
}

// ── Auto-attach to inputs with data-media-picker attribute ───────
function autoAttachMediaPickers() {
  document.querySelectorAll("[data-media-picker]").forEach((input) => {
    const type = input.dataset.mediaPicker || "image";
    const wrapper = document.createElement("div");
    wrapper.className = "input-with-picker";
    input.parentNode.insertBefore(wrapper, input);
    wrapper.appendChild(input);

    const btn = document.createElement("button");
    btn.type = "button";
    btn.className = "btn btn-secondary btn-picker";
    if (type === "video") {
      btn.innerHTML = "\uD83C\uDFAC Browse";
    } else if (type === "file") {
      btn.innerHTML = "\uD83D\uDCC1 Browse";
    } else {
      btn.innerHTML = "\uD83D\uDDBC\uFE0F Browse";
    }
    btn.onclick = () => {
      openMediaPicker(input, type);
    };
    wrapper.appendChild(btn);
  });
}

// ── Boot ─────────────────────────────────────────────────────────
if (document.readyState === "loading") {
  document.addEventListener("DOMContentLoaded", () => {
    initMediaPickerHandlers();
    bindMediaPickerUi();
    initYouTubePreview();
    initUrlPreview();
    autoAttachMediaPickers();
  });
} else {
  initMediaPickerHandlers();
  bindMediaPickerUi();
  initYouTubePreview();
  initUrlPreview();
  autoAttachMediaPickers();
}
