/**
 * Admin Panel JavaScript
 */

document.addEventListener("DOMContentLoaded", function () {
  initModals();
  initDropZones();
  initSortable();
  initConfirmDialogs();
  initProductEditor();
  initAdminSidebarToggle();
  initAdminTicketPolling();
  initChatImageLightbox();
  initLazyImages();
  initImageErrorHider();
});

const adminCsrfMeta = document.querySelector('meta[name="csrf-token"]');
const adminCsrfToken = adminCsrfMeta
  ? adminCsrfMeta.getAttribute("content")
  : null;

function isSameOriginRequest(input) {
  try {
    let url = "";
    if (typeof input === "string") {
      url = input;
    } else if (typeof input?.url === "string") {
      url = input.url;
    }
    if (!url) return true;
    if (url.startsWith("/")) return true;
    const parsed = new URL(url, globalThis.location.origin);
    return parsed.origin === globalThis.location.origin;
  } catch (error) {
    console.warn("Failed to verify request origin", error);
    return true;
  }
}

const originalAdminFetch = globalThis.fetch;
globalThis.fetch = function (...args) {
  const input = args[0];
  const init = args[1] || {};
  const method = (init.method || "GET").toUpperCase();
  const headers = new Headers(init.headers || {});
  if (
    adminCsrfToken &&
    method !== "GET" &&
    method !== "HEAD" &&
    isSameOriginRequest(input) &&
    !headers.has("X-CSRF-Token")
  ) {
    headers.set("X-CSRF-Token", adminCsrfToken);
  }
  init.headers = headers;
  args[1] = init;
  return originalAdminFetch.apply(this, args);
};

document.addEventListener(
  "submit",
  function (event) {
    const form = event.target;
    if (form?.tagName !== "FORM") return;
    const method = (form.getAttribute("method") || "GET").toUpperCase();
    if (method !== "POST") return;
    if (!adminCsrfToken) return;
    if (form.querySelector('input[name="csrf_token"]')) return;

    const input = document.createElement("input");
    input.type = "hidden";
    input.name = "csrf_token";
    input.value = adminCsrfToken;
    form.appendChild(input);
  },
  true,
);

/**
 * Lazy Image Loading with IntersectionObserver
 * Loads images only when they're visible and aborts on page exit
 */
let lazyImageObserver = null;
let pendingImageLoads = new Set();

function initLazyImages() {
  const lazyImages = document.querySelectorAll("img[data-src]");
  if (lazyImages.length === 0) return;

  if ("IntersectionObserver" in globalThis) {
    lazyImageObserver = new IntersectionObserver(
      function (entries) {
        entries.forEach(function (entry) {
          if (entry.isIntersecting) {
            const img = entry.target;
            loadImage(img);
            lazyImageObserver.unobserve(img);
          }
        });
      },
      {
        rootMargin: "100px 0px", // Start loading 100px before visible
        threshold: 0.01,
      },
    );

    lazyImages.forEach(function (img) {
      lazyImageObserver.observe(img);
    });
  } else {
    // Fallback for old browsers - load all immediately
    lazyImages.forEach(function (img) {
      loadImage(img);
    });
  }

  // Abort pending image loads when leaving the page
  globalThis.addEventListener("beforeunload", abortPendingImages);
  globalThis.addEventListener("pagehide", abortPendingImages);
}

function initImageErrorHider() {
  document.addEventListener(
    "error",
    function (event) {
      const target = event.target;
      if (
        target?.tagName === "IMG" &&
        target.dataset.hideOnError !== undefined
      ) {
        target.classList.add("is-hidden");
      }
    },
    true,
  );
}

function loadImage(img) {
  const src = img.dataset.src;
  if (!src) return;

  // Track this load
  pendingImageLoads.add(img);

  // Create a new image to preload
  const tempImg = new Image();
  tempImg.onload = function () {
    pendingImageLoads.delete(img);
    img.src = src;
    delete img.dataset.src;
    img.classList.add("lazy-loaded");
  };
  tempImg.onerror = function () {
    pendingImageLoads.delete(img);
    img.classList.add("lazy-error");
  };
  tempImg.src = src;
}

function abortPendingImages() {
  // Clear all pending loads
  pendingImageLoads.forEach(function (img) {
    img.src = "";
  });
  pendingImageLoads.clear();

  // Disconnect observer
  if (lazyImageObserver) {
    lazyImageObserver.disconnect();
  }
}

/**
 * Chat Image Lightbox
 * Opens chat images in a fullscreen overlay
 */
function initChatImageLightbox() {
  // Create lightbox element if it doesn't exist
  let lightbox = document.querySelector(".chat-lightbox");
  if (!lightbox) {
    lightbox = document.createElement("div");
    lightbox.className = "chat-lightbox";
    lightbox.innerHTML = `
            <button class="chat-lightbox-close" aria-label="Close">&times;</button>
            <img class="chat-lightbox-img" src="" alt="Full size image">
        `;
    document.body.appendChild(lightbox);
  }

  const lightboxImg = lightbox.querySelector(".chat-lightbox-img");
  const closeBtn = lightbox.querySelector(".chat-lightbox-close");

  // Close lightbox on button click
  closeBtn.addEventListener("click", function () {
    lightbox.classList.remove("is-open");
  });

  // Close lightbox on background click
  lightbox.addEventListener("click", function (e) {
    if (e.target === lightbox) {
      lightbox.classList.remove("is-open");
    }
  });

  // Close on Escape key
  document.addEventListener("keydown", function (e) {
    if (e.key === "Escape" && lightbox.classList.contains("is-open")) {
      lightbox.classList.remove("is-open");
    }
  });

  // Delegate click events to chat images
  document.addEventListener("click", function (e) {
    if (e.target.classList.contains("chat-image")) {
      e.preventDefault();
      lightboxImg.src = e.target.src;
      lightbox.classList.add("is-open");
    }
  });
}

/**
 * Modal Handling
 */
function initModals() {
  // Open modal buttons
  document.querySelectorAll("[data-modal]").forEach(function (btn) {
    btn.addEventListener("click", function (e) {
      e.preventDefault();
      const modalId = this.dataset.modal;
      openModal(modalId);
    });
  });

  // Close modal buttons
  document
    .querySelectorAll(".modal-close, .modal-cancel")
    .forEach(function (btn) {
      btn.addEventListener("click", function (e) {
        e.preventDefault();
        closeAllModals();
      });
    });

  // Close on overlay click
  document.querySelectorAll(".modal-overlay").forEach(function (overlay) {
    overlay.addEventListener("click", function (e) {
      if (e.target === this) {
        closeAllModals();
      }
    });
  });

  // Close on Escape key
  document.addEventListener("keydown", function (e) {
    if (e.key === "Escape") {
      closeAllModals();
    }
  });
}

function openModal(modalId) {
  const modal = document.getElementById(modalId);
  if (modal) {
    modal.classList.add("active");
    document.body.classList.add("is-modal-open");
  }
}

function closeAllModals() {
  document.querySelectorAll(".modal-overlay").forEach(function (modal) {
    modal.classList.remove("active");
  });
  document.body.classList.remove("is-modal-open");
}

function initAdminTicketPolling() {
  const container = document.querySelector("[data-ticket-chat-admin]");
  if (!container) {
    return;
  }

  const chatBox = container.querySelector("[data-ticket-chat-box]");
  const chatBody = container.querySelector("[data-ticket-chat-body]");
  const replyForm = container.querySelector(".ticket-reply-form");
  const listLinks = container.querySelectorAll(".ticket-list-link");
  const listItems = container.querySelectorAll("[data-ticket-item]");
  const toggleButton = container.querySelector("[data-ticket-toggle-list]");
  if (!chatBox || !chatBody) {
    return;
  }

  const userId = Number.parseInt(chatBox.dataset.ticketUserId || "0", 10);
  if (!userId) {
    return;
  }

  let lastId = Number.parseInt(chatBody.dataset.ticketLastId || "0", 10);
  let isFetching = false;

  function renderMessages(items) {
    const sorted = items
      .slice()
      .sort((a, b) => (a.created_at || 0) - (b.created_at || 0));
    chatBody.innerHTML = "";
    sorted.forEach((item) => {
      const isAdmin = (item.sender_type || "admin") === "admin";
      const bubble = document.createElement("div");
      bubble.className = `ticket-bubble ${isAdmin ? "ticket-bubble--admin" : "ticket-bubble--user"}`;
      bubble.dataset.ticketMessageId = String(item.id);

      const meta = document.createElement("div");
      meta.className = "ticket-bubble-meta";
      const date = item.created_at ? new Date(item.created_at * 1000) : null;
      meta.textContent = `${isAdmin ? "Admin" : "User"} · ${date ? date.toLocaleString() : ""}`;
      if (isAdmin && item.is_read) {
        const receipt = document.createElement("span");
        receipt.className = "read-receipt read-receipt--read";
        receipt.title = "Read";
        receipt.innerHTML =
          '<svg viewBox="0 0 16 16" fill="currentColor"><path d="M13.854 3.646a.5.5 0 0 1 0 .708l-7 7a.5.5 0 0 1-.708 0l-3.5-3.5a.5.5 0 1 1 .708-.708L6.5 10.293l6.646-6.647a.5.5 0 0 1 .708 0z"/></svg>';
        meta.appendChild(receipt);
      }

      bubble.appendChild(meta);

      // Render image if present
      if (item.image_url) {
        const imageWrap = document.createElement("div");
        imageWrap.className = "ticket-bubble-image";
        const img = document.createElement("img");
        img.src = item.image_url;
        img.alt = "Attached image";
        img.className = "chat-image";
        img.loading = "lazy";
        imageWrap.appendChild(img);
        bubble.appendChild(imageWrap);
      }

      const text = document.createElement("div");
      text.className = "ticket-bubble-text";
      text.textContent = item.body || "";

      bubble.appendChild(text);
      chatBody.appendChild(bubble);
    });

    if (sorted.length) {
      const last = sorted[sorted.length - 1];
      lastId = last.id;
      chatBody.dataset.ticketLastId = String(lastId);
    }

    // Auto-scroll to latest message
    chatBody.scrollTop = chatBody.scrollHeight;
  }

  async function fetchMessages() {
    if (isFetching) {
      return;
    }
    isFetching = true;
    try {
      const response = await fetch(`/admin/tickets-api.php?user_id=${userId}`);
      const data = await response.json();
      if (data?.success && Array.isArray(data.items)) {
        renderMessages(data.items);
      }
    } catch (err) {
      console.debug("Ticket polling failed", err);
    } finally {
      isFetching = false;
    }
  }

  if (replyForm) {
    const fileInput = replyForm.querySelector('input[name="chat_image"]');
    const previewWrap = replyForm.querySelector("[data-upload-preview]");
    const previewImg = previewWrap
      ? previewWrap.querySelector(".upload-preview-img")
      : null;
    const removeBtn = replyForm.querySelector("[data-remove-preview]");

    // Handle image preview
    if (fileInput) {
      fileInput.addEventListener("change", function () {
        const file = fileInput.files[0];
        if (file && previewWrap && previewImg) {
          // Validate file size (5MB)
          if (file.size > 5 * 1024 * 1024) {
            alert("Image must be under 5MB");
            fileInput.value = "";
            previewWrap.classList.add("is-hidden");
            return;
          }
          // Validate type
          const validTypes = [
            "image/jpeg",
            "image/png",
            "image/gif",
            "image/webp",
          ];
          if (!validTypes.includes(file.type)) {
            alert("Only JPG, PNG, GIF and WEBP images are allowed");
            fileInput.value = "";
            previewWrap.classList.add("is-hidden");
            return;
          }
          const reader = new FileReader();
          reader.onload = function (e) {
            previewImg.src = e.target.result;
            previewWrap.classList.remove("is-hidden");
          };
          reader.readAsDataURL(file);
        } else if (previewWrap) {
          previewWrap.classList.add("is-hidden");
        }
      });
    }

    // Remove preview button
    if (removeBtn) {
      removeBtn.addEventListener("click", function () {
        if (fileInput) {
          fileInput.value = "";
        }
        if (previewWrap) {
          previewWrap.classList.add("is-hidden");
        }
      });
    }

    replyForm.addEventListener("submit", function (event) {
      event.preventDefault();
      const formData = new FormData(replyForm);
      fetch("/admin/tickets.php", {
        method: "POST",
        body: formData,
      })
        .then(() => {
          const textarea = replyForm.querySelector(
            'textarea[name="reply_body"]',
          );
          if (textarea) {
            textarea.value = "";
          }
          // Clear file input and preview
          if (fileInput) {
            fileInput.value = "";
          }
          if (previewWrap) {
            previewWrap.classList.add("is-hidden");
          }
          fetchMessages();
        })
        .catch(() => {
          // ignore errors
        });
    });
  }

  let selectedUserId = Number.parseInt(
    container.dataset.selectedUserId || "0",
    10,
  );

  if (selectedUserId) {
    container.classList.add("is-collapsed");
  }

  function handleTicketSelect(link, event, fromItem) {
    const href = link.getAttribute("href") || "";
    const match = href.match(/user=(\d+)/);
    const linkUserId = match ? Number.parseInt(match[1], 10) : 0;

    if (linkUserId && linkUserId === selectedUserId) {
      event.preventDefault();
      container.classList.toggle("is-collapsed");
      return;
    }

    container.classList.add("is-collapsed");

    if (fromItem) {
      globalThis.location.href = href;
    }
  }

  listLinks.forEach((link) => {
    link.addEventListener("click", (event) => {
      handleTicketSelect(link, event, false);
    });
  });

  listItems.forEach((item) => {
    item.addEventListener("click", (event) => {
      if (event.target.closest("button")) {
        return;
      }
      const link = item.querySelector("[data-ticket-link]");
      if (!link) {
        return;
      }
      handleTicketSelect(link, event, true);
    });
  });

  if (toggleButton) {
    toggleButton.addEventListener("click", () => {
      container.classList.remove("is-collapsed");
      // Reset selected user so clicking any ticket will navigate to load it
      selectedUserId = 0;
      container.dataset.selectedUserId = "0";
      // Update URL to remove user parameter for consistent state
      const url = new URL(globalThis.location.href);
      url.searchParams.delete("user");
      globalThis.history.replaceState({}, "", url.toString());
    });
  }

  // Initial scroll to bottom for server-rendered messages
  chatBody.scrollTop = chatBody.scrollHeight;

  fetchMessages();
  setInterval(fetchMessages, 5000);
}

/**
 * File Upload Drop Zones
 */
function initDropZones() {
  const dropZones = document.querySelectorAll(".upload-zone");

  dropZones.forEach(function (zone) {
    // Skip if zone has custom handler (e.g., files.php)
    if (zone.dataset.bound) return;

    const input = zone.querySelector('input[type="file"]');

    zone.addEventListener("click", function () {
      if (input) input.click();
    });

    zone.addEventListener("dragover", function (e) {
      e.preventDefault();
      zone.classList.add("dragover");
    });

    zone.addEventListener("dragleave", function () {
      zone.classList.remove("dragover");
    });

    zone.addEventListener("drop", function (e) {
      e.preventDefault();
      zone.classList.remove("dragover");

      if (input && e.dataTransfer.files.length) {
        input.files = e.dataTransfer.files;
        input.dispatchEvent(new Event("change"));
      }
    });
  });
}

/**
 * Sortable Lists (Menu Items, Sections)
 */
function initSortable() {
  const sortableLists = document.querySelectorAll("[data-sortable]");

  sortableLists.forEach(function (list) {
    let draggedItem = null;

    list.addEventListener("dragstart", function (e) {
      if (
        e.target.classList.contains("menu-item") ||
        e.target.classList.contains("page-section-item")
      ) {
        draggedItem = e.target;
        e.target.classList.add("is-dragging");
      }
    });

    list.addEventListener("dragend", function (e) {
      if (draggedItem) {
        draggedItem.classList.remove("is-dragging");
        draggedItem = null;
      }
    });

    list.addEventListener("dragover", function (e) {
      e.preventDefault();
    });

    list.addEventListener("drop", function (e) {
      e.preventDefault();

      if (draggedItem) {
        const dropTarget = e.target.closest(".menu-item, .page-section-item");
        if (dropTarget && dropTarget !== draggedItem) {
          const rect = dropTarget.getBoundingClientRect();
          const midY = rect.top + rect.height / 2;

          if (e.clientY < midY) {
            dropTarget.parentNode.insertBefore(draggedItem, dropTarget);
          } else {
            dropTarget.parentNode.insertBefore(
              draggedItem,
              dropTarget.nextSibling,
            );
          }

          // Trigger save order event
          list.dispatchEvent(new CustomEvent("orderChanged"));
        }
      }
    });
  });
}

/**
 * Confirm Dialogs
 */
function initConfirmDialogs() {
  document.addEventListener(
    "submit",
    function (event) {
      const form = event.target;
      if (form?.tagName !== "FORM") return;
      const message = form.dataset.confirm;
      if (!message) return;
      if (!confirm(message)) {
        event.preventDefault();
        event.stopPropagation();
      }
    },
    true,
  );

  document.addEventListener(
    "click",
    function (event) {
      const element = event.target.closest("[data-confirm]");
      if (!element || element.tagName === "FORM") return;
      const message = element.dataset.confirm || "Are you sure?";
      if (!confirm(message)) {
        event.preventDefault();
        event.stopPropagation();
      }
    },
    true,
  );
}

/**
 * Admin Sidebar Toggle
 */
function initAdminSidebarToggle() {
  const toggle = document.querySelector("[data-admin-sidebar-toggle]");
  const overlay = document.querySelector("[data-admin-sidebar-overlay]");

  if (!toggle) {
    return;
  }

  const closeSidebar = () => {
    document.body.classList.remove("admin-sidebar-open");
  };

  toggle.addEventListener("click", () => {
    document.body.classList.toggle("admin-sidebar-open");
  });

  if (overlay) {
    overlay.addEventListener("click", closeSidebar);
  }

  document.addEventListener("keydown", (event) => {
    if (event.key === "Escape") {
      closeSidebar();
    }
  });

  globalThis.addEventListener("resize", () => {
    if (globalThis.innerWidth > 768) {
      closeSidebar();
    }
  });
}

/**
 * AJAX Form Submission
 */
function submitFormAjax(form, callback) {
  const formData = new FormData(form);

  fetch(form.action, {
    method: form.method || "POST",
    body: formData,
  })
    .then(function (response) {
      return response.json();
    })
    .then(function (data) {
      if (callback) callback(data);
    })
    .catch(function (error) {
      console.error("Error:", error);
      showNotification("An error occurred", "error");
    });
}

/**
 * Notifications
 */
function showNotification(message, type, durationMs) {
  type = type || "success";
  const duration = Number.isFinite(durationMs)
    ? Math.max(800, Math.min(15000, durationMs))
    : 5000;

  // Remove existing notifications
  const existing = document.querySelector(".notification");
  if (existing) existing.remove();

  const notification = document.createElement("div");
  notification.className = "notification notification-" + type;
  notification.innerHTML =
    "<span>" +
    message +
    '</span><button class="notification-close">&times;</button>';

  document.body.appendChild(notification);

  // Auto remove after configured duration
  setTimeout(function () {
    notification.remove();
  }, duration);

  // Manual close
  notification
    .querySelector(".notification-close")
    .addEventListener("click", function () {
      notification.remove();
    });
}

function renderProductTemplate(template, index, numberLabel) {
  if (!template) return null;

  let html = template.innerHTML.replaceAll("__INDEX__", String(index));
  if (numberLabel !== undefined) {
    html = html.replaceAll("__NUMBER__", String(numberLabel));
  }

  const wrapper = document.createElement("div");
  wrapper.innerHTML = html.trim();
  return wrapper.firstElementChild;
}

function attachVariantActionToggle(element) {
  if (!element) return;

  element.querySelectorAll("[data-variant-action]").forEach(function (select) {
    const wrapper = select.closest(".product-variant-item");
    const downloadField = wrapper
      ? wrapper.querySelector(".variant-download-field")
      : null;
    const linkLabelField = wrapper
      ? wrapper.querySelector(".variant-link-label-field")
      : null;
    const versionField = wrapper
      ? wrapper.querySelector(".variant-version-field")
      : null;
    const changelogRow = wrapper
      ? wrapper.querySelector(".variant-changelog-row")
      : null;
    const label = downloadField ? downloadField.querySelector("[data-download-label]") : null;
    const input = downloadField ? downloadField.querySelector("input") : null;

    const toggle = function () {
      const isDownload = select.value === "download";
      const isExternal = select.value === "external_link";

      if (downloadField) {
        downloadField.classList.toggle(
          "is-hidden",
          !(isDownload || isExternal),
        );
      }
      if (linkLabelField) {
        linkLabelField.classList.toggle("is-hidden", !isExternal);
      }
      if (versionField) {
        versionField.classList.toggle("is-hidden", !isDownload);
      }
      if (changelogRow) {
        changelogRow.classList.toggle("is-hidden", !isDownload);
      }

      if (label) {
        const downloadLabel = label.dataset.downloadLabel || "Download File";
        const linkLabel = label.dataset.linkLabel || "External Link URL";
        label.textContent = isExternal ? linkLabel : downloadLabel;
      }

      if (input) {
        const downloadPlaceholder =
          input.dataset.downloadPlaceholder || "/admin-file/file.zip";
        const linkPlaceholder =
          input.dataset.linkPlaceholder || "https://example.com";
        input.placeholder = isExternal ? linkPlaceholder : downloadPlaceholder;

        if (isExternal) {
          delete input.dataset.mediaPicker;
        } else {
          input.dataset.mediaPicker = "file";
        }
      }
    };

    select.addEventListener("change", toggle);
    toggle();
  });
}

function updateProductMediaThumb(input) {
  const card = input.closest(".product-media-item");
  if (!card) return;

  const img = card.querySelector(".product-media-thumb-img");
  if (!img) return;

  const val = (input.value || "").trim();
  const isYouTube = /youtu\.?be/.test(val);
  const ext = val.split(".").pop().toLowerCase();
  const isImage = ["jpg", "jpeg", "png", "gif", "webp", "svg", "avif"].includes(
    ext,
  );

  if (!isYouTube && isImage && val !== "") {
    img.src = val.replace("/admin-file/", "/site-file/");
    img.hidden = false;
    return;
  }

  img.src = "";
  img.hidden = true;
}

function bindAddMediaButton(
  addMediaBtn,
  mediaTemplate,
  mediaList,
  state,
  attachMediaPicker,
) {
  if (!addMediaBtn || !mediaTemplate) return;

  addMediaBtn.addEventListener("click", function () {
    const element = renderProductTemplate(
      mediaTemplate,
      state.mediaIndex++,
      null,
    );
    if (!element) return;
    mediaList.appendChild(element);
    attachMediaPicker(element);
  });
}

function bindAddMediaLibraryButton(
  addMediaLibraryBtn,
  collectMediaValues,
  addProductMediaItems,
) {
  if (!addMediaLibraryBtn || typeof openMediaPicker !== "function") return;

  addMediaLibraryBtn.addEventListener("click", function () {
    const preselected = collectMediaValues();
    openMediaPicker(
      function (selectedPaths) {
        addProductMediaItems(selectedPaths);
      },
      "media",
      { multi: true, preselected: preselected },
    );
  });
}

function bindAddBlockButton(addBlockBtn, blockTemplate, blockList, state) {
  if (!addBlockBtn || !blockTemplate || !blockList) return;

  addBlockBtn.addEventListener("click", function () {
    const element = renderProductTemplate(
      blockTemplate,
      state.blockIndex,
      state.blockIndex + 1,
    );
    if (!element) return;
    state.blockIndex += 1;
    blockList.appendChild(element);
  });
}

function bindAddVariantButton(
  addVariantBtn,
  variantTemplate,
  variantList,
  state,
  attachMediaPicker,
) {
  if (!addVariantBtn || !variantTemplate || !variantList) return;

  addVariantBtn.addEventListener("click", function () {
    const element = renderProductTemplate(
      variantTemplate,
      state.variantIndex++,
      null,
    );
    if (!element) return;
    variantList.appendChild(element);
    const anyChecked = variantList.querySelector(
      'input[name="variant_default"]:checked',
    );
    const newRadio = element.querySelector('input[name="variant_default"]');
    if (!anyChecked && newRadio) {
      newRadio.checked = true;
    }
    const unlimited = element.querySelector('input[name$="[unlimited]"]');
    const stockInput = element.querySelector('input[name$="[stock]"]');
    if (unlimited && stockInput) {
      stockInput.disabled = unlimited.checked;
    }
    attachMediaPicker(element);
    attachVariantActionToggle(element);
  });
}

function bindAddAttributeButton(
  addAttributeBtn,
  attributeTemplate,
  attributeList,
  state,
) {
  if (!addAttributeBtn || !attributeTemplate || !attributeList) return;

  addAttributeBtn.addEventListener("click", function () {
    const element = renderProductTemplate(
      attributeTemplate,
      state.attributeIndex++,
      null,
    );
    if (!element) return;
    attributeList.appendChild(element);
  });
}

function bindProductMediaListInteractions(mediaList) {
  mediaList.addEventListener("input", function (e) {
    const input = e.target.closest('input[name^="media_urls["]');
    if (input) updateProductMediaThumb(input);
  });

  mediaList
    .querySelectorAll('input[name^="media_urls["]')
    .forEach(updateProductMediaThumb);

  mediaList.addEventListener("click", function (e) {
    let btn = e.target.closest("[data-move-media]");
    if (btn) {
      const card = btn.closest(".product-media-item");
      if (!card) return;
      const direction = btn.dataset.moveMedia;
      if (direction === "up" && card.previousElementSibling) {
        mediaList.insertBefore(card, card.previousElementSibling);
      } else if (direction === "down" && card.nextElementSibling) {
        mediaList.insertBefore(card.nextElementSibling, card);
      }

      mediaList
        .querySelectorAll(".product-media-item")
        .forEach(function (item, idx) {
          const input = item.querySelector('input[name^="media_urls["]');
          if (input) input.name = "media_urls[" + idx + "]";
          item.dataset.index = String(idx);
        });
      return;
    }

    btn = e.target.closest('[data-remove="media"]');
    if (!btn) return;
    const card = btn.closest(".product-media-item");
    if (card) card.remove();
  });
}

function bindRemoveItemClick(list, removeType, itemSelector) {
  if (!list) return;

  list.addEventListener("click", function (e) {
    const btn = e.target.closest(`[data-remove="${removeType}"]`);
    if (!btn) return;
    const card = btn.closest(itemSelector);
    if (card) card.remove();
  });
}

function initVariantUnlimitedToggles() {
  document
    .querySelectorAll('input[name$="[unlimited]"]')
    .forEach(function (checkbox) {
      const toggle = function () {
        const wrapper = checkbox.closest(".product-variant-item");
        const stockInput = wrapper
          ? wrapper.querySelector('input[name$="[stock]"]')
          : null;
        if (stockInput) {
          stockInput.disabled = checkbox.checked;
        }
      };

      checkbox.addEventListener("change", toggle);
      toggle();
    });
}

function initProductStockToggle(productUnlimited, productStockInput) {
  if (!productUnlimited || !productStockInput) return;

  const toggleProductStock = function () {
    productStockInput.disabled = productUnlimited.checked;
  };
  productUnlimited.addEventListener("change", toggleProductStock);
  toggleProductStock();
}

function initProductActionSelectToggle(
  productActionSelect,
  productDownloadField,
  productVersionFields,
  productLinkLabelField,
) {
  if (!productActionSelect || !productDownloadField) return;

  const toggleDownloadField = function () {
    const actionValue = productActionSelect.value;
    const isDownload = actionValue === "download";
    const isExternal = actionValue === "external_link";
    const label = productDownloadField.querySelector("label");
    const input = productDownloadField.querySelector("input");

    productDownloadField.classList.toggle(
      "is-hidden",
      !(isDownload || isExternal),
    );
    if (productVersionFields) {
      productVersionFields.classList.toggle("is-hidden", !isDownload);
    }
    if (productLinkLabelField) {
      productLinkLabelField.classList.toggle("is-hidden", !isExternal);
    }

    if (label) {
      const downloadLabel = label.dataset.downloadLabel || "Download File";
      const linkLabel = label.dataset.linkLabel || "External Link URL";
      label.textContent = isExternal ? linkLabel : downloadLabel;
    }

    if (input) {
      const downloadPlaceholder =
        input.dataset.downloadPlaceholder || "/admin-file/file.zip";
      const linkPlaceholder =
        input.dataset.linkPlaceholder || "https://example.com";
      input.placeholder = isExternal ? linkPlaceholder : downloadPlaceholder;
      if (isExternal) {
        delete input.dataset.mediaPicker;
      } else {
        input.dataset.mediaPicker = "file";
      }
    }
  };

  productActionSelect.addEventListener("change", toggleDownloadField);
  toggleDownloadField();
}

/**
 * Product Editor (media, blocks, variants)
 */
function initProductEditor() {
  const mediaList = document.getElementById("product-media-list");
  if (!mediaList) return;

  const blockList = document.getElementById("product-blocks-list");
  const variantList = document.getElementById("product-variants-list");
  const attributeList = document.getElementById("product-attributes-list");
  const mediaTemplate = document.getElementById("product-media-template");
  const blockTemplate = document.getElementById("product-block-template");
  const variantTemplate = document.getElementById("product-variant-template");
  const attributeTemplate = document.getElementById(
    "product-attribute-template",
  );

  let mediaIndex = mediaList.querySelectorAll(".product-media-item").length;
  let blockIndex = blockList
    ? blockList.querySelectorAll(".product-block-item").length
    : 0;
  let variantIndex = variantList
    ? variantList.querySelectorAll(".product-variant-item").length
    : 0;
  let attributeIndex = attributeList
    ? attributeList.querySelectorAll(".product-attribute-item").length
    : 0;

  const addMediaBtn = document.getElementById("add-product-media");
  const addMediaLibraryBtn = document.getElementById(
    "add-product-media-library",
  );
  const addBlockBtn = document.getElementById("add-product-block");
  const addVariantBtn = document.getElementById("add-product-variant");
  const addAttributeBtn = document.getElementById("add-product-attribute");

  const state = {
    mediaIndex,
    blockIndex,
    variantIndex,
    attributeIndex,
  };

  function attachMediaPicker(element) {
    if (!element) return;
    element.querySelectorAll("[data-media-picker]").forEach(function (input) {
      if (input.closest(".input-with-picker")) return;
      const type = input.dataset.mediaPicker || "image";
      const wrapper = document.createElement("div");
      wrapper.className = "input-with-picker";
      input.parentNode.insertBefore(wrapper, input);
      wrapper.appendChild(input);

      const btn = document.createElement("button");
      btn.type = "button";
      btn.className = "btn btn-secondary btn-picker";
      if (type === "video") {
        btn.innerHTML = "🎬 Browse";
      } else if (type === "file") {
        btn.innerHTML = "📁 Browse";
      } else {
        btn.innerHTML = "🖼️ Browse";
      }
      btn.onclick = function () {
        openMediaPicker(input, type);
      };
      wrapper.appendChild(btn);
    });

    if (typeof initMediaPickerHandlers === "function") {
      setTimeout(initMediaPickerHandlers, 0);
    }
  }

  function collectMediaValues() {
    return Array.from(mediaList.querySelectorAll('input[name^="media_urls["]'))
      .map(function (input) {
        return (input.value || "").trim();
      })
      .filter(function (value) {
        return value !== "";
      });
  }

  function addProductMediaItems(paths) {
    if (!paths) return;
    const items = Array.isArray(paths) ? paths : [paths];
    const normalized = items
      .map(function (path) {
        return typeof path === "string" ? path.trim() : "";
      })
      .filter(function (path) {
        return path !== "";
      });
    if (!normalized.length) return;

    const inputs = Array.from(
      mediaList.querySelectorAll('input[name^="media_urls["]'),
    );
    const existing = new Set(
      inputs
        .map(function (input) {
          return (input.value || "").trim();
        })
        .filter(function (value) {
          return value !== "";
        }),
    );
    const emptyInputs = inputs.filter(function (input) {
      return (input.value || "").trim() === "";
    });

    normalized.forEach(function (path) {
      if (existing.has(path)) return;
      if (emptyInputs.length) {
        const input = emptyInputs.shift();
        input.value = path;
        existing.add(path);
        return;
      }
      const element = renderProductTemplate(mediaTemplate, mediaIndex++, null);
      if (!element) return;
      const input = element.querySelector('input[name^="media_urls["]');
      if (input) {
        input.value = path;
      }
      mediaList.appendChild(element);
      attachMediaPicker(element);
      existing.add(path);
    });
  }

  bindAddMediaButton(
    addMediaBtn,
    mediaTemplate,
    mediaList,
    state,
    attachMediaPicker,
  );
  bindAddMediaLibraryButton(
    addMediaLibraryBtn,
    collectMediaValues,
    addProductMediaItems,
  );
  bindAddBlockButton(addBlockBtn, blockTemplate, blockList, state);
  bindAddVariantButton(
    addVariantBtn,
    variantTemplate,
    variantList,
    state,
    attachMediaPicker,
  );
  bindAddAttributeButton(
    addAttributeBtn,
    attributeTemplate,
    attributeList,
    state,
  );

  bindProductMediaListInteractions(mediaList);
  bindRemoveItemClick(blockList, "block", ".product-block-item");
  bindRemoveItemClick(variantList, "variant", ".product-variant-item");
  bindRemoveItemClick(attributeList, "attribute", ".product-attribute-item");
  initVariantUnlimitedToggles();

  const productUnlimited = document.querySelector(
    'input[name="stock_unlimited"]',
  );
  const productStockInput = document.getElementById("stock");
  initProductStockToggle(productUnlimited, productStockInput);

  const productActionSelect = document.querySelector("[data-product-action]");
  const productDownloadField = document.getElementById(
    "product-download-field",
  );
  const productVersionFields = document.getElementById(
    "product-version-fields",
  );
  const productLinkLabelField = document.getElementById(
    "product-link-label-field",
  );
  initProductActionSelectToggle(
    productActionSelect,
    productDownloadField,
    productVersionFields,
    productLinkLabelField,
  );

  document.querySelectorAll(".product-media-item").forEach(function (item) {
    attachMediaPicker(item);
  });

  document.querySelectorAll(".product-variant-item").forEach(function (item) {
    attachMediaPicker(item);
    attachVariantActionToggle(item);
  });
}

/**
 * Debounce function
 */
function debounce(func, wait) {
  let timeout;
  return function executedFunction(...args) {
    const later = function () {
      clearTimeout(timeout);
      func(...args);
    };
    clearTimeout(timeout);
    timeout = setTimeout(later, wait);
  };
}

/**
 * Theme Editor Functions
 */
const ThemeEditor = {
  previewFrame: null,

  init: function () {
    this.previewFrame = document.getElementById("preview-frame");
    this.initDeviceSwitcher();
    this.initSectionDrag();
    this.initSettingsChange();
  },

  initDeviceSwitcher: function () {
    const buttons = document.querySelectorAll(".device-btn");
    const frame = this.previewFrame;

    buttons.forEach(function (btn) {
      btn.addEventListener("click", function () {
        buttons.forEach(function (b) {
          b.classList.remove("active");
        });
        this.classList.add("active");

        const device = this.dataset.device;
        frame.classList.remove("tablet", "mobile");
        if (device !== "desktop") {
          frame.classList.add(device);
        }
      });
    });
  },

  initSectionDrag: function () {
    const sectionTypes = document.querySelectorAll(".section-type-item");

    sectionTypes.forEach(function (item) {
      item.draggable = true;

      item.addEventListener("dragstart", function (e) {
        e.dataTransfer.setData("section-type", this.dataset.type || "");
      });
    });
  },

  initSettingsChange: function () {
    const settingsInputs = document.querySelectorAll(
      ".theme-editor-settings-body input, .theme-editor-settings-body select",
    );

    settingsInputs.forEach((input) => {
      input.addEventListener(
        "change",
        debounce(() => {
          this.updatePreview();
        }, 300),
      );
    });
  },

  updatePreview: function () {
    this.previewFrame?.contentWindow?.location.reload();
  },

  addSection: function (type, pageId) {
    fetch("/admin/api/sections.php", {
      method: "POST",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify({ action: "add", type: type, page_id: pageId }),
    })
      .then(function (response) {
        return response.json();
      })
      .then(function (data) {
        if (data.success) {
          location.reload();
        } else {
          showNotification(data.error || "Failed to add section", "error");
        }
      });
  },

  removeSection: function (sectionId) {
    if (!confirm("Remove this section?")) return;

    fetch("/admin/api/sections.php", {
      method: "POST",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify({ action: "remove", section_id: sectionId }),
    })
      .then(function (response) {
        return response.json();
      })
      .then(function (data) {
        if (data.success) {
          location.reload();
        } else {
          showNotification(data.error || "Failed to remove section", "error");
        }
      });
  },

  saveOrder: function (pageId, order) {
    fetch("/admin/api/sections.php", {
      method: "POST",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify({
        action: "reorder",
        page_id: pageId,
        order: order,
      }),
    })
      .then(function (response) {
        return response.json();
      })
      .then(function (data) {
        if (data.success) {
          showNotification("Order saved", "success");
          ThemeEditor.updatePreview();
        }
      });
  },
};

// Initialize theme editor if on that page
if (document.querySelector(".theme-editor-wrapper")) {
  ThemeEditor.init();
}
