<?php

/**
 * API Endpoint for Section Management
 *
 * Routed through the admin front controller (admin/index.php).
 */

if (!defined('ROOT_PATH')) {
    http_response_code(403);
    exit;
}

use NewSite\Auth\AdminAuth;
use NewSite\Database\DatabaseManager;
use NewSite\Database\DbHelper;
use NewSite\Http\Response;

header('Content-Type: application/json');

/** @var string Reusable error message for invalid section IDs */
const SECTIONS_ERR_INVALID_ID = 'Invalid section ID';

// Check admin authentication
if (!AdminAuth::isLoggedIn()) {
    Response::json(['success' => false, 'error' => 'Unauthorized'], 401);
}

$db = DatabaseManager::getWriteConnection();
$input = json_decode(file_get_contents('php://input'), true);
$action = $input['action'] ?? $_POST['action'] ?? '';

switch ($action) {
    case 'add':
        $type = $input['type'] ?? '';
        $pageId = (int)($input['page_id'] ?? 0);

        if (empty($type) || !$pageId) {
            Response::json(['success' => false, 'error' => 'Invalid parameters']);
        }

        // Default settings based on type
        $defaultSettings = [];
        if ($type === 'split_slideshow') {
            $defaultSettings = [
                'slides' => [
                    [
                        'title' => 'Welcome',
                        'description' => 'Your description here',
                        'button_text' => 'Learn More',
                        'button_url' => '#',
                        'image' => '/assets/images/placeholder.jpg'
                    ]
                ],
                'autoplay' => true,
                'show_arrows' => true,
                'show_dots' => true
            ];
        }

        // Get max sort order
        $stmt = $db->prepare("SELECT MAX(sort_order) FROM sections WHERE page_id = ?");
        $stmt->execute([$pageId]);
        $maxOrder = (int)$stmt->fetchColumn();

        $stmt = $db->prepare("INSERT INTO sections (page_id, section_type, settings, sort_order) VALUES (?, ?, ?, ?)");
        $stmt->execute([$pageId, $type, json_encode($defaultSettings), $maxOrder + 1]);

        Response::json(['success' => true, 'id' => DbHelper::lastInsertId($db, 'sections')]);
        break;

    case 'remove':
        $sectionId = (int)($input['section_id'] ?? 0);

        if (!$sectionId) {
            Response::json(['success' => false, 'error' => SECTIONS_ERR_INVALID_ID]);
        }

        $stmt = $db->prepare("DELETE FROM sections WHERE id = ?");
        $stmt->execute([$sectionId]);

        Response::json(['success' => true]);
        break;

    case 'reorder':
        $pageId = (int)($input['page_id'] ?? 0);
        $order = $input['order'] ?? [];

        if (!$pageId || empty($order)) {
            Response::json(['success' => false, 'error' => 'Invalid parameters']);
        }

        foreach ($order as $index => $sectionId) {
            $stmt = $db->prepare("UPDATE sections SET sort_order = ? WHERE id = ? AND page_id = ?");
            $stmt->execute([$index, $sectionId, $pageId]);
        }

        Response::json(['success' => true]);
        break;

    case 'update':
        $sectionId = (int)($input['section_id'] ?? 0);
        $settings = $input['settings'] ?? [];

        if (!$sectionId) {
            Response::json(['success' => false, 'error' => SECTIONS_ERR_INVALID_ID]);
        }

        $stmt = $db->prepare("UPDATE sections SET settings = ? WHERE id = ?");
        $stmt->execute([json_encode($settings), $sectionId]);

        Response::json(['success' => true]);
        break;

    case 'get':
        $sectionId = (int)($input['section_id'] ?? $_GET['section_id'] ?? 0);

        if (!$sectionId) {
            Response::json(['success' => false, 'error' => SECTIONS_ERR_INVALID_ID]);
        }

        $stmt = $db->prepare("SELECT * FROM sections WHERE id = ?");
        $stmt->execute([$sectionId]);
        $section = $stmt->fetch();

        if ($section) {
            $section['settings'] = json_decode($section['settings'], true);
            Response::json(['success' => true, 'section' => $section]);
        } else {
            Response::json(['success' => false, 'error' => 'Section not found']);
        }
        break;

    default:
        Response::json(['success' => false, 'error' => 'Invalid action']);
}
