<?php

/**
 * Core initialization file
 * Sets up database, session, and global functions
 * COMPREHENSIVE SITE-WIDE LOGGING ENABLED
*/

$isHttpsRequest = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off')
    || (!empty($_SERVER['HTTP_X_FORWARDED_PROTO']) && $_SERVER['HTTP_X_FORWARDED_PROTO'] === 'https')
    || (!empty($_SERVER['SERVER_PORT']) && (int)$_SERVER['SERVER_PORT'] === 443);

$bootstrapConfig = [];
$bootstrapConfigDir = dirname(__DIR__) . '/data';
$bootstrapJsonPath = $bootstrapConfigDir . '/config.json';
if (is_file($bootstrapJsonPath)) {
    $bootstrapRaw = file_get_contents($bootstrapJsonPath);
    if ($bootstrapRaw !== false) {
        $decoded = json_decode($bootstrapRaw, true);
        if (is_array($decoded)) {
            $bootstrapConfig = $decoded;
        }
    }
}

$forceHttps = (string)($bootstrapConfig['FORCE_HTTPS'] ?? '0') === '1';
if ($forceHttps && php_sapi_name() !== 'cli' && !$isHttpsRequest && !headers_sent()) {
    $host = (string)($_SERVER['HTTP_HOST'] ?? '');
    $uri = (string)($_SERVER['REQUEST_URI'] ?? '/');
    if ($host !== '') {
        header('Location: https://' . $host . $uri, true, 301);
        exit;
    }
}

ini_set('expose_php', '0');

ini_set('session.use_strict_mode', '1');
ini_set('session.use_only_cookies', '1');
ini_set('session.use_trans_sid', '0');
ini_set('session.cookie_httponly', '1');
ini_set('session.cookie_samesite', 'Lax');
ini_set('session.cookie_secure', $isHttpsRequest ? '1' : '0');
session_name('CDSESSID');

$cookieParams = session_get_cookie_params();
session_set_cookie_params([
    'lifetime' => 0,
    'path' => $cookieParams['path'] ?? '/',
    'domain' => $cookieParams['domain'] ?? '',
    'secure' => $isHttpsRequest,
    'httponly' => true,
    'samesite' => 'Lax'
]);

session_start();

if (!defined('CSRF_ERROR_MESSAGE')) {
    define('CSRF_ERROR_MESSAGE', 'Invalid CSRF token.');
}
if (!defined('DB_DATETIME_FORMAT')) {
    define('DB_DATETIME_FORMAT', 'Y-m-d H:i:s');
}
if (!defined('SQL_UPDATE_SITE_USER_LAST_ACTIVITY')) {
    define('SQL_UPDATE_SITE_USER_LAST_ACTIVITY', 'UPDATE site_users SET last_activity = ? WHERE id = ?');
}
if (!defined('SQL_COUNT_MENUS_BY_LOCATION')) {
    define('SQL_COUNT_MENUS_BY_LOCATION', 'SELECT COUNT(*) FROM menus WHERE location = ?');
}
if (!defined('SQL_INSERT_MENU_NAME_LOCATION')) {
    define('SQL_INSERT_MENU_NAME_LOCATION', 'INSERT INTO menus (name, location) VALUES (?, ?)');
}
if (!defined('HTTP_URL_PATTERN')) {
    define('HTTP_URL_PATTERN', '~^https?://~i');
}
if (!defined('UPLOAD_ERROR_NO_FILE')) {
    define('UPLOAD_ERROR_NO_FILE', 'No file uploaded');
}
if (!defined('MIME_IMAGE_JPEG')) {
    define('MIME_IMAGE_JPEG', 'image/jpeg');
}
if (!defined('CHAT_IMAGE_ROUTE_PREFIX')) {
    define('CHAT_IMAGE_ROUTE_PREFIX', '/chat-image/');
}
if (!defined('PROFILE_PHOTO_ROUTE_PREFIX')) {
    define('PROFILE_PHOTO_ROUTE_PREFIX', '/profile-photo/');
}
if (!defined('SMTP_DOUBLE_EOL')) {
    define('SMTP_DOUBLE_EOL', "\r\n\r\n");
}
if (!defined('HTTP_CLIENT_USER_AGENT')) {
    define('HTTP_CLIENT_USER_AGENT', 'NewSite/1.0');
}

if (empty($GLOBALS['CSP_NONCE'])) {
    $GLOBALS['CSP_NONCE'] = bin2hex(random_bytes(16));
}

function getCspNonce(): string
{
    return $GLOBALS['CSP_NONCE'] ?? '';
}

if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

function getCsrfToken(): string
{
    return $_SESSION['csrf_token'] ?? '';
}

function rotateSessionForPrivilegeChange(): void
{
    if (session_status() === PHP_SESSION_ACTIVE) {
        @session_regenerate_id(true);
    }
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

function getCsrfTokenFromRequest(): string
{
    if (!empty($_SERVER['HTTP_X_CSRF_TOKEN'])) {
        return (string)$_SERVER['HTTP_X_CSRF_TOKEN'];
    }
    return (string)($_POST['csrf_token'] ?? '');
}

function verifyCSRF(): void
{
    $token = getCsrfTokenFromRequest();
    $sessionToken = getCsrfToken();
    if ($token === '' || $sessionToken === '' || !hash_equals($sessionToken, $token)) {
        http_response_code(403);
        die(CSRF_ERROR_MESSAGE);
    }
}

function enforceCsrfForPost(): void
{
    if (php_sapi_name() === 'cli') {
        return;
    }
    $method = $_SERVER['REQUEST_METHOD'] ?? 'GET';
    if (strtoupper($method) !== 'POST') {
        return;
    }

    $requestUri = $_SERVER['REQUEST_URI'] ?? '';
    if (strpos($requestUri, '/stripe-webhook') !== false
        || strpos($requestUri, '/templates/api/stripe-webhook.php') !== false
        || strpos($requestUri, 'route=stripe-webhook') !== false) {
        return;
    }

    $token = getCsrfTokenFromRequest();
    $sessionToken = getCsrfToken();
    if ($token === '' || $sessionToken === '' || !hash_equals($sessionToken, $token)) {
        $isJson = (isset($_SERVER['HTTP_ACCEPT']) && strpos($_SERVER['HTTP_ACCEPT'], 'application/json') !== false)
            || (isset($_SERVER['CONTENT_TYPE']) && strpos($_SERVER['CONTENT_TYPE'], 'application/json') !== false)
            || (isset($_SERVER['REQUEST_URI']) && strpos($_SERVER['REQUEST_URI'], '/templates/api/') !== false);

        http_response_code(403);
        if ($isJson) {
            header('Content-Type: application/json');
            echo json_encode(['success' => false, 'error' => CSRF_ERROR_MESSAGE]);
        } else {
            echo CSRF_ERROR_MESSAGE;
        }
        exit;
    }
}

enforceCsrfForPost();

if (php_sapi_name() !== 'cli' && !headers_sent()) {
    $nonce = getCspNonce();
    $requestUri = $_SERVER['REQUEST_URI'] ?? '';
    $isAdminRequest = strpos($requestUri, '/admin') === 0;
    $styleSrc = "style-src 'self'";
    $scriptSrc = "script-src 'nonce-{$nonce}' 'strict-dynamic'";
    $frameSrc = "frame-src 'self' https://www.youtube-nocookie.com https://www.youtube.com";
    /* blob: needed in img-src for File System Access API thumbnails (easy-media) */
    $csp = "default-src 'self'; {$scriptSrc}; {$styleSrc}; {$frameSrc}; img-src 'self' data: blob: https:; media-src 'self' blob:; font-src 'self' data:; connect-src 'self' wss: ws: https:; object-src 'none'; base-uri 'self'; form-action 'self'; frame-ancestors 'self'; upgrade-insecure-requests";
    header_remove('X-Powered-By');
    header('Content-Security-Policy: ' . $csp);
    header('X-Content-Type-Options: nosniff');
    header('X-Frame-Options: SAMEORIGIN');
    header('Referrer-Policy: strict-origin-when-cross-origin');
    header('Permissions-Policy: geolocation=(), microphone=(), camera=(), payment=(), usb=(), bluetooth=(), magnetometer=(), gyroscope=(), accelerometer=()');
    header('X-XSS-Protection: 1; mode=block');
    if ($isHttpsRequest) {
        header('Strict-Transport-Security: max-age=31536000; includeSubDomains; preload');
    }

    $method = $_SERVER['REQUEST_METHOD'] ?? 'GET';
    $requestUri = $_SERVER['REQUEST_URI'] ?? '';
    $isAuthPage = (bool)preg_match('~^/(login|account)\b~', $requestUri)
        || (bool)preg_match('~^/admin/login(\.php)?\b~', $requestUri);
    $isSensitive = ($method !== 'GET' && $method !== 'HEAD')
        || !empty($_SESSION['site_user_id'])
        || !empty($_SESSION['admin_user_id'])
        || $isAuthPage;
    if ($isSensitive && !headers_sent()) {
        header('Cache-Control: private, no-store, no-cache, must-revalidate, max-age=0');
        header('Pragma: no-cache');
        header('Expires: 0');
    }
}

// Error reporting - capture ALL errors
error_reporting(E_ALL);
$debugMode = (string)($bootstrapConfig['APP_DEBUG'] ?? '');
if ($debugMode === '') {
    $debugMode = (string)getenv('APP_DEBUG');
}
$debugEnabled = $debugMode === '1';
ini_set('display_errors', $debugEnabled ? '1' : '0');
ini_set('log_errors', 1);

// ===========================================
// GLOBAL ERROR/WARNING/NOTICE LOGGING
// This runs for EVERY file on the entire site
// ===========================================

// Track which file initiated the request
define('INIT_SCRIPT', $_SERVER['SCRIPT_FILENAME'] ?? 'unknown');
define('INIT_TIME', microtime(true));

// Global error types mapping
$GLOBALS['_ERROR_TYPES'] = [
    E_ERROR => 'FATAL',
    E_WARNING => 'WARNING',
    E_PARSE => 'PARSE',
    E_NOTICE => 'NOTICE',
    E_CORE_ERROR => 'CORE_ERROR',
    E_CORE_WARNING => 'CORE_WARNING',
    E_COMPILE_ERROR => 'COMPILE_ERROR',
    E_COMPILE_WARNING => 'COMPILE_WARNING',
    E_USER_ERROR => 'USER_ERROR',
    E_USER_WARNING => 'USER_WARNING',
    E_USER_NOTICE => 'USER_NOTICE',
    E_RECOVERABLE_ERROR => 'RECOVERABLE_ERROR',
    E_DEPRECATED => 'DEPRECATED',
    E_USER_DEPRECATED => 'USER_DEPRECATED',
    E_ALL => 'ALL'
];

if (PHP_VERSION_ID < 80400 && defined('E_STRICT')) {
    $strictErrorType = (int)constant('E_STRICT');
    $GLOBALS['_ERROR_TYPES'][$strictErrorType] = 'STRICT';
}

// High-frequency endpoints (log sampling)
$GLOBALS['_HIGH_FREQ_ENDPOINTS'] = [
    'cart-api',
    'notifications-api',
    'messages-api',
    'messenger-api',
    'friends-list',
    'friends-search',
    'friends-action',
    'log-error'
];

// Endpoints to completely skip logging (only errors will be logged)
$GLOBALS['_SKIP_LOG_ENDPOINTS'] = [
    'log-error',  // Don't log the logging endpoint itself
];

function uriContainsAny(string $uri, array $needles): bool
{
    foreach ($needles as $needle) {
        $needle = (string)$needle;
        if ($needle !== '' && strpos($uri, $needle) !== false) {
            return true;
        }
    }
    return false;
}

function getRequestLogType(string $uri): string
{
    if (strpos(INIT_SCRIPT, '/admin/') !== false) {
        return 'admin_request';
    }
    if (strpos(INIT_SCRIPT, '/api/') !== false || strpos($uri, '-api') !== false) {
        return 'api_request';
    }
    return 'request';
}

function isSensitivePostField(string $key, array $sensitiveFields): bool
{
    return in_array($key, $sensitiveFields, true)
        || str_contains($key, 'password')
        || str_contains($key, 'token')
        || str_contains($key, 'secret')
        || str_contains($key, 'auth')
        || str_contains($key, 'cookie')
        || str_contains($key, 'session')
        || str_contains($key, 'email')
        || str_contains($key, 'phone')
        || str_contains($key, 'address');
}

function sanitizePostDataForLogging(array $postData): array
{
    $sensitiveFields = [
        'password', 'password_confirm', 'current_password', 'new_password',
        'card_number', 'cvv', 'cvc', 'token', 'secret', 'api_key',
        'email', 'phone', 'address', 'name', 'first_name', 'last_name',
        'session', 'session_id', 'cookie', 'auth', 'authorization', 'oauth',
        'csrf', 'csrf_token', 'ssn', 'iban', 'bic'
    ];

    foreach ($postData as $key => $_value) {
        $lowerKey = strtolower((string)$key);
        if (isSensitivePostField($lowerKey, $sensitiveFields)) {
            $postData[$key] = '[REDACTED]';
        }
    }

    return $postData;
}

function formatUploadedFilesForLogging(array $files): array
{
    $uploads = [];
    foreach ($files as $key => $file) {
        $name = $file['name'] ?? '';
        if (is_array($name)) {
            foreach ($name as $i => $singleName) {
                $uploads[] = "$key: $singleName (" . ($file['size'][$i] ?? 0) . ' bytes)';
            }
            continue;
        }

        $uploads[] = "$key: {$file['name']} ({$file['size']} bytes)";
    }

    return $uploads;
}

function logCurrentRequest()
{
    if (empty($GLOBALS['_LOG_REQUEST'])) {
        return;
    }

    $GLOBALS['_LOG_REQUEST'] = false;

    $method = $_SERVER['REQUEST_METHOD'] ?? 'GET';
    $uri = $_SERVER['REQUEST_URI'] ?? '/';
    $script = basename(INIT_SCRIPT);
    $referer = $_SERVER['HTTP_REFERER'] ?? 'Direct';

    if (uriContainsAny($uri, $GLOBALS['_SKIP_LOG_ENDPOINTS'])) {
        return;
    }

    $isHighFreq = uriContainsAny($uri, $GLOBALS['_HIGH_FREQ_ENDPOINTS']);
    if ($isHighFreq && mt_rand(1, 100) > 1) {
        return;
    }

    $message = "$method $uri";
    if ($isHighFreq) {
        $message .= ' [SAMPLED 1%]';
    }

    \NewSite\Logging\LogService::add(getRequestLogType($uri), $message, "Script: $script\nReferer: $referer");

    if ($method === 'POST' && !empty($_POST)) {
        $postData = sanitizePostDataForLogging($_POST);
        \NewSite\Logging\LogService::add('post_data', "POST to $uri", json_encode($postData, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    }

    if (!empty($_FILES)) {
        $uploads = formatUploadedFilesForLogging($_FILES);
        \NewSite\Logging\LogService::add('file_upload', "Files uploaded to $uri", implode("\n", $uploads));
    }
}

// ===========================================
// END GLOBAL ERROR LOGGING
// ===========================================

// Define paths
define('ROOT_PATH', dirname(__DIR__));
define('DATA_PATH', ROOT_PATH . '/data');
define('INCLUDES_PATH', ROOT_PATH . '/includes');
define('PUBLIC_PATH', ROOT_PATH . '/public');

// ===========================================
// COMPOSER PSR-4 AUTOLOADER
// ===========================================

// Classes under NewSite\ namespace are loaded on demand via Composer.
require_once ROOT_PATH . '/vendor/autoload.php';
use NewSite\Config\ConfigLoader;
use NewSite\Config\SetupService;
use NewSite\Database\DbHelper;
use NewSite\Http\Response;

// Seed ConfigLoader with the bootstrap-loaded config so all subsequent
// reads go through the PSR-4 class instead of re-including config.php.
ConfigLoader::init($bootstrapConfig);

// =====================================================================
// GLOBAL HELPER — kept for template ergonomics (used in ~56 template files)
// =====================================================================

/**
 * Escape a string for safe HTML output (XSS prevention).
 *
 * Security: delegates to Response::escape() which uses htmlspecialchars()
 * with ENT_QUOTES | ENT_SUBSTITUTE and UTF-8 encoding.
 */
function e(?string $string): string
{
    return Response::escape($string);
}

// =====================================================================
// INITIALIZE
// =====================================================================

// Only redirect to setup if this is a web request (not CLI)
if (php_sapi_name() !== 'cli') {
    SetupService::redirectToSetupIfNeeded();
}

// Initialize database on first load
DbHelper::initDatabase();

// Log current request (now that LogService is available)
logCurrentRequest();
