<?php
/**
 * First-Time Setup Wizard
 *
 * INSTALLATION:
 * 1. This file ships in /data/setup.php (not publicly accessible)
 * 2. When you need to run setup, move it to /public/setup.php via FTP/SFTP
 * 3. Visit https://yourdomain.com/setup.php
 * 4. After successful setup, this file automatically moves back to /data/
 *
 * SECURITY: This file is protected by multiple layers:
 * - Ships in non-public folder
 * - Auto-moves back after completion
 * - Returns 404 if setup is already complete
 * - Rate limiting on connection attempts
 * - Bot/scanner detection
 * - All access attempts are logged
 * - CSRF token protection
 */

// ============================================================================
// SECURITY CHECKS - Run before anything else
// ============================================================================

const SETUP_SCRIPT_PATH = '/setup.php';
const SETUP_PUBLIC_SCRIPT_PATH = '/public/setup.php';
const SETUP_DATETIME_FORMAT = 'Y-m-d H:i:s';
const SETUP_LOCATION_HEADER_PREFIX = 'Location: ';

final class SetupConfigWriteException extends RuntimeException
{
}

// Detect current location
$currentFile = __FILE__;
$isInPublic = strpos($currentFile, DIRECTORY_SEPARATOR . 'public' . DIRECTORY_SEPARATOR) !== false;
$publicPath = dirname($currentFile) . SETUP_SCRIPT_PATH;
$dataPath = dirname(dirname($currentFile)) . '/data';
$safeStoragePath = $dataPath . '/setup.php';

// If we're being included from data folder, the paths are different
if (!$isInPublic) {
    $dataPath = dirname($currentFile);
    $publicPath = dirname(dirname($currentFile)) . SETUP_PUBLIC_SCRIPT_PATH;
    $safeStoragePath = $currentFile;
}

$configPath = $dataPath . '/config.json';
$setupLockPath = $dataPath . '/.setup_complete';
$setupLogPath = $dataPath . '/setup_access.log';
$rateLimitPath = $dataPath . '/setup_rate_limit.json';

// Ensure data directory exists
if (!is_dir($dataPath)) {
    @mkdir($dataPath, 0755, true);
}

// Helper to get client IP (defined early for logging)
function getSetupClientIP(): string
{
    $headers = ['HTTP_CF_CONNECTING_IP', 'HTTP_X_FORWARDED_FOR', 'HTTP_X_REAL_IP', 'REMOTE_ADDR'];
    foreach ($headers as $header) {
        if (!empty($_SERVER[$header])) {
            $ip = $_SERVER[$header];
            if (strpos($ip, ',') !== false) {
                $ip = trim(explode(',', $ip)[0]);
            }
            if (filter_var($ip, FILTER_VALIDATE_IP)) {
                return $ip;
            }
        }
    }
    return '127.0.0.1';
}

// Log all access attempts to setup page
function logSetupAccess(string $action, string $details = ''): void
{
    global $setupLogPath;
    $ip = getSetupClientIP();
    $ua = $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown';
    $time = date(SETUP_DATETIME_FORMAT);
    $log = "[{$time}] IP: {$ip} | Action: {$action} | UA: " . substr($ua, 0, 100) . " | {$details}\n";
    @file_put_contents($setupLogPath, $log, FILE_APPEND | LOCK_EX);
}

// Move setup file back to data folder (safe storage)
function moveSetupToSafe(): bool
{
    global $currentFile, $safeStoragePath, $isInPublic;

    $isMoved = true;
    if ($isInPublic && realpath($currentFile) !== realpath($safeStoragePath)) {
        $isMoved = @copy($currentFile, $safeStoragePath);
        if ($isMoved) {
            // Then delete from public
            @unlink($currentFile);
        }
    }

    return $isMoved;
}

// Check rate limiting (max 10 attempts per hour per IP)
function isRateLimited(): bool
{
    global $rateLimitPath;
    $ip = getSetupClientIP();
    $now = time();
    $hourAgo = $now - 3600;

    $limits = [];
    if (file_exists($rateLimitPath)) {
        $limits = json_decode(@file_get_contents($rateLimitPath), true) ?: [];
    }

    // Clean old entries
    foreach ($limits as $k => $v) {
        if ($v['time'] < $hourAgo) {
            unset($limits[$k]);
        }
    }

    // Count attempts from this IP
    $attempts = 0;
    foreach ($limits as $entry) {
        if ($entry['ip'] === $ip && $entry['time'] > $hourAgo) {
            $attempts++;
        }
    }

    // Add this attempt
    $limits[] = ['ip' => $ip, 'time' => $now];
    @file_put_contents($rateLimitPath, json_encode($limits), LOCK_EX);

    return $attempts >= 10;
}

// Detect common bots and scanners
function isBot(): bool
{
    $ua = strtolower($_SERVER['HTTP_USER_AGENT'] ?? '');
    $botPatterns = [
        'bot', 'crawler', 'spider', 'scan', 'curl', 'wget', 'python',
        'java', 'perl', 'ruby', 'go-http', 'httpie', 'postman', 'insomnia',
        'nuclei', 'nikto', 'sqlmap', 'nmap', 'masscan', 'zgrab', 'gobuster',
        'dirbuster', 'wfuzz', 'ffuf', 'burp', 'zap', 'acunetix', 'nessus',
        'openvas', 'qualys', 'w3af', 'arachni', 'skipfish', 'whatweb'
    ];

    foreach ($botPatterns as $pattern) {
        if (strpos($ua, $pattern) !== false) {
            return true;
        }
    }

    // No user agent is suspicious
    if (empty($ua) || strlen($ua) < 10) {
        return true;
    }

    return false;
}

// SECURITY: If setup is complete, show 404 (don't reveal file exists)
if (file_exists($setupLockPath)) {
    logSetupAccess('BLOCKED', 'Setup already complete - showing 404');
    http_response_code(404);
    echo '<!DOCTYPE html><html><head><title>404 Not Found</title></head><body><h1>404 Not Found</h1><p>The requested URL was not found on this server.</p></body></html>';
    exit;
}

// SECURITY: Rate limiting
if (isRateLimited()) {
    logSetupAccess('RATE_LIMITED', 'Too many attempts');
    http_response_code(429);
    echo '<!DOCTYPE html><html><head><title>Too Many Requests</title></head><body><h1>429 Too Many Requests</h1><p>Please try again later.</p></body></html>';
    exit;
}

// SECURITY: Block bots (but log for monitoring)
if (isBot()) {
    logSetupAccess('BOT_BLOCKED', 'Bot/scanner detected');
    http_response_code(404);
    echo '<!DOCTYPE html><html><head><title>404 Not Found</title></head><body><h1>404 Not Found</h1><p>The requested URL was not found on this server.</p></body></html>';
    exit;
}

// Log legitimate access
logSetupAccess('ACCESS', 'Setup page accessed');

// ============================================================================
// SETUP WIZARD BEGINS
// ============================================================================

session_start();
if ($_SERVER['REQUEST_METHOD'] !== 'POST' && !isset($_SESSION['setup_token'])) {
    $_SESSION['setup_token'] = bin2hex(random_bytes(32));
}

function getSetupScriptUrl(string $query = ''): string
{
    $path = parse_url($_SERVER['REQUEST_URI'] ?? SETUP_SCRIPT_PATH, PHP_URL_PATH);
    if (!is_string($path) || $path === '') {
        $path = SETUP_SCRIPT_PATH;
    }

    $url = $path;
    if ($query !== '') {
        $url .= '?' . $query;
    }

    return $url;
}

function getSetupSiteRootUrl(string $query = ''): string
{
    $path = parse_url($_SERVER['REQUEST_URI'] ?? SETUP_SCRIPT_PATH, PHP_URL_PATH);
    if (!is_string($path) || $path === '') {
        $path = SETUP_SCRIPT_PATH;
    }

    $dir = dirname($path);
    $dir = ($dir === '.' || $dir === DIRECTORY_SEPARATOR) ? '/' : rtrim(str_replace('\\', '/', $dir), '/') . '/';
    if ($query !== '') {
        return $dir . '?' . $query;
    }

    return $dir;
}

$step = (int)($_GET['step'] ?? 1);
$error = '';
$success = '';

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Verify CSRF token
    if (!isset($_SESSION['setup_token'], $_POST['setup_token']) || !hash_equals((string)$_SESSION['setup_token'], (string)$_POST['setup_token'])) {
        logSetupAccess('CSRF_FAIL', 'Invalid CSRF token');
        $error = 'Invalid security token. Please refresh and try again.';
    } else {
        $action = $_POST['action'] ?? '';

        if ($action === 'test_connection') {
            // Step 1: Test PostgreSQL connection
            $host = trim($_POST['db_host'] ?? '');
            $port = trim($_POST['db_port'] ?? '5432');
            $name = trim($_POST['db_name'] ?? '');
            $user = trim($_POST['db_user'] ?? '');
            $pass = trim($_POST['db_pass'] ?? '');
            $sslmode = trim($_POST['db_sslmode'] ?? 'require');

            if (empty($host) || empty($name) || empty($user) || empty($pass)) {
                $error = 'All database fields are required.';
            } else {
                try {
                    $dsn = "pgsql:host={$host};port={$port};dbname={$name};sslmode={$sslmode}";
                    $testDb = new PDO($dsn, $user, $pass);
                    $testDb->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
                    $testDb->query('SELECT 1');

                    // Connection successful - save to session for next step
                    $_SESSION['setup_db'] = [
                        'host' => $host,
                        'port' => $port,
                        'name' => $name,
                        'user' => $user,
                        'pass' => $pass,
                        'sslmode' => $sslmode
                    ];

                    logSetupAccess('DB_SUCCESS', 'Database connection successful');
                    header(SETUP_LOCATION_HEADER_PREFIX . getSetupScriptUrl('step=2'));
                    exit;
                } catch (PDOException $e) {
                    logSetupAccess('DB_FAIL', 'Connection failed: ' . $e->getMessage());
                    $error = 'Database connection failed: ' . $e->getMessage();
                }
            }
        } elseif ($action === 'create_admin') {
            // Step 2: Create admin user and finalize setup
            $username = trim($_POST['admin_username'] ?? '');
            $password = trim($_POST['admin_password'] ?? '');
            $passwordConfirm = trim($_POST['admin_password_confirm'] ?? '');
            $addIp = isset($_POST['add_current_ip']);
            $ipLabel = trim($_POST['ip_label'] ?? 'Setup');

            if (empty($username) || strlen($username) < 3) {
                $error = 'Username must be at least 3 characters.';
            } elseif (empty($password) || strlen($password) < 8) {
                $error = 'Password must be at least 8 characters.';
            } elseif ($password !== $passwordConfirm) {
                $error = 'Passwords do not match.';
            } elseif (!isset($_SESSION['setup_db'])) {
                $error = 'Database configuration lost. Please start over.';
                header(SETUP_LOCATION_HEADER_PREFIX . getSetupScriptUrl('step=1'));
                exit;
            } else {
                try {
                    $dbConfig = $_SESSION['setup_db'];

                    // Create config.json (JSON format — no PHP include required)
                    $configArray = [
                        'DB_HOST'    => $dbConfig['host'],
                        'DB_PORT'    => $dbConfig['port'],
                        'DB_NAME'    => $dbConfig['name'],
                        'DB_USER'    => $dbConfig['user'],
                        'DB_PASS'    => $dbConfig['pass'],
                        'DB_SSLMODE' => $dbConfig['sslmode'],
                    ];
                    $configContent = json_encode(
                        $configArray,
                        JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES
                    );
                    if ($configContent === false) {
                        throw new SetupConfigWriteException('Failed to encode config as JSON.');
                    }

                    // Ensure data directory exists
                    $dataDir = dirname($configPath);
                    if (!is_dir($dataDir)) {
                        mkdir($dataDir, 0755, true);
                    }

                    // Write config file
                    if (file_put_contents($configPath, $configContent) === false) {
                        throw new SetupConfigWriteException('Failed to write config.json. Check directory permissions.');
                    }

                    // Now load the full application via Composer autoloader
                    require_once __DIR__ . '/../vendor/autoload.php';

                    // Initialize database schema
                    \NewSite\Database\DbHelper::initDatabase();

                    $db = \NewSite\Database\DatabaseManager::getWriteConnection();

                    // Check if target username already exists.
                    // Special case: setup may have bootstrapped default admin/admin123,
                    // so allow claiming that account by setting a new password.
                    $stmt = $db->prepare("SELECT id FROM users WHERE username = ?");
                    $stmt->execute([$username]);
                    $existingUser = $stmt->fetch();

                    if ($existingUser && $username !== 'admin') {
                        $error = 'An admin with that username already exists.';
                    } else {
                        $now = date(SETUP_DATETIME_FORMAT);

                        if ($existingUser && $username === 'admin') {
                            $hash = password_hash($password, PASSWORD_ARGON2ID);
                            $stmt = $db->prepare("UPDATE users SET password = ?, role = 'admin', created_at = ? WHERE id = ?");
                            $stmt->execute([$hash, $now, $existingUser['id']]);
                        } else {
                            // Create admin user — Argon2id is the project standard
                            $hash = password_hash($password, PASSWORD_ARGON2ID);
                            $stmt = $db->prepare("INSERT INTO users (username, password, role, created_at) VALUES (?, ?, 'admin', ?)");
                            $stmt->execute([$username, $hash, $now]);
                        }

                        // Delete default admin if it exists and is different
                        if ($username !== 'admin') {
                            $db->exec("DELETE FROM users WHERE username = 'admin'");
                        }

                        // Add current IP to allowlist
                        if ($addIp) {
                            $currentIp = getSetupClientIP();
                            $stmt = $db->prepare("INSERT INTO admin_allowed_ips (ip_address, label, added_at) VALUES (?, ?, ?) ON CONFLICT (ip_address) DO UPDATE SET label = EXCLUDED.label, added_at = EXCLUDED.added_at");
                            $stmt->execute([$currentIp, $ipLabel, $now]);
                        }

                        // Create setup lock file
                        file_put_contents($setupLockPath, date(SETUP_DATETIME_FORMAT) . "\nSetup completed by: " . getSetupClientIP());

                        // Log successful setup completion
                        logSetupAccess('SETUP_COMPLETE', 'Admin created: ' . $username);

                        // Clear session
                        unset($_SESSION['setup_db']);
                        unset($_SESSION['setup_token']);

                        // Move setup file back to data folder for security
                        $moved = moveSetupToSafe();
                        logSetupAccess('FILE_MOVED', $moved ? 'Setup file moved to data folder' : 'Could not move file - please delete manually');

                        // Store success message in session for display after redirect
                        $_SESSION['setup_success'] = true;
                        $_SESSION['setup_moved'] = $moved;

                        // Redirect to home page
                        header(SETUP_LOCATION_HEADER_PREFIX . getSetupSiteRootUrl('setup=complete'));
                        exit;
                    }
                } catch (Exception $e) {
                    $error = 'Setup failed: ' . $e->getMessage();
                    // Clean up config if it was created
                    if (file_exists($configPath) && !file_exists($setupLockPath)) {
                        @unlink($configPath);
                    }
                }
            }
        }
    }
}

$currentIp = getSetupClientIP();
$stepOneClass = getStepIndicatorClass($step, 1);
$stepTwoClass = getStepIndicatorClass($step, 2);
$stepThreeClass = getStepIndicatorClass($step, 3);

function getStepIndicatorClass(int $currentStep, int $indicatorStep): string
{
    if ($currentStep === $indicatorStep) {
        return 'active';
    }

    if ($currentStep > $indicatorStep) {
        return 'completed';
    }

    return '';
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Site Setup Wizard</title>
    <style>
        :root {
            --primary: #a78bff;
            --secondary: #6f5bff;
            --accent: #c9b8ff;
            --bg: #0f0d16;
            --bg-secondary: #151325;
            --bg-tertiary: #1a162a;
            --text: #f4f2ff;
            --text-muted: #a9a3c7;
            --border: rgba(167, 139, 255, 0.2);
            --success: #00ff88;
            --error: #ff4466;
            --radius: 12px;
        }

        * { margin: 0; padding: 0; box-sizing: border-box; }

        body {
            font-family: 'Inter', system-ui, -apple-system, sans-serif;
            background: radial-gradient(900px 500px at 20% -10%, rgba(167, 139, 255, 0.07), transparent 60%),
                        radial-gradient(900px 500px at 90% 10%, rgba(111, 91, 255, 0.08), transparent 55%),
                        linear-gradient(180deg, #0f0d16 0%, #121024 100%);
            background-attachment: fixed;
            color: var(--text);
            line-height: 1.6;
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }

        .setup-container {
            max-width: 600px;
            width: 100%;
        }

        .setup-header {
            text-align: center;
            margin-bottom: 30px;
        }

        .setup-header h1 {
            font-size: 2rem;
            margin-bottom: 10px;
            background: linear-gradient(135deg, var(--primary), var(--accent));
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
        }

        .setup-header p {
            color: var(--text-muted);
        }

        .steps {
            display: flex;
            justify-content: center;
            gap: 20px;
            margin-bottom: 30px;
        }

        .step-indicator {
            display: flex;
            align-items: center;
            gap: 8px;
            color: var(--text-muted);
        }

        .step-indicator.active {
            color: var(--primary);
        }

        .step-indicator.completed {
            color: var(--success);
        }

        .step-number {
            width: 32px;
            height: 32px;
            border-radius: 50%;
            background: var(--bg-tertiary);
            border: 2px solid var(--border);
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 600;
            font-size: 0.9rem;
        }

        .step-indicator.active .step-number {
            background: var(--primary);
            border-color: var(--primary);
            color: var(--bg);
        }

        .step-indicator.completed .step-number {
            background: var(--success);
            border-color: var(--success);
            color: var(--bg);
        }

        .card {
            background: var(--bg-tertiary);
            border: 1px solid var(--border);
            border-radius: var(--radius);
            overflow: hidden;
        }

        .card-header {
            padding: 20px 25px;
            border-bottom: 1px solid var(--border);
            display: flex;
            align-items: center;
            gap: 12px;
        }

        .card-header h2 {
            font-size: 1.25rem;
        }

        .card-header .icon {
            font-size: 1.5rem;
        }

        .card-body {
            padding: 25px;
        }

        .form-group {
            margin-bottom: 20px;
        }

        .form-group:last-child {
            margin-bottom: 0;
        }

        .form-group label {
            display: block;
            margin-bottom: 8px;
            font-weight: 500;
        }

        .form-group .hint {
            display: block;
            font-size: 0.85rem;
            color: var(--text-muted);
            margin-top: 5px;
        }

        .form-control {
            width: 100%;
            padding: 12px 15px;
            background: var(--bg-secondary);
            border: 1px solid var(--border);
            border-radius: 8px;
            color: var(--text);
            font-size: 1rem;
            transition: all 0.3s ease;
        }

        .form-control:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(167, 139, 255, 0.2);
        }

        .form-control::placeholder {
            color: var(--text-muted);
        }

        .form-row {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 15px;
        }

        .checkbox-group {
            display: flex;
            align-items: flex-start;
            gap: 10px;
        }

        .checkbox-group input[type="checkbox"] {
            width: 20px;
            height: 20px;
            margin-top: 2px;
            accent-color: var(--primary);
        }

        .checkbox-group label {
            margin-bottom: 0;
        }

        .btn {
            display: inline-block;
            padding: 12px 24px;
            border: none;
            border-radius: 8px;
            font-size: 1rem;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            text-decoration: none;
        }

        .btn-primary {
            background: linear-gradient(135deg, var(--primary), var(--secondary));
            color: white;
        }

        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 20px rgba(167, 139, 255, 0.4);
        }

        .btn-block {
            width: 100%;
            display: block;
            text-align: center;
        }

        .alert {
            padding: 15px 20px;
            border-radius: 8px;
            margin-bottom: 20px;
        }

        .alert-error {
            background: rgba(255, 68, 102, 0.15);
            border: 1px solid rgba(255, 68, 102, 0.3);
            color: #ff6b8a;
        }

        .alert-success {
            background: rgba(0, 255, 136, 0.15);
            border: 1px solid rgba(0, 255, 136, 0.3);
            color: #00ff88;
        }

        .info-box {
            background: rgba(167, 139, 255, 0.1);
            border: 1px solid rgba(167, 139, 255, 0.2);
            border-radius: 8px;
            padding: 15px;
            margin-bottom: 20px;
            font-size: 0.9rem;
        }

        .info-box strong {
            color: var(--primary);
        }

        .success-icon {
            font-size: 4rem;
            display: block;
            text-align: center;
            margin-bottom: 20px;
        }

        .success-message {
            text-align: center;
        }

        .success-message h2 {
            margin-bottom: 15px;
            color: var(--success);
        }

        .success-message p {
            color: var(--text-muted);
            margin-bottom: 25px;
        }

        code {
            background: var(--bg-secondary);
            padding: 2px 6px;
            border-radius: 4px;
            font-family: 'Monaco', 'Consolas', monospace;
            font-size: 0.9em;
        }

        .tooltip-icon {
            display: inline-flex;
            align-items: center;
            justify-content: center;
            width: 18px;
            height: 18px;
            background: var(--border);
            border-radius: 50%;
            font-size: 0.75rem;
            cursor: help;
            margin-left: 5px;
            position: relative;
        }

        .tooltip-icon:hover::after {
            content: attr(data-tooltip);
            position: absolute;
            bottom: 100%;
            left: 50%;
            transform: translateX(-50%);
            background: var(--bg);
            border: 1px solid var(--border);
            padding: 8px 12px;
            border-radius: 6px;
            font-size: 0.8rem;
            white-space: nowrap;
            max-width: 300px;
            white-space: normal;
            z-index: 100;
            margin-bottom: 5px;
        }

        @media (max-width: 600px) {
            .form-row {
                grid-template-columns: 1fr;
            }

            .steps {
                flex-direction: column;
                align-items: center;
            }
        }
    </style>
</head>
<body>
    <div class="setup-container">
        <div class="setup-header">
            <h1>🚀 Site Setup Wizard</h1>
            <p>Configure your website in a few simple steps</p>
        </div>

        <div class="steps">
            <div class="step-indicator <?php echo $stepOneClass; ?>">
                <span class="step-number"><?php echo $step > 1 ? '✓' : '1'; ?></span>
                <span>Database</span>
            </div>
            <div class="step-indicator <?php echo $stepTwoClass; ?>">
                <span class="step-number"><?php echo $step > 2 ? '✓' : '2'; ?></span>
                <span>Admin</span>
            </div>
            <div class="step-indicator <?php echo $stepThreeClass; ?>">
                <span class="step-number"><?php echo $step >= 3 ? '✓' : '3'; ?></span>
                <span>Complete</span>
            </div>
        </div>

        <?php if ($error): ?>
        <div class="alert alert-error"><?php echo htmlspecialchars($error); ?></div>
        <?php endif; ?>

        <?php if ($step === 1): ?>
        <!-- Step 1: Database Configuration -->
        <div class="card">
            <div class="card-header">
                <span class="icon">🗄️</span>
                <h2>PostgreSQL Database</h2>
            </div>
            <div class="card-body">
                <div class="info-box">
                    <strong>ℹ️ PostgreSQL Required</strong><br>
                    This site uses PostgreSQL for enhanced security and performance.
                    You'll need a PostgreSQL database from your hosting provider.
                </div>

                <form method="POST">
                    <input type="hidden" name="setup_token" value="<?php echo htmlspecialchars($_SESSION['setup_token']); ?>">
                    <input type="hidden" name="action" value="test_connection">

                    <div class="form-group">
                        <label for="db_host">
                            Database Host
                            <span class="tooltip-icon" data-tooltip="The hostname or IP address of your PostgreSQL server. Usually provided by your hosting.">?</span>
                        </label>
                        <input type="text" id="db_host" name="db_host" class="form-control" placeholder="e.g., db.example.com or 192.168.1.1" value="<?php echo htmlspecialchars($_POST['db_host'] ?? ''); ?>" required>
                        <span class="hint">The server address where your PostgreSQL database is hosted</span>
                    </div>

                    <div class="form-row">
                        <div class="form-group">
                            <label for="db_port">
                                Port
                                <span class="tooltip-icon" data-tooltip="PostgreSQL default port is 5432. Some hosts use custom ports.">?</span>
                            </label>
                            <input type="text" id="db_port" name="db_port" class="form-control" placeholder="5432" value="<?php echo htmlspecialchars($_POST['db_port'] ?? '5432'); ?>">
                        </div>
                        <div class="form-group">
                            <label for="db_name">
                                Database Name
                                <span class="tooltip-icon" data-tooltip="The name of your PostgreSQL database. Create one in your hosting panel first.">?</span>
                            </label>
                            <input type="text" id="db_name" name="db_name" class="form-control" placeholder="mysite_db" value="<?php echo htmlspecialchars($_POST['db_name'] ?? ''); ?>" required>
                        </div>
                    </div>

                    <div class="form-row">
                        <div class="form-group">
                            <label for="db_user">
                                Username
                                <span class="tooltip-icon" data-tooltip="The PostgreSQL user with access to the database.">?</span>
                            </label>
                            <input type="text" id="db_user" name="db_user" class="form-control" placeholder="db_user" value="<?php echo htmlspecialchars($_POST['db_user'] ?? ''); ?>" required>
                        </div>
                        <div class="form-group">
                            <label for="db_pass">
                                Password
                                <span class="tooltip-icon" data-tooltip="The password for your database user.">?</span>
                            </label>
                            <input type="password" id="db_pass" name="db_pass" class="form-control" placeholder="••••••••" required>
                        </div>
                    </div>

                    <div class="form-group">
                        <label for="db_sslmode">
                            SSL Mode
                            <span class="tooltip-icon" data-tooltip="'require' enforces encrypted connection (recommended). Use 'disable' only for local development.">?</span>
                        </label>
                        <select id="db_sslmode" name="db_sslmode" class="form-control">
                            <option value="require" <?php echo ($_POST['db_sslmode'] ?? 'require') === 'require' ? 'selected' : ''; ?>>require (Recommended - encrypted connection)</option>
                            <option value="prefer" <?php echo ($_POST['db_sslmode'] ?? '') === 'prefer' ? 'selected' : ''; ?>>prefer (Use SSL if available)</option>
                            <option value="disable" <?php echo ($_POST['db_sslmode'] ?? '') === 'disable' ? 'selected' : ''; ?>>disable (No encryption - local only)</option>
                        </select>
                        <span class="hint">Always use 'require' for production servers</span>
                    </div>

                    <button type="submit" class="btn btn-primary btn-block">Test Connection & Continue →</button>
                </form>
            </div>
        </div>

        <?php elseif ($step === 2): ?>
        <!-- Step 2: Admin Account -->
        <div class="card">
            <div class="card-header">
                <span class="icon">👤</span>
                <h2>Create Admin Account</h2>
            </div>
            <div class="card-body">
                <div class="info-box">
                    <strong>✅ Database Connected!</strong><br>
                    Now create your administrator account to manage the site.
                </div>

                <form method="POST">
                    <input type="hidden" name="setup_token" value="<?php echo htmlspecialchars($_SESSION['setup_token']); ?>">
                    <input type="hidden" name="action" value="create_admin">

                    <div class="form-group">
                        <label for="admin_username">
                            Admin Username
                            <span class="tooltip-icon" data-tooltip="Choose a unique username. Avoid 'admin' for better security.">?</span>
                        </label>
                        <input type="text" id="admin_username" name="admin_username" class="form-control" placeholder="e.g., myadmin" minlength="3" value="<?php echo htmlspecialchars($_POST['admin_username'] ?? ''); ?>" required>
                        <span class="hint">Minimum 3 characters. This is what you'll use to log in.</span>
                    </div>

                    <div class="form-group">
                        <label for="admin_password">
                            Password
                            <span class="tooltip-icon" data-tooltip="Use a strong password with letters, numbers, and symbols.">?</span>
                        </label>
                        <input type="password" id="admin_password" name="admin_password" class="form-control" placeholder="••••••••" minlength="8" required>
                        <span class="hint">Minimum 8 characters. Use a strong, unique password.</span>
                    </div>

                    <div class="form-group">
                        <label for="admin_password_confirm">Confirm Password</label>
                        <input type="password" id="admin_password_confirm" name="admin_password_confirm" class="form-control" placeholder="••••••••" minlength="8" required>
                    </div>

                    <div class="form-group">
                        <div class="checkbox-group">
                            <input type="checkbox" name="add_current_ip" id="add_current_ip" checked>
                            <label for="add_current_ip">
                                <strong>Add my current IP to admin allowlist</strong><br>
                                <span class="hint" style="margin-top: 5px;">Your IP: <code><?php echo htmlspecialchars($currentIp); ?></code><br>
                                Only allowed IPs can access the admin panel. You can add more IPs later using the allowlist tool.</span>
                            </label>
                        </div>
                    </div>

                    <div class="form-group">
                        <label for="ip_label">IP Label (optional)</label>
                        <input type="text" id="ip_label" name="ip_label" class="form-control" placeholder="e.g., Home, Office" value="Setup">
                        <span class="hint">A friendly name to remember this IP address</span>
                    </div>

                    <button type="submit" class="btn btn-primary btn-block">Create Admin & Finish Setup →</button>
                </form>
            </div>
        </div>
        <?php endif; ?>
    </div>
</body>
</html>
