<?php
/**
 * Admin Allowlist Tool
 * Upload this file via FTP when needed, use it, then delete it.
 * Set a strong token below before use.
 */

// Bootstrap Composer autoloader (includes _init.php via files autoload)
if (!file_exists(__DIR__ . '/../vendor/autoload.php')) {
    http_response_code(500);
    echo 'Autoloader not found. Ensure Composer dependencies are installed.';
    exit;
}
require_once __DIR__ . '/../vendor/autoload.php';

$tokenFile = __DIR__ . '/allowlist_tool.token';
if (!is_file($tokenFile)) {
    $generatedToken = bin2hex(random_bytes(24));
    file_put_contents($tokenFile, $generatedToken, LOCK_EX);
    @chmod($tokenFile, 0600);
}
$token = trim((string)file_get_contents($tokenFile));
if ($token === '') {
    http_response_code(500);
    echo 'Allowlist tool token is missing.';
    exit;
}
$tokenHash = hash('sha256', $token);
$provided = $_POST['token'] ?? '';
$authorized = false;

if (!empty($_SESSION['allowlist_tool_auth']) && hash_equals($tokenHash, (string)$_SESSION['allowlist_tool_auth'])) {
    $authorized = true;
} elseif ($provided !== '' && hash_equals($token, $provided)) {
    $_SESSION['allowlist_tool_auth'] = $tokenHash;
    $authorized = true;
}

if (!$authorized) {
    http_response_code(200);
    echo '<!DOCTYPE html>';
    echo '<html lang="en">';
    echo '<head>';
    echo '<meta charset="UTF-8">';
    echo '<meta name="viewport" content="width=device-width, initial-scale=1.0">';
    echo '<title>Admin Allowlist Tool</title>';
    echo '<link rel="stylesheet" href="/admin/assets/css/admin.css">';
    echo '</head>';
    echo '<body class="admin-body">';
    echo '<main class="admin-main">';
    echo '<div class="admin-content">';
    echo '<div class="content-header">';
    echo '<h1>Admin Allowlist Tool</h1>';
    echo '<p>Enter the access token to continue.</p>';
    echo '</div>';
    echo '<div class="card">';
    echo '<div class="card-body">';
    echo '<form method="POST">';
    echo '<input type="hidden" name="csrf_token" value="' . e(getCsrfToken()) . '">';
    echo '<div class="form-group">';
    echo '<label for="allowlist-tool-token">Access Token</label>';
    echo '<input id="allowlist-tool-token" type="password" name="token" class="form-control" required>';
    echo '</div>';
    echo '<button type="submit" class="btn btn-primary">Unlock</button>';
    echo '</form>';
    echo '</div>';
    echo '</div>';
    echo '</div>';
    echo '</main>';
    echo '</body>';
    echo '</html>';
    exit;
}

$message = '';
$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    $ip = trim($_POST['ip'] ?? '');
    $label = trim($_POST['label'] ?? '');
    $db = \NewSite\Database\DatabaseManager::getWriteConnection();

    if ($action === 'add') {
        if (\NewSite\Auth\AdminAuth::addAllowedIp($ip, $label)) {
            $message = 'IP added to allowlist.';
        } else {
            $error = 'Failed to add IP. Make sure it is valid.';
        }
    } elseif ($action === 'remove') {
        if (\NewSite\Auth\AdminAuth::removeAllowedIp($ip)) {
            $message = 'IP removed from allowlist.';
        } else {
            $error = 'Failed to remove IP. Make sure it is valid.';
        }
    } elseif ($action === 'update_ip') {
        $oldIp = trim($_POST['old_ip'] ?? '');
        $newIp = trim($_POST['new_ip'] ?? '');
        $newLabel = trim($_POST['label'] ?? '');
        if ($oldIp === '' || $newIp === '') {
            $error = 'Both current and new IP are required.';
        } else {
            if ($oldIp !== $newIp) {
                \NewSite\Auth\AdminAuth::removeAllowedIp($oldIp);
            }
            if (\NewSite\Auth\AdminAuth::addAllowedIp($newIp, $newLabel)) {
                $message = 'Allowed IP updated.';
            } else {
                $error = 'Failed to update IP. Make sure it is valid.';
            }
        }
    } elseif ($action === 'add_admin') {
        $username = trim($_POST['username'] ?? '');
        $password = trim($_POST['password'] ?? '');
        if ($username === '' || $password === '') {
            $error = 'Username and password are required.';
        } elseif (strlen($username) < 3) {
            $error = 'Username must be at least 3 characters.';
        } elseif (strlen($password) < 6) {
            $error = 'Password must be at least 6 characters.';
        } else {
            $stmt = $db->prepare("SELECT id FROM users WHERE username = ? LIMIT 1");
            $stmt->execute([$username]);
            if ($stmt->fetch()) {
                $error = 'That username is already in use.';
            } else {
                $hash = password_hash($password, PASSWORD_ARGON2ID);
                $now = \NewSite\Database\DbHelper::nowString();
                $stmt = $db->prepare("INSERT INTO users (username, password, role, created_at) VALUES (?, ?, 'admin', ?)");
                $stmt->execute([$username, $hash, $now]);
                $message = 'Admin user added.';
            }
        }
    } elseif ($action === 'update_admin') {
        $adminId = (int)($_POST['admin_id'] ?? 0);
        $username = trim($_POST['username'] ?? '');
        $password = trim($_POST['password'] ?? '');
        if ($adminId <= 0) {
            $error = 'Invalid admin user.';
        } elseif ($username === '') {
            $error = 'Username is required.';
        } elseif (strlen($username) < 3) {
            $error = 'Username must be at least 3 characters.';
        } else {
            $stmt = $db->prepare("SELECT id FROM users WHERE username = ? AND id != ? LIMIT 1");
            $stmt->execute([$username, $adminId]);
            if ($stmt->fetch()) {
                $error = 'That username is already in use.';
            } else {
                $stmt = $db->prepare("UPDATE users SET username = ? WHERE id = ?");
                $stmt->execute([$username, $adminId]);
                if ($password !== '') {
                    if (strlen($password) < 6) {
                        $error = 'Password must be at least 6 characters.';
                    } else {
                        $hash = password_hash($password, PASSWORD_ARGON2ID);
                        $stmt = $db->prepare("UPDATE users SET password = ? WHERE id = ?");
                        $stmt->execute([$hash, $adminId]);
                    }
                }
                if ($error === '') {
                    $message = 'Admin user updated.';
                }
            }
        }
    } elseif ($action === 'delete_admin') {
        $adminId = (int)($_POST['admin_id'] ?? 0);
        $confirm = trim($_POST['confirm_delete'] ?? '');
        $currentAdminId = (int)($_SESSION['admin_user_id'] ?? 0);
        if ($adminId <= 0) {
            $error = 'Invalid admin user.';
        } elseif ($confirm !== 'DELETE') {
            $error = 'Type DELETE to confirm admin deletion.';
        } elseif ($currentAdminId > 0 && $adminId === $currentAdminId) {
            $error = 'Cannot delete the currently logged-in admin.';
        } else {
            $stmt = $db->query("SELECT COUNT(*) FROM users WHERE role = 'admin'");
            $adminCount = (int)$stmt->fetchColumn();
            if ($adminCount <= 1) {
                $error = 'Cannot delete the last admin user.';
            } else {
                $stmt = $db->prepare("DELETE FROM users WHERE id = ? AND role = 'admin'");
                $stmt->execute([$adminId]);
                if ($stmt->rowCount() > 0) {
                    $message = 'Admin user removed.';
                } else {
                    $error = 'Admin user not found.';
                }
            }
        }
    }
}

$allowed = \NewSite\Auth\AdminAuth::getAllowedIps();
$db = \NewSite\Database\DatabaseManager::getWriteConnection();
$dbName = $db->query('SELECT current_database()')->fetchColumn() ?: 'unknown';
$adminUsers = $db->query("SELECT id, username, role, created_at FROM users WHERE role = 'admin' ORDER BY id ASC")->fetchAll();
$currentIp = \NewSite\Security\IpService::getClientIP();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Allowlist Tool</title>
    <link rel="stylesheet" href="/admin/assets/css/admin.css">
</head>
<body class="admin-body">
    <main class="admin-main">
        <div class="admin-content">
            <div class="content-header">
                <h1>Admin Allowlist Tool</h1>
                <p>Open: <strong>https://yourdomain.com/allowlist_tool.php</strong> (enter token)</p>
                <p>Database: <strong>PostgreSQL</strong> (<?php echo e((string)$dbName); ?>)</p>
                <p>Manage allowed admin IPs. Delete this file after use.</p>
            </div>

            <?php if ($message): ?>
            <div class="alert alert-success"><?php echo e($message); ?></div>
            <?php endif; ?>

            <?php if ($error): ?>
            <div class="alert alert-error"><?php echo e($error); ?></div>
            <?php endif; ?>

            <div class="card">
                <div class="card-header">
                    <h3>Add IP</h3>
                </div>
                <div class="card-body">
                    <form method="POST">
                        <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                        <input type="hidden" name="action" value="add">
                        <div class="form-group">
                            <label for="allowlist-add-ip">IP Address</label>
                            <input id="allowlist-add-ip" type="text" name="ip" class="form-control" value="<?php echo e($currentIp); ?>" required>
                        </div>
                        <div class="form-group">
                            <label for="allowlist-add-label">Label (optional)</label>
                            <input id="allowlist-add-label" type="text" name="label" class="form-control" placeholder="Home / Office">
                        </div>
                        <button type="submit" class="btn btn-primary">Add IP</button>
                    </form>
                </div>
            </div>

            <div class="card">
                <div class="card-header">
                    <h3>Remove IP</h3>
                </div>
                <div class="card-body">
                    <form method="POST">
                        <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                        <input type="hidden" name="action" value="remove">
                        <div class="form-group">
                            <label for="allowlist-remove-ip">IP Address</label>
                            <input id="allowlist-remove-ip" type="text" name="ip" class="form-control" required>
                        </div>
                        <button type="submit" class="btn btn-danger">Remove IP</button>
                    </form>
                </div>
            </div>

            <div class="card">
                <div class="card-header">
                    <h3>Allowed IPs</h3>
                </div>
                <div class="card-body">
                    <?php if (empty($allowed)): ?>
                    <p class="text-muted">No allowed IPs yet.</p>
                    <?php else: ?>
                    <div class="table-responsive">
                        <table class="admin-table">
                            <thead>
                                <tr>
                                    <th>IP</th>
                                    <th>Label</th>
                                    <th>Added</th>
                                    <th>Update</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($allowed as $row): ?>
                                <tr>
                                    <td><code><?php echo e($row['ip_address']); ?></code></td>
                                    <td><?php echo e($row['label'] ?? ''); ?></td>
                                    <td><?php echo e($row['added_at']); ?></td>
                                    <td>
                                        <form method="POST" class="form-inline">
                                            <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                                            <input type="hidden" name="action" value="update_ip">
                                            <input type="hidden" name="old_ip" value="<?php echo e($row['ip_address']); ?>">
                                            <div class="form-group">
                                                <label class="sr-only" for="allowlist-new-ip-<?php echo (int)$row['id']; ?>">New IP</label>
                                                <input id="allowlist-new-ip-<?php echo (int)$row['id']; ?>" type="text" name="new_ip" class="form-control" value="<?php echo e($row['ip_address']); ?>" required>
                                            </div>
                                            <div class="form-group">
                                                <label class="sr-only" for="allowlist-label-<?php echo (int)$row['id']; ?>">Label</label>
                                                <input id="allowlist-label-<?php echo (int)$row['id']; ?>" type="text" name="label" class="form-control" value="<?php echo e($row['label'] ?? ''); ?>" placeholder="Label">
                                            </div>
                                            <button type="submit" class="btn btn-secondary btn-sm">Save</button>
                                        </form>
                                        <form method="POST" class="form-inline">
                                            <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                                            <input type="hidden" name="action" value="remove">
                                            <input type="hidden" name="ip" value="<?php echo e($row['ip_address']); ?>">
                                            <button type="submit" class="btn btn-danger btn-sm">Remove</button>
                                        </form>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                    <?php endif; ?>
                </div>
            </div>

            <div class="card">
                <div class="card-header">
                    <h3>Add Admin User</h3>
                </div>
                <div class="card-body">
                    <form method="POST">
                        <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                        <input type="hidden" name="action" value="add_admin">
                        <div class="form-row">
                            <div class="form-group">
                                <label for="allowlist-admin-username">Username</label>
                                <input id="allowlist-admin-username" type="text" name="username" class="form-control" required>
                            </div>
                            <div class="form-group">
                                <label for="allowlist-admin-password">Password</label>
                                <input id="allowlist-admin-password" type="password" name="password" class="form-control" required>
                            </div>
                        </div>
                        <button type="submit" class="btn btn-primary">Add Admin</button>
                    </form>
                </div>
            </div>

            <div class="card">
                <div class="card-header">
                    <h3>Admin Users</h3>
                </div>
                <div class="card-body">
                    <?php if (empty($adminUsers)): ?>
                        <p class="text-muted">No admin users found.</p>
                    <?php else: ?>
                        <div class="table-responsive">
                            <table class="admin-table">
                                <thead>
                                    <tr>
                                        <th>Username</th>
                                        <th>Role</th>
                                        <th>Created</th>
                                        <th>Update</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($adminUsers as $admin): ?>
                                        <tr>
                                            <td><?php echo e($admin['username']); ?></td>
                                            <td><?php echo e($admin['role']); ?></td>
                                            <td><?php echo e($admin['created_at']); ?></td>
                                            <td>
                                                <div class="form-inline">
                                                    <form method="POST" class="form-inline">
                                                        <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                                                        <input type="hidden" name="action" value="update_admin">
                                                        <input type="hidden" name="admin_id" value="<?php echo (int)$admin['id']; ?>">
                                                        <div class="form-group">
                                                            <label class="sr-only" for="allowlist-admin-edit-username-<?php echo (int)$admin['id']; ?>">Username</label>
                                                            <input id="allowlist-admin-edit-username-<?php echo (int)$admin['id']; ?>" type="text" name="username" class="form-control" value="<?php echo e($admin['username']); ?>" required>
                                                        </div>
                                                        <div class="form-group">
                                                            <label class="sr-only" for="allowlist-admin-edit-password-<?php echo (int)$admin['id']; ?>">New Password</label>
                                                            <input id="allowlist-admin-edit-password-<?php echo (int)$admin['id']; ?>" type="password" name="password" class="form-control" placeholder="New password">
                                                        </div>
                                                        <button type="submit" class="btn btn-secondary btn-sm">Save</button>
                                                    </form>
                                                    <form method="POST" class="form-inline">
                                                        <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                                                        <input type="hidden" name="action" value="delete_admin">
                                                        <input type="hidden" name="admin_id" value="<?php echo (int)$admin['id']; ?>">
                                                        <div class="form-group">
                                                            <label class="sr-only" for="allowlist-admin-delete-confirm-<?php echo (int)$admin['id']; ?>">Confirm</label>
                                                            <input id="allowlist-admin-delete-confirm-<?php echo (int)$admin['id']; ?>" type="text" name="confirm_delete" class="form-control" placeholder="Type DELETE">
                                                        </div>
                                                        <button type="submit" class="btn btn-danger btn-sm">Delete</button>
                                                    </form>
                                                </div>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </main>
</body>
</html>
