$ErrorActionPreference = "Stop"

$scriptDir = Split-Path -Parent $MyInvocation.MyCommand.Path
$appRoot = Resolve-Path (Join-Path $scriptDir "..")
$venvPath = Join-Path $appRoot ".venv"
$activate = Join-Path $venvPath "Scripts\Activate.ps1"

if (-not (Test-Path $activate)) {
    Write-Host "Virtualenv not found. Run setup.ps1 first." -ForegroundColor Red
    exit 1
}

. $activate

$certDir = Join-Path $appRoot "cert"
$certPem = Join-Path $certDir "cert.pem"
$keyPem = Join-Path $certDir "key.pem"

$settingsPath = Join-Path $appRoot "run_settings.json"

function Load-RunSettings {
    if (Test-Path $settingsPath) {
        try {
            $loaded = Get-Content $settingsPath -Raw | ConvertFrom-Json
            if ($loaded -is [hashtable]) { return $loaded }
            if ($loaded -is [pscustomobject]) {
                $ht = @{}
                $loaded.PSObject.Properties | ForEach-Object { $ht[$_.Name] = $_.Value }
                return $ht
            }
            return @{}
        } catch { }
    }
    return @{ }
}

function Save-RunSettings($settings) {
    try {
        $settings | ConvertTo-Json | Set-Content -Path $settingsPath -Encoding UTF8
    } catch { }
}

function Apply-Defaults {
    $settings = Load-RunSettings

    if (-not $env:FINETUNE_ENABLED) { $env:FINETUNE_ENABLED = $settings.FINETUNE_ENABLED }
    if (-not $env:FINETUNE_STEPS) { $env:FINETUNE_STEPS = $settings.FINETUNE_STEPS }
    if (-not $env:FINETUNE_LR) { $env:FINETUNE_LR = $settings.FINETUNE_LR }
    if (-not $env:FREEZE_VISUAL) { $env:FREEZE_VISUAL = $settings.FREEZE_VISUAL }
    if (-not $env:BATCH_SIZE) { $env:BATCH_SIZE = $settings.BATCH_SIZE }
    if (-not $env:USE_BLIP2) { $env:USE_BLIP2 = $settings.USE_BLIP2 }
    if (-not $env:BLIP2_MODEL_ID) { $env:BLIP2_MODEL_ID = $settings.BLIP2_MODEL_ID }
    if (-not $env:HF_TOKEN) { $env:HF_TOKEN = $settings.HF_TOKEN }
    if (-not $env:REQUIRE_CUDA) { $env:REQUIRE_CUDA = $settings.REQUIRE_CUDA }
    if (-not $env:OPENCLIP_MODEL) { $env:OPENCLIP_MODEL = $settings.OPENCLIP_MODEL }
    if (-not $env:OPENCLIP_PRETRAINED) { $env:OPENCLIP_PRETRAINED = $settings.OPENCLIP_PRETRAINED }
    if (-not $env:USE_HTTPS) { $env:USE_HTTPS = $settings.USE_HTTPS }
    if (-not $env:TAGGING_MODE) { $env:TAGGING_MODE = $settings.TAGGING_MODE }
    if (-not $env:VLM_ENABLED) { $env:VLM_ENABLED = $settings.VLM_ENABLED }
    if (-not $env:VLM_MODEL_ID) { $env:VLM_MODEL_ID = $settings.VLM_MODEL_ID }
    if (-not $env:VLM_LORA_ENABLED) { $env:VLM_LORA_ENABLED = $settings.VLM_LORA_ENABLED }
    if (-not $env:VLM_LORA_LR) { $env:VLM_LORA_LR = $settings.VLM_LORA_LR }
    if (-not $env:VLM_LORA_RANK) { $env:VLM_LORA_RANK = $settings.VLM_LORA_RANK }
    if (-not $env:VLM_MAX_TAGS) { $env:VLM_MAX_TAGS = $settings.VLM_MAX_TAGS }

    # === Default Settings for New Users ===
    # FINETUNE_ENABLED: 1 = allow model learning, 0 = disabled
    if (-not $env:FINETUNE_ENABLED) { $env:FINETUNE_ENABLED = "1" }
    
    # FINETUNE_STEPS: Number of training iterations per batch (8 is good balance)
    if (-not $env:FINETUNE_STEPS) { $env:FINETUNE_STEPS = "8" }
    
    # FINETUNE_LR: Learning rate - MUST be small like 1e-5 (0.00001)
    # WARNING: Values like 5 or 0.1 will destroy model weights!
    if (-not $env:FINETUNE_LR) { $env:FINETUNE_LR = "1e-5" }
    
    # BATCH_SIZE: Number of images processed together (1-16, higher needs more VRAM)
    if (-not $env:BATCH_SIZE) { $env:BATCH_SIZE = "4" }

    # FREEZE_VISUAL: 1 = freeze CLIP image encoder during fine-tune, 0 = train everything
    if (-not $env:FREEZE_VISUAL) { $env:FREEZE_VISUAL = "0" }
    
    # USE_BLIP2: 0 = disabled (recommended), 1 = use BLIP2 for captions (slower, needs HF token)
    if (-not $env:USE_BLIP2) { $env:USE_BLIP2 = "0" }
    if ($null -eq $env:BLIP2_MODEL_ID) { $env:BLIP2_MODEL_ID = "" }
    if ($null -eq $env:HF_TOKEN) { $env:HF_TOKEN = "" }
    
    # REQUIRE_CUDA: 0 = works on CPU too, 1 = require GPU (faster but needs NVIDIA)
    if (-not $env:REQUIRE_CUDA) { $env:REQUIRE_CUDA = "0" }
    
    # OPENCLIP_MODEL: ViT-L-14 is recommended (larger, more accurate)
    # Options: ViT-B-32 (smaller/faster), ViT-L-14 (recommended), ViT-H-14 (largest)
    if (-not $env:OPENCLIP_MODEL) { $env:OPENCLIP_MODEL = "ViT-L-14" }
    if (-not $env:OPENCLIP_PRETRAINED) { $env:OPENCLIP_PRETRAINED = "openai" }
    
    # USE_HTTPS: 1 = required for browser security, 0 = HTTP only (not recommended)
    if (-not $env:USE_HTTPS) { $env:USE_HTTPS = "1" }

    # TAGGING_MODE: Which AI system to use for tagging
    # "clip" = CLIP only, "vlm" = Vision Language Model only, "hybrid" = both merged
    if (-not $env:TAGGING_MODE) { $env:TAGGING_MODE = "clip" }

    # VLM_ENABLED: 1 = load Vision Language Model, 0 = disabled
    if (-not $env:VLM_ENABLED) { $env:VLM_ENABLED = "0" }

    # VLM_MODEL_ID: HuggingFace model ID for the VLM
    # Options by VRAM:
    #   vikhyatk/moondream2         (~4 GB) - Fast, good for 6-8GB GPUs
    #   microsoft/Florence-2-base   (~3 GB) - Lightweight, good captions
    #   llava-hf/llava-1.5-7b-hf   (~8 GB) - High quality, needs 10GB+
    #   Qwen/Qwen2-VL-7B-Instruct  (~8 GB) - Strong multilingual, 10GB+
    if (-not $env:VLM_MODEL_ID) { $env:VLM_MODEL_ID = "vikhyatk/moondream2" }

    # VLM_LORA_ENABLED: 1 = fine-tune VLM with LoRA adapters, 0 = use as-is
    if (-not $env:VLM_LORA_ENABLED) { $env:VLM_LORA_ENABLED = "0" }
    if (-not $env:VLM_LORA_LR) { $env:VLM_LORA_LR = "1e-4" }
    if (-not $env:VLM_LORA_RANK) { $env:VLM_LORA_RANK = "8" }
    if (-not $env:VLM_MAX_TAGS) { $env:VLM_MAX_TAGS = "10" }
}

Apply-Defaults

function Show-Menu {
    Write-Host "" 
    Write-Host "AI Tag Server" -ForegroundColor Cyan
    Write-Host "1) Start server"
    Write-Host "2) Configure settings"
    Write-Host "3) Reset model (start from 0)"
    Write-Host "4) Show current settings"
    Write-Host "5) Exit"
}

function Start-Server {
    Set-Location $appRoot
    if ($env:USE_HTTPS -eq "1") {
        if (-not (Test-Path $certPem) -or -not (Test-Path $keyPem)) {
            Write-Host "HTTPS enabled but cert not found. Run generate_cert.ps1 or set USE_HTTPS=0." -ForegroundColor Red
            return
        }
        Write-Host "Starting AI Tag Server on https://localhost:8443 ..." -ForegroundColor Cyan
        uvicorn app:app --host 0.0.0.0 --port 8443 --ssl-certfile $certPem --ssl-keyfile $keyPem
    } else {
        Write-Host "Starting AI Tag Server on http://localhost:8443 ..." -ForegroundColor Cyan
        uvicorn app:app --host 0.0.0.0 --port 8443
    }
}

function Configure-Settings {
    Write-Host ""
    Write-Host "═══════════════════════════════════════════════════════════════" -ForegroundColor Cyan
    Write-Host "                    AI Tag Server Configuration                 " -ForegroundColor Cyan
    Write-Host "═══════════════════════════════════════════════════════════════" -ForegroundColor Cyan
    Write-Host "Press Enter to keep current value" -ForegroundColor DarkGray
    Write-Host ""
    
    Write-Host "┌─────────────────────────────────────────────────────────────┐" -ForegroundColor DarkCyan
    Write-Host "│  FINE-TUNING SETTINGS                                       │" -ForegroundColor DarkCyan
    Write-Host "└─────────────────────────────────────────────────────────────┘" -ForegroundColor DarkCyan
    Write-Host "  FINETUNE_ENABLED: Train model on your tags (1=on, 0=off)" -ForegroundColor Gray
    Write-Host "  FINETUNE_STEPS:   Training iterations per image (4-16, higher=better but slower)" -ForegroundColor Gray
    Write-Host "                    Recommended: 8 for quality, 4 for speed" -ForegroundColor DarkGray
    Write-Host "  FINETUNE_LR:      Learning rate - MUST be tiny like 1e-5 (0.00001)" -ForegroundColor Gray
    Write-Host "                    ⚠️  WARNING: Values like 5 or 0.1 will DESTROY weights!" -ForegroundColor Yellow
    Write-Host "  BATCH_SIZE:       Images processed together during training" -ForegroundColor Gray
    Write-Host "                    Recommended by VRAM:" -ForegroundColor DarkGray
    Write-Host "                      4GB  VRAM → BATCH_SIZE=1-2" -ForegroundColor DarkGray
    Write-Host "                      6GB  VRAM → BATCH_SIZE=2-4" -ForegroundColor DarkGray
    Write-Host "                      8GB  VRAM → BATCH_SIZE=4-8" -ForegroundColor DarkGray
    Write-Host "                      12GB+VRAM → BATCH_SIZE=8-16" -ForegroundColor DarkGray
    Write-Host "  FREEZE_VISUAL:    Freeze CLIP image encoder during fine-tune" -ForegroundColor Gray
    Write-Host "                    0 = train both (best with pretrained openai)" -ForegroundColor DarkGray
    Write-Host "                    1 = freeze image encoder (safe for small datasets)" -ForegroundColor DarkGray
    Write-Host ""
    
    Write-Host "┌─────────────────────────────────────────────────────────────┐" -ForegroundColor DarkCyan
    Write-Host "│  CLIP MODEL (Core image understanding)                      │" -ForegroundColor DarkCyan
    Write-Host "└─────────────────────────────────────────────────────────────┘" -ForegroundColor DarkCyan
    Write-Host "  Choose model based on your GPU VRAM:" -ForegroundColor DarkGray
    Write-Host "  OPENCLIP_MODEL options:" -ForegroundColor Gray
    Write-Host "    1) ViT-B-32        (~400 MB, 2-3GB VRAM) - Fast, good for 4GB GPUs" -ForegroundColor Gray
    Write-Host "    2) ViT-B-16        (~600 MB, 3-4GB VRAM) - Better quality than B-32" -ForegroundColor Gray
    Write-Host "    3) ViT-L-14        (~1.5 GB, 4-6GB VRAM) - ⭐ BEST BALANCE" -ForegroundColor Green
    Write-Host "    4) ViT-L-14-336    (~1.6 GB, 5-7GB VRAM) - Higher resolution input" -ForegroundColor Gray
    Write-Host "    5) ViT-H-14        (~2.5 GB, 8-10GB VRAM) - High quality" -ForegroundColor Gray
    Write-Host "    6) ViT-g-14        (~1.7 GB, 6-8GB VRAM) - Good alternative to H-14" -ForegroundColor Gray
    Write-Host "    7) ViT-bigG-14     (~5+ GB, 12-16GB VRAM) - Top quality, needs good GPU" -ForegroundColor Gray
    Write-Host "    8) ViT-SO400M-14   (~2+ GB, 8-10GB VRAM) - Specialized variant" -ForegroundColor Gray
    Write-Host "    9) RN50            (~100 MB, 1-2GB VRAM) - CPU-friendly, basic quality" -ForegroundColor Gray
    Write-Host "   10) RN101           (~200 MB, 2GB VRAM) - Slightly better than RN50" -ForegroundColor Gray
    Write-Host ""
    Write-Host "  OPENCLIP_PRETRAINED options:" -ForegroundColor Gray
    Write-Host "    1) openai              - ⭐ Best for most use cases" -ForegroundColor Green
    Write-Host "    2) laion2b_s32b_b82k   - Trained on 2B images, diverse content" -ForegroundColor Gray
    Write-Host "    3) laion400m_e32       - Smaller dataset, faster fine-tuning" -ForegroundColor Gray
    Write-Host "    4) scratch             - ⚠️  No pretrained weights (experts only)" -ForegroundColor Yellow
    Write-Host "                             Requires 1000s of training samples" -ForegroundColor DarkYellow
    Write-Host ""
    
    Write-Host "┌─────────────────────────────────────────────────────────────┐" -ForegroundColor DarkCyan
    Write-Host "│  BLIP-2 (Optional caption generation)                       │" -ForegroundColor DarkCyan
    Write-Host "└─────────────────────────────────────────────────────────────┘" -ForegroundColor DarkCyan
    Write-Host "  USE_BLIP2: Enable automatic captioning (1=on, 0=off)" -ForegroundColor Gray
    Write-Host "             ⚠️  Adds 2-8GB+ extra VRAM usage on top of CLIP" -ForegroundColor Yellow
    Write-Host "             Captions help generate better tag suggestions" -ForegroundColor DarkGray
    Write-Host "  BLIP2 models (if enabled):" -ForegroundColor Gray
    Write-Host "    1) blip2-flan-t5-base  (~4 GB)  - ⭐ RECOMMENDED for 8GB GPUs" -ForegroundColor Green
    Write-Host "    2) blip2-flan-t5-xl    (~8 GB)  - Better captions, needs 12GB+" -ForegroundColor Gray
    Write-Host "    3) blip2-flan-t5-xxl   (~16 GB) - Best quality, needs 24GB+" -ForegroundColor Gray
    Write-Host "    4) blip2-opt-2.7b      (~7 GB)  - Alternative, similar to xl" -ForegroundColor Gray
    Write-Host "    5) blip2-opt-2.7b-coco (~7 GB)  - Tuned for real-world photos" -ForegroundColor Gray
    Write-Host "    6) blip2-opt-6.7b      (~14 GB) - High quality, needs 16GB+" -ForegroundColor Gray
    Write-Host "    7) blip2-itm-vit-g     (~2 GB)  - ⚠️  NOT for captioning" -ForegroundColor DarkYellow
    Write-Host ""
    Write-Host "  HF_TOKEN: Required for downloading BLIP2 models" -ForegroundColor Magenta
    Write-Host "            Get one free at: https://huggingface.co/settings/tokens" -ForegroundColor DarkMagenta
    Write-Host ""
    
    Write-Host "┌─────────────────────────────────────────────────────────────┐" -ForegroundColor DarkCyan
    Write-Host "│  OTHER SETTINGS                                             │" -ForegroundColor DarkCyan
    Write-Host "└─────────────────────────────────────────────────────────────┘" -ForegroundColor DarkCyan
    Write-Host "  REQUIRE_CUDA: Require NVIDIA GPU (1) or allow CPU fallback (0)" -ForegroundColor Gray
    Write-Host "                CPU is 10-50x slower but works without NVIDIA GPU" -ForegroundColor DarkGray
    Write-Host "  USE_HTTPS:    HTTPS enabled (1) or HTTP only (0)" -ForegroundColor Gray
    Write-Host "                ⚠️  HTTPS required for browser access from websites" -ForegroundColor Yellow
    Write-Host "                Set to 0 only for local testing or if using reverse proxy" -ForegroundColor DarkGray
    Write-Host ""
    Write-Host "┌─────────────────────────────────────────────────────────────┐" -ForegroundColor Magenta
    Write-Host "│  VISION LANGUAGE MODEL (VLM) - True AI Vision               │" -ForegroundColor Magenta
    Write-Host "└─────────────────────────────────────────────────────────────┘" -ForegroundColor Magenta
    Write-Host "  VLM understands images like GPT-4V/Claude Vision. It reads" -ForegroundColor Gray
    Write-Host "  the image and picks tags from YOUR vocabulary intelligently." -ForegroundColor Gray
    Write-Host "" 
    Write-Host "  TAGGING_MODE:   Which system to use for auto-tagging" -ForegroundColor Gray
    Write-Host "    clip   = CLIP + EmbedNet + Classifier (fast, ~50ms)" -ForegroundColor DarkGray
    Write-Host "    vlm    = Vision Language Model only (smart, ~1-3s)" -ForegroundColor DarkGray
    Write-Host "    hybrid = Both systems merged (best accuracy, ~2-4s)" -ForegroundColor DarkGray
    Write-Host "" 
    Write-Host "  VLM_ENABLED:    Load VLM model (1=yes, 0=no)" -ForegroundColor Gray
    Write-Host "                  ⚠️  Uses 3-8GB+ extra VRAM depending on model" -ForegroundColor Yellow
    Write-Host "  VLM_MODEL_ID:   Choose model by VRAM:" -ForegroundColor Gray
    Write-Host "    1) vikhyatk/moondream2         (~4 GB) - ⭐ Best for 6-10GB" -ForegroundColor Green
    Write-Host "    2) microsoft/Florence-2-base   (~3 GB) - Lightweight" -ForegroundColor Gray
    Write-Host "    3) llava-hf/llava-1.5-7b-hf   (~8 GB) - High quality, 12GB+" -ForegroundColor Gray
    Write-Host "    4) Qwen/Qwen2-VL-7B-Instruct  (~8 GB) - Strong, 12GB+" -ForegroundColor Gray
    Write-Host "  VLM_MAX_TAGS:   Max tags VLM can return per image (default: 10)" -ForegroundColor Gray
    Write-Host "" 
    Write-Host "  VLM_LORA_ENABLED: Fine-tune VLM on your tagged images (1/0)" -ForegroundColor Gray
    Write-Host "    LoRA adapters add ~2GB during training, <100MB on disk" -ForegroundColor DarkGray
    Write-Host "  VLM_LORA_LR:      LoRA learning rate (default: 1e-4)" -ForegroundColor Gray
    Write-Host "  VLM_LORA_RANK:    LoRA rank (4=small/fast, 8=balanced, 16=quality)" -ForegroundColor Gray
    Write-Host ""
    Write-Host "───────────────────────────────────────────────────────────────" -ForegroundColor DarkGray

    $settings = Load-RunSettings
    $prevModel = $settings.OPENCLIP_MODEL
    $prevPretrained = $settings.OPENCLIP_PRETRAINED

    # Fine-tuning settings
    Write-Host ""
    Write-Host "[Fine-tuning]" -ForegroundColor Yellow
    $finetuneEnabled = Read-Host "FINETUNE_ENABLED (current: $($env:FINETUNE_ENABLED)) [1/0]"
    if ($finetuneEnabled -ne "") { $env:FINETUNE_ENABLED = $finetuneEnabled }

    $finetuneSteps = Read-Host "FINETUNE_STEPS (current: $($env:FINETUNE_STEPS)) [recommended: 8]"
    if ($finetuneSteps -ne "") { $env:FINETUNE_STEPS = $finetuneSteps }

    $finetuneLr = Read-Host "FINETUNE_LR (current: $($env:FINETUNE_LR)) [recommended: 1e-5]"
    if ($finetuneLr -ne "") { 
        # Validate learning rate
        if ($finetuneLr -match "^[0-9]+$" -and [int]$finetuneLr -ge 1) {
            Write-Host "⚠️  WARNING: '$finetuneLr' looks too high! Use scientific notation like 1e-5" -ForegroundColor Red
            $confirm = Read-Host "Are you sure? (Y to confirm, Enter to use 1e-5)"
            if ($confirm -ne "Y") { $finetuneLr = "1e-5" }
        }
        $env:FINETUNE_LR = $finetuneLr 
    }

    $batchSize = Read-Host "BATCH_SIZE (current: $($env:BATCH_SIZE)) [recommended: 4, range: 1-16]"
    if ($batchSize -ne "") {
        # Validate batch size
        if ($batchSize -match "^[0-9]+$") {
            $bs = [int]$batchSize
            if ($bs -lt 1) { $batchSize = "1" }
            elseif ($bs -gt 16) {
                Write-Host "⚠️  WARNING: Batch size $bs is very high, may cause out-of-memory errors" -ForegroundColor Yellow
                $confirm = Read-Host "Are you sure? (Y to confirm, Enter to use 16)"
                if ($confirm -ne "Y") { $batchSize = "16" }
            }
        }
        $env:BATCH_SIZE = $batchSize
    }

    $freezeVisual = Read-Host "FREEZE_VISUAL (current: $($env:FREEZE_VISUAL)) [0=train all, 1=freeze image encoder]"
    if ($freezeVisual -ne "") { $env:FREEZE_VISUAL = $freezeVisual }

    # CLIP model settings
    Write-Host ""
    Write-Host "[CLIP Model]" -ForegroundColor Yellow
    $openclipModelChoice = Read-Host "OPENCLIP_MODEL (current: $($env:OPENCLIP_MODEL)) [1-10, M=manual, Enter=keep]"
    switch ($openclipModelChoice) {
        "1" { $env:OPENCLIP_MODEL = "ViT-B-32" }
        "2" { $env:OPENCLIP_MODEL = "ViT-B-16" }
        "3" { $env:OPENCLIP_MODEL = "ViT-L-14" }
        "4" { $env:OPENCLIP_MODEL = "ViT-L-14-336" }
        "5" { $env:OPENCLIP_MODEL = "ViT-H-14" }
        "6" { $env:OPENCLIP_MODEL = "ViT-g-14" }
        "7" { $env:OPENCLIP_MODEL = "ViT-bigG-14" }
        "8" { $env:OPENCLIP_MODEL = "ViT-SO400M-14" }
        "9" { $env:OPENCLIP_MODEL = "RN50" }
        "10" { $env:OPENCLIP_MODEL = "RN101" }
        "M" { 
            $openclipModel = Read-Host "Enter model name"
            if ($openclipModel -ne "") {
                $knownModels = @("ViT-B-32","ViT-B-16","ViT-L-14","ViT-L-14-336","ViT-H-14","ViT-g-14","ViT-bigG-14","ViT-SO400M-14","RN50","RN101")
                if ($openclipModel -notin $knownModels) {
                    Write-Host "  Warning: '$openclipModel' is not a known model." -ForegroundColor Yellow
                    Write-Host "  Known: $($knownModels -join ', ')" -ForegroundColor DarkGray
                    $confirm = Read-Host "  Use anyway? (Y to confirm, Enter to cancel)"
                    if ($confirm -ne "Y" -and $confirm -ne "y") { $openclipModel = "" }
                }
                if ($openclipModel -ne "") { $env:OPENCLIP_MODEL = $openclipModel }
            }
        }
        default { }
    }

    $openclipPretrainedChoice = Read-Host "OPENCLIP_PRETRAINED (current: $($env:OPENCLIP_PRETRAINED)) [1-4, M=manual, Enter=keep]"
    switch ($openclipPretrainedChoice) {
        "1" { $env:OPENCLIP_PRETRAINED = "openai" }
        "2" { $env:OPENCLIP_PRETRAINED = "laion2b_s32b_b82k" }
        "3" { $env:OPENCLIP_PRETRAINED = "laion400m_e32" }
        "4" { $env:OPENCLIP_PRETRAINED = "scratch" }
        "M" { 
            $openclipPretrained = Read-Host "Enter pretrained name"
            if ($openclipPretrained -ne "") { $env:OPENCLIP_PRETRAINED = $openclipPretrained }
        }
        default { }
    }

    # BLIP-2 settings
    Write-Host ""
    Write-Host "[BLIP-2 Caption Model]" -ForegroundColor Yellow
    $useBlip2 = Read-Host "USE_BLIP2 (current: $($env:USE_BLIP2)) [1/0, recommended: 0]"
    if ($useBlip2 -ne "") { $env:USE_BLIP2 = $useBlip2 }

    if ($env:USE_BLIP2 -eq "1") {
        $modelChoice = Read-Host "BLIP2 model (current: $($env:BLIP2_MODEL_ID)) [1-7, M=manual, Enter=keep]"
        switch ($modelChoice) {
            "1" { $env:BLIP2_MODEL_ID = "Salesforce/blip2-flan-t5-base" }
            "2" { $env:BLIP2_MODEL_ID = "Salesforce/blip2-flan-t5-xl" }
            "3" { $env:BLIP2_MODEL_ID = "Salesforce/blip2-flan-t5-xxl" }
            "4" { $env:BLIP2_MODEL_ID = "Salesforce/blip2-opt-2.7b" }
            "5" { $env:BLIP2_MODEL_ID = "Salesforce/blip2-opt-2.7b-coco" }
            "6" { $env:BLIP2_MODEL_ID = "Salesforce/blip2-opt-6.7b" }
            "7" { $env:BLIP2_MODEL_ID = "Salesforce/blip2-itm-vit-g" }
            "M" {
                $blip2Model = Read-Host "Enter BLIP2 model ID"
                if ($blip2Model -ne "") { $env:BLIP2_MODEL_ID = $blip2Model }
            }
            default { }
        }

        if ($env:HF_TOKEN -eq "") {
            Write-Host "⚠️  BLIP-2 requires a HuggingFace token to download models" -ForegroundColor Yellow
            Write-Host "   Get one at: https://huggingface.co/settings/tokens" -ForegroundColor Cyan
        }
        $hfToken = Read-Host "HF_TOKEN (current: $(if($env:HF_TOKEN){'[set]'}else{'[empty]'}))"
        if ($hfToken -ne "") { $env:HF_TOKEN = $hfToken }
    }

    # Other settings
    Write-Host ""
    Write-Host "[Other]" -ForegroundColor Yellow
    $requireCuda = Read-Host "REQUIRE_CUDA (current: $($env:REQUIRE_CUDA)) [1/0]"
    if ($requireCuda -ne "") { $env:REQUIRE_CUDA = $requireCuda }

    $useHttps = Read-Host "USE_HTTPS (current: $($env:USE_HTTPS)) [1/0, recommended: 1]"
    if ($useHttps -ne "") { $env:USE_HTTPS = $useHttps }

    # VLM settings
    Write-Host ""
    Write-Host "[Vision Language Model]" -ForegroundColor Magenta
    $taggingMode = Read-Host "TAGGING_MODE (current: $($env:TAGGING_MODE)) [clip/vlm/hybrid]"
    if ($taggingMode -ne "") {
        if ($taggingMode -notin @("clip", "vlm", "hybrid")) {
            Write-Host "  Invalid mode. Must be clip, vlm, or hybrid. Keeping current." -ForegroundColor Red
        } else {
            $env:TAGGING_MODE = $taggingMode
        }
    }

    $vlmEnabled = Read-Host "VLM_ENABLED (current: $($env:VLM_ENABLED)) [1/0]"
    if ($vlmEnabled -ne "") { $env:VLM_ENABLED = $vlmEnabled }

    if ($env:VLM_ENABLED -eq "1" -or $env:TAGGING_MODE -ne "clip") {
        $vlmChoice = Read-Host "VLM_MODEL_ID (current: $($env:VLM_MODEL_ID)) [1-4, M=manual, Enter=keep]"
        switch ($vlmChoice) {
            "1" { $env:VLM_MODEL_ID = "vikhyatk/moondream2" }
            "2" { $env:VLM_MODEL_ID = "microsoft/Florence-2-base" }
            "3" { $env:VLM_MODEL_ID = "llava-hf/llava-1.5-7b-hf" }
            "4" { $env:VLM_MODEL_ID = "Qwen/Qwen2-VL-7B-Instruct" }
            "M" {
                $vlmModel = Read-Host "Enter VLM model ID (HuggingFace format)"
                if ($vlmModel -ne "") { $env:VLM_MODEL_ID = $vlmModel }
            }
            default { }
        }

        $vlmMaxTags = Read-Host "VLM_MAX_TAGS (current: $($env:VLM_MAX_TAGS)) [max tags per image, default: 10]"
        if ($vlmMaxTags -ne "") { $env:VLM_MAX_TAGS = $vlmMaxTags }

        $vlmLora = Read-Host "VLM_LORA_ENABLED (current: $($env:VLM_LORA_ENABLED)) [1/0]"
        if ($vlmLora -ne "") { $env:VLM_LORA_ENABLED = $vlmLora }

        if ($env:VLM_LORA_ENABLED -eq "1") {
            $vlmLoraLr = Read-Host "VLM_LORA_LR (current: $($env:VLM_LORA_LR)) [recommended: 1e-4]"
            if ($vlmLoraLr -ne "") { $env:VLM_LORA_LR = $vlmLoraLr }

            $vlmLoraRank = Read-Host "VLM_LORA_RANK (current: $($env:VLM_LORA_RANK)) [4/8/16, default: 8]"
            if ($vlmLoraRank -ne "") { $env:VLM_LORA_RANK = $vlmLoraRank }
        }

        if ($env:HF_TOKEN -eq "" -and $env:VLM_MODEL_ID -ne "") {
            Write-Host "⚠️  Some VLM models require a HuggingFace token" -ForegroundColor Yellow
            $hfToken = Read-Host "HF_TOKEN (current: $(if($env:HF_TOKEN){'[set]'}else{'[empty]'}))"
            if ($hfToken -ne "") { $env:HF_TOKEN = $hfToken }
        }
    }

    # Save settings
    $settings["FINETUNE_ENABLED"] = $env:FINETUNE_ENABLED
    $settings["FINETUNE_STEPS"] = $env:FINETUNE_STEPS
    $settings["FINETUNE_LR"] = $env:FINETUNE_LR
    $settings["FREEZE_VISUAL"] = $env:FREEZE_VISUAL
    $settings["BATCH_SIZE"] = $env:BATCH_SIZE
    $settings["USE_BLIP2"] = $env:USE_BLIP2
    $settings["BLIP2_MODEL_ID"] = $env:BLIP2_MODEL_ID
    $settings["HF_TOKEN"] = $env:HF_TOKEN
    $settings["REQUIRE_CUDA"] = $env:REQUIRE_CUDA
    $settings["OPENCLIP_MODEL"] = $env:OPENCLIP_MODEL
    $settings["OPENCLIP_PRETRAINED"] = $env:OPENCLIP_PRETRAINED
    $settings["USE_HTTPS"] = $env:USE_HTTPS
    $settings["TAGGING_MODE"] = $env:TAGGING_MODE
    $settings["VLM_ENABLED"] = $env:VLM_ENABLED
    $settings["VLM_MODEL_ID"] = $env:VLM_MODEL_ID
    $settings["VLM_LORA_ENABLED"] = $env:VLM_LORA_ENABLED
    $settings["VLM_LORA_LR"] = $env:VLM_LORA_LR
    $settings["VLM_LORA_RANK"] = $env:VLM_LORA_RANK
    $settings["VLM_MAX_TAGS"] = $env:VLM_MAX_TAGS
    Save-RunSettings $settings

    # Check if model/pretrained changed
    if ($prevModel -ne $env:OPENCLIP_MODEL -or $prevPretrained -ne $env:OPENCLIP_PRETRAINED) {
        Write-Host ""
        Write-Host "⚠️  CLIP model/pretrained changed" -ForegroundColor Yellow
        Write-Host "   Model: $prevModel -> $($env:OPENCLIP_MODEL)" -ForegroundColor Yellow
        Write-Host "   Pretrained: $prevPretrained -> $($env:OPENCLIP_PRETRAINED)" -ForegroundColor Yellow
        Write-Host "   Fine-tuned weights are incompatible with different models/pretrained." -ForegroundColor Yellow
        $reset = Read-Host "Reset fine-tuned weights? [Y/N]"
        if ($reset -eq "Y" -or $reset -eq "y") {
            $paths = @(
                (Join-Path $appRoot "tag_db.json"),
                (Join-Path $appRoot "tag_embeddings.pt"),
                (Join-Path $appRoot "tag_classifier.pt"),
                (Join-Path $appRoot "finetuned_clip.pt")
            )
            foreach ($p in $paths) {
                if (Test-Path $p) {
                    Remove-Item $p -Force
                    Write-Host "  Deleted: $p" -ForegroundColor DarkYellow
                }
            }
            Write-Host "  Weights reset. Model will start fresh." -ForegroundColor Green
        }
    }

    Write-Host ""
    Write-Host "✅ Settings saved to run_settings.json" -ForegroundColor Green
}

function Reset-Model {
    Write-Host ""
    Write-Host "⚠️  This will delete ALL trained model data:" -ForegroundColor Yellow
    Write-Host "   • tag_db.json (your tag vocabulary)" -ForegroundColor Gray
    Write-Host "   • tag_embeddings.pt (tag embedding network)" -ForegroundColor Gray
    Write-Host "   • tag_classifier.pt (classifier network)" -ForegroundColor Gray
    Write-Host "   • finetuned_clip.pt (fine-tuned CLIP weights)" -ForegroundColor Gray
    Write-Host "   • vlm_lora_adapter/ (VLM LoRA fine-tune)" -ForegroundColor Gray
    Write-Host ""
    $confirm = Read-Host "Type YES to confirm"
    if ($confirm -ne "YES") { 
        Write-Host "Cancelled." -ForegroundColor DarkGray
        return 
    }

    $paths = @(
        (Join-Path $appRoot "tag_db.json"),
        (Join-Path $appRoot "tag_embeddings.pt"),
        (Join-Path $appRoot "tag_classifier.pt"),
        (Join-Path $appRoot "finetuned_clip.pt"),
        (Join-Path $appRoot "vlm_lora_adapter")
    )

    $deleted = 0
    foreach ($p in $paths) {
        if (Test-Path $p) {
            Remove-Item $p -Recurse -Force
            Write-Host "  ✗ Deleted: $(Split-Path $p -Leaf)" -ForegroundColor DarkYellow
            $deleted++
        }
    }

    if ($deleted -eq 0) {
        Write-Host "No model files found to delete." -ForegroundColor DarkGray
    } else {
        Write-Host ""
        Write-Host "✅ Model reset complete. Server will start fresh with base CLIP." -ForegroundColor Green
    }
}

function Show-Settings {
    Apply-Defaults
    Write-Host ""
    Write-Host "═══════════════════════════════════════════════════════════════" -ForegroundColor Cyan
    Write-Host "                      Current Settings                          " -ForegroundColor Cyan
    Write-Host "═══════════════════════════════════════════════════════════════" -ForegroundColor Cyan
    Write-Host ""
    Write-Host "  [Fine-tuning]" -ForegroundColor Yellow
    Write-Host "    FINETUNE_ENABLED    = $($env:FINETUNE_ENABLED)" -ForegroundColor White
    Write-Host "    FINETUNE_STEPS      = $($env:FINETUNE_STEPS)" -ForegroundColor White
    Write-Host "    FINETUNE_LR         = $($env:FINETUNE_LR)" -ForegroundColor White
    Write-Host "    FREEZE_VISUAL       = $($env:FREEZE_VISUAL)" -ForegroundColor White
    Write-Host "    BATCH_SIZE          = $($env:BATCH_SIZE)" -ForegroundColor White
    Write-Host ""
    Write-Host "  [CLIP Model]" -ForegroundColor Yellow
    Write-Host "    OPENCLIP_MODEL      = $($env:OPENCLIP_MODEL)" -ForegroundColor White
    Write-Host "    OPENCLIP_PRETRAINED = $($env:OPENCLIP_PRETRAINED)" -ForegroundColor White
    Write-Host ""
    Write-Host "  [BLIP-2]" -ForegroundColor Yellow
    Write-Host "    USE_BLIP2           = $($env:USE_BLIP2)" -ForegroundColor White
    Write-Host "    BLIP2_MODEL_ID      = $(if($env:BLIP2_MODEL_ID){$env:BLIP2_MODEL_ID}else{'(not set)'})" -ForegroundColor $(if($env:BLIP2_MODEL_ID){'White'}else{'DarkGray'})
    Write-Host "    HF_TOKEN            = $(if($env:HF_TOKEN){'[set]'}else{'(not set)'})" -ForegroundColor $(if($env:HF_TOKEN){'White'}else{'DarkGray'})
    Write-Host ""
    Write-Host "  [Other]" -ForegroundColor Yellow
    Write-Host "    REQUIRE_CUDA        = $($env:REQUIRE_CUDA)" -ForegroundColor White
    Write-Host "    USE_HTTPS           = $($env:USE_HTTPS)" -ForegroundColor White
    Write-Host ""
    Write-Host "  [Vision Language Model]" -ForegroundColor Magenta
    Write-Host "    TAGGING_MODE        = $($env:TAGGING_MODE)" -ForegroundColor White
    Write-Host "    VLM_ENABLED         = $($env:VLM_ENABLED)" -ForegroundColor White
    Write-Host "    VLM_MODEL_ID        = $(if($env:VLM_MODEL_ID){$env:VLM_MODEL_ID}else{'(not set)'})" -ForegroundColor $(if($env:VLM_MODEL_ID){'White'}else{'DarkGray'})
    Write-Host "    VLM_MAX_TAGS        = $($env:VLM_MAX_TAGS)" -ForegroundColor White
    Write-Host "    VLM_LORA_ENABLED    = $($env:VLM_LORA_ENABLED)" -ForegroundColor White
    Write-Host "    VLM_LORA_LR         = $($env:VLM_LORA_LR)" -ForegroundColor White
    Write-Host "    VLM_LORA_RANK       = $($env:VLM_LORA_RANK)" -ForegroundColor White
    Write-Host ""
    
    # Show model files status
    Write-Host "  [Model Files]" -ForegroundColor Yellow
    $files = @("tag_db.json", "tag_embeddings.pt", "tag_classifier.pt", "finetuned_clip.pt", "vlm_lora_adapter")
    foreach ($f in $files) {
        $path = Join-Path $appRoot $f
        if (Test-Path $path) {
            if (Test-Path $path -PathType Container) {
                $dirSize = (Get-ChildItem $path -Recurse -File | Measure-Object -Property Length -Sum).Sum
                $sizeStr = if ($dirSize -gt 1MB) { "{0:N1} MB" -f ($dirSize/1MB) } else { "{0:N0} KB" -f ($dirSize/1KB) }
                Write-Host "    ✓ $f/ ($sizeStr)" -ForegroundColor Green
            } else {
                $size = (Get-Item $path).Length
                $sizeStr = if ($size -gt 1MB) { "{0:N1} MB" -f ($size/1MB) } else { "{0:N0} KB" -f ($size/1KB) }
                Write-Host "    ✓ $f ($sizeStr)" -ForegroundColor Green
            }
        } else {
            Write-Host "    ✗ $f (not created)" -ForegroundColor DarkGray
        }
    }
    Write-Host ""
}

while ($true) {
    Show-Menu
    $choice = (Read-Host "Select option").Trim()
    switch ($choice) {
        "1" { Start-Server }
        "2" { Configure-Settings }
        "3" { Reset-Model }
        "4" { Show-Settings }
        "5" { return }
        default { Write-Host "Invalid choice." -ForegroundColor Red }
    }
}
