// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#pragma once

#include "CoreMinimal.h"
#include "Widgets/SCompoundWidget.h"
#include "Widgets/Input/SComboBox.h"
#include "Widgets/Input/SMultiLineEditableTextBox.h"
#include "Widgets/Views/SListView.h"
#include "AI/IAIProvider.h"

class FConversationManager;

/** A single message entry for the chat display */
struct FChatMessageEntry
{
	FString Sender;   // "User", "AI", "System", "Error", or tool name
	FString Message;
	FDateTime Timestamp;
	bool bIsToolAction = false;
};

/**
 * Main chat panel widget for the Blueprint AI Assistant.
 * Displays conversation messages and provides an input area for the user.
 */
class SBlueprintAIChat : public SCompoundWidget
{
public:
	SLATE_BEGIN_ARGS(SBlueprintAIChat) {}
	SLATE_END_ARGS()

	void Construct(const FArguments& InArgs);

	/** Set the conversation manager to interact with */
	void SetConversationManager(TSharedPtr<FConversationManager> InManager);

private:
	/** Generate a row widget for the message list */
	TSharedRef<ITableRow> OnGenerateRow(TSharedPtr<FChatMessageEntry> Item, const TSharedRef<STableViewBase>& OwnerTable);

	/** Handle send button / Enter key */
	void OnSendMessage();

	/** Handle clear button */
	void OnClearConversation();

	/** Handle stop generation button */
	void OnStopGeneration();

	/** Download chat log as .txt to user's Desktop */
	void OnDownloadChat();

	/** Callback from ConversationManager when a new message arrives */
	void OnNewChatMessage(const FString& Sender, const FString& Message);

	/** Callback from ConversationManager for tool execution updates */
	void OnToolExecutionUpdate(const FString& ToolName, const FString& Status);

	/** Callback for busy state changes */
	void OnBusyStateChanged(bool bIsBusy);

	/** Callback for context usage updates */
	void OnContextUsageUpdated(int32 InputUsed, int32 OutputUsed, int32 ContextWindow);

	/** Provider selector changed */
	void OnProviderChanged(TSharedPtr<FString> NewValue, ESelectInfo::Type SelectionType);

	/** Model selector changed */
	void OnModelChanged(TSharedPtr<FModelInfo> NewValue, ESelectInfo::Type SelectionType);

	/** Fetch models from the current provider */
	void RefreshModelList();

	/** Scroll chat to bottom */
	void ScrollToBottom();

	// Widgets
	TSharedPtr<SListView<TSharedPtr<FChatMessageEntry>>> MessageListView;
	TSharedPtr<SMultiLineEditableTextBox> InputTextBox;
	TSharedPtr<SButton> SendButton;
	TSharedPtr<SComboBox<TSharedPtr<FModelInfo>>> ModelComboBox;

	// Data
	TArray<TSharedPtr<FChatMessageEntry>> Messages;
	TSharedPtr<FConversationManager> ConversationManager;

	// Provider options
	TArray<TSharedPtr<FString>> ProviderOptions;

	// Model options (fetched from provider)
	TArray<TSharedPtr<FModelInfo>> ModelOptions;
	FString CurrentModelDisplay;
	bool bFetchingModels = false;

	bool bIsBusy = false;

	// Context tracking
	int32 CurrentInputTokens = 0;
	int32 CurrentOutputTokens = 0;
	int32 CurrentContextWindow = 0;
};
