// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#pragma once

#include "CoreMinimal.h"
#include "Engine/DeveloperSettings.h"
#include "BlueprintAISettings.generated.h"

UENUM(BlueprintType)
enum class EAIProviderType : uint8
{
	Claude       UMETA(DisplayName = "Anthropic Claude"),
	OpenAI       UMETA(DisplayName = "OpenAI"),
	OpenRouter   UMETA(DisplayName = "OpenRouter"),
	Custom       UMETA(DisplayName = "Custom Endpoint"),
};

/**
 * Settings for the Blueprint AI Assistant plugin.
 * Appears in Project Settings → Plugins → Blueprint AI Assistant.
 * API keys are stored in Saved/Config/ (gitignored by default).
 */
UCLASS(config = BlueprintAIAssistant, meta = (DisplayName = "Blueprint AI Assistant"))
class UBlueprintAISettings : public UDeveloperSettings
{
	GENERATED_BODY()

public:
	UBlueprintAISettings();

	static UBlueprintAISettings* Get();

	// --- Provider Selection ---

	UPROPERTY(config, EditAnywhere, Category = "Provider")
	EAIProviderType ActiveProvider = EAIProviderType::Claude;

	/** Quick reference of common endpoint URLs per provider. Copy-paste the one you need. */
	UPROPERTY(VisibleAnywhere, Category = "Provider", meta = (DisplayName = "Endpoint Reference (copy-paste)", MultiLine = true))
	FText EndpointReference;

	// --- Claude Settings ---

	UPROPERTY(config, EditAnywhere, Category = "Provider|Claude", meta = (DisplayName = "API Key", PasswordField = true))
	FString ClaudeApiKey;

	UPROPERTY(config, EditAnywhere, Category = "Provider|Claude", meta = (DisplayName = "Model"))
	FString ClaudeModel;

	UPROPERTY(config, EditAnywhere, Category = "Provider|Claude", meta = (DisplayName = "API Version"))
	FString ClaudeApiVersion;

	// --- OpenAI Settings ---

	UPROPERTY(config, EditAnywhere, Category = "Provider|OpenAI", meta = (DisplayName = "API Key", PasswordField = true))
	FString OpenAIApiKey;

	UPROPERTY(config, EditAnywhere, Category = "Provider|OpenAI", meta = (DisplayName = "Model"))
	FString OpenAIModel;

	/** OpenAI API endpoint URL.
	  * Use https://api.openai.com/v1/chat/completions for GPT-4o, o3, o4, etc.
	  * Use https://api.openai.com/v1/responses for Codex models (gpt-5.3-codex, etc.).
	  * The plugin auto-detects the format from the URL. */
	UPROPERTY(config, EditAnywhere, Category = "Provider|OpenAI", meta = (DisplayName = "Endpoint URL",
		ToolTip = "Chat models (GPT-4o, o3, o4): .../v1/chat/completions\nCodex models: .../v1/responses\nThe request format adapts automatically."))
	FString OpenAIEndpoint;

	// --- OpenRouter Settings ---

	UPROPERTY(config, EditAnywhere, Category = "Provider|OpenRouter", meta = (DisplayName = "API Key", PasswordField = true))
	FString OpenRouterApiKey;

	UPROPERTY(config, EditAnywhere, Category = "Provider|OpenRouter", meta = (DisplayName = "Model"))
	FString OpenRouterModel;

	// --- Custom Endpoint Settings ---

	UPROPERTY(config, EditAnywhere, Category = "Provider|Custom", meta = (DisplayName = "Endpoint URL"))
	FString CustomEndpoint;

	UPROPERTY(config, EditAnywhere, Category = "Provider|Custom", meta = (DisplayName = "API Key", PasswordField = true))
	FString CustomApiKey;

	UPROPERTY(config, EditAnywhere, Category = "Provider|Custom", meta = (DisplayName = "Model"))
	FString CustomModel;

	UPROPERTY(config, EditAnywhere, Category = "Provider|Custom", meta = (DisplayName = "API Format (Claude or OpenAI)"))
	EAIProviderType CustomApiFormat = EAIProviderType::OpenAI;

	// --- Generation Params ---

	UPROPERTY(config, EditAnywhere, Category = "Generation", meta = (ClampMin = "0.0", ClampMax = "2.0"))
	float Temperature = 0.3f;

	UPROPERTY(config, EditAnywhere, Category = "Generation", meta = (ClampMin = "256", ClampMax = "65536"))
	int32 MaxTokens = 8192;

	// --- Behavior ---

	UPROPERTY(config, EditAnywhere, Category = "Behavior", meta = (DisplayName = "Auto-compile Blueprints after modification"))
	bool bAutoCompile = true;

	UPROPERTY(config, EditAnywhere, Category = "Behavior", meta = (DisplayName = "Max agent tool-call iterations per turn", ClampMin = "1", ClampMax = "500"))
	int32 MaxToolIterations = 200;

	// --- Payload Limits ---

	/** Maximum characters per individual message sent to the AI. Larger messages are truncated.
	  * Lower values reduce payload size (useful for bridges/proxies with size limits). */
	UPROPERTY(config, EditAnywhere, Category = "Payload", meta = (DisplayName = "Max chars per message", ClampMin = "500", ClampMax = "100000"))
	int32 MaxMessageChars = 8000;

	/** Maximum characters for a single tool result. Tool outputs (e.g. get_blueprint_info)
	  * can be very large. This cap is applied before the general message cap. */
	UPROPERTY(config, EditAnywhere, Category = "Payload", meta = (DisplayName = "Max chars per tool result", ClampMin = "500", ClampMax = "50000"))
	int32 MaxToolResultChars = 12000;

	/** Maximum total characters across ALL messages in a single request.
	  * When exceeded, oldest messages are dropped (sliding window). Set higher for direct API usage,
	  * lower for bridges/proxies. 0 = unlimited. */
	UPROPERTY(config, EditAnywhere, Category = "Payload", meta = (DisplayName = "Max total payload chars", ClampMin = "0", ClampMax = "500000"))
	int32 MaxTotalPayloadChars = 32000;

	// --- Logging ---

	/** Enable comprehensive debug logging to a .txt file on your Desktop.
	  * Captures everything: chat messages, AI responses, HTTP requests/responses,
	  * tool calls, tool results, errors, system events, and raw payloads.
	  * A new timestamped file is created each editor session. */
	UPROPERTY(config, EditAnywhere, Category = "Logging", meta = (DisplayName = "Enable debug log file (Desktop)"))
	bool bEnableFileLogging = true;

	// --- Network ---

	/** HTTP request timeout in seconds. Reasoning models (GPT-5, Claude Opus) may need
	  * 60-120s to respond. Bridges/proxies (CopilotAPI, LM Studio) add extra latency.
	  * Increase this if you see timeout errors. Default: 180s (3 minutes). */
	UPROPERTY(config, EditAnywhere, Category = "Network", meta = (DisplayName = "HTTP request timeout (seconds)", ClampMin = "30", ClampMax = "600"))
	int32 HttpRequestTimeoutSeconds = 180;

	// --- UDeveloperSettings overrides ---
	virtual FName GetContainerName() const override { return TEXT("Project"); }
	virtual FName GetCategoryName() const override { return TEXT("Plugins"); }
	virtual FName GetSectionName() const override { return TEXT("Blueprint AI Assistant"); }

#if WITH_EDITOR
	virtual FText GetSectionText() const override;
	virtual FText GetSectionDescription() const override;
	virtual void PostEditChangeProperty(FPropertyChangedEvent& PropertyChangedEvent) override;
#endif
};
