// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#pragma once

#include "CoreMinimal.h"
#include "Misc/DateTime.h"

/**
 * A single persistent memory entry.
 */
struct FMemoryEntry
{
	/** Unique identifier (e.g. "M1", "M2") */
	FString Id;

	/** The content/fact to remember */
	FString Content;

	/** When this memory was created */
	FDateTime CreatedAt;
};

/**
 * Persistent AI memory system.
 * Saves/loads facts to a JSON file so the AI can remember user preferences,
 * project conventions, and important learnings across sessions.
 *
 * Storage: {ProjectDir}/Saved/BlueprintAI/memories.json
 */
class FBlueprintAIMemory
{
public:
	/** Get the singleton instance */
	static FBlueprintAIMemory& Get();

	/** Load memories from disk */
	void Load();

	/** Save memories to disk */
	void Save();

	/** Add a new memory entry. Returns the assigned Id. */
	FString AddMemory(const FString& Content);

	/** Remove a memory by its Id. Returns true if found and removed. */
	bool RemoveMemory(const FString& Id);

	/** Get all memories */
	const TArray<FMemoryEntry>& GetMemories() const { return Memories; }

	/**
	 * Build the system prompt section for memories.
	 * Returns empty string if no memories exist.
	 * Format: "=== AI MEMORY (persistent across sessions) ===\n[M1] fact one\n..."
	 */
	FString BuildMemoryPromptSection() const;

private:
	FBlueprintAIMemory();

	/** Get the file path for the memories JSON file */
	FString GetMemoryFilePath() const;

	TArray<FMemoryEntry> Memories;
	int32 NextMemoryId = 1;
	bool bLoaded = false;
};
