// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#pragma once

#include "CoreMinimal.h"
#include "HAL/CriticalSection.h"

/**
 * Centralized file logger for the Blueprint AI Assistant plugin.
 *
 * Writes a comprehensive .txt log to the user's Desktop containing every
 * interaction detail: chat messages, HTTP requests/responses, tool calls,
 * tool results, errors, system events, model info, and raw payloads.
 *
 * Usage:
 *   FBlueprintAILogger::Get().Log(TEXT("Category"), TEXT("message..."));
 *
 * A new log file is created each session (timestamped).
 * Can be toggled on/off via Project Settings → Plugins → Blueprint AI Assistant → Logging.
 */
class FBlueprintAILogger
{
public:
	static FBlueprintAILogger& Get();

	/** Initialize / open the log file. Call once at startup. */
	void Initialize();

	/** Close the file handle and flush. Call at shutdown. */
	void Shutdown();

	/** Log a message with a category tag. Thread-safe. */
	void Log(const FString& Category, const FString& Message);

	/** Convenience: log with printf-style formatting */
	void Logf(const FString& Category, const TCHAR* Fmt, ...);

	/** Check if logging is enabled (setting + file open) */
	bool IsEnabled() const;

	/** Get the current log file path (empty if not open) */
	const FString& GetLogFilePath() const { return LogFilePath; }

private:
	FBlueprintAILogger() = default;
	~FBlueprintAILogger();

	FBlueprintAILogger(const FBlueprintAILogger&) = delete;
	FBlueprintAILogger& operator=(const FBlueprintAILogger&) = delete;

	/** Get a timestamp string for the current moment */
	FString GetTimestamp() const;

	FString LogFilePath;
	TUniquePtr<FArchive> FileWriter;
	FCriticalSection WriteLock;
	bool bInitialized = false;
};
