// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#pragma once

#include "CoreMinimal.h"
#include "Dom/JsonObject.h"

/**
 * Result of executing a Blueprint action / tool.
 */
struct FBlueprintActionResult
{
	bool bSuccess = false;
	FString Message;

	/** Structured result data (returned to the AI as JSON) */
	TSharedPtr<FJsonObject> Data;

	static FBlueprintActionResult Success(const FString& Msg, TSharedPtr<FJsonObject> InData = nullptr)
	{
		FBlueprintActionResult R;
		R.bSuccess = true;
		R.Message = Msg;
		R.Data = InData;
		return R;
	}

	static FBlueprintActionResult Failure(const FString& Msg)
	{
		FBlueprintActionResult R;
		R.bSuccess = false;
		R.Message = Msg;
		return R;
	}

	/** Serialize to JSON string for returning to the AI */
	FString ToJsonString() const;
};

/**
 * Interface for a single Blueprint manipulation action (tool).
 * Each action maps to one tool the AI can call.
 */
class IBlueprintAction : public TSharedFromThis<IBlueprintAction>
{
public:
	virtual ~IBlueprintAction() = default;

	/** Unique tool/action name (e.g. "create_blueprint"). Must match AI tool call names. */
	virtual FString GetName() const = 0;

	/** Human-readable description shown in the AI system prompt */
	virtual FString GetDescription() const = 0;

	/**
	 * JSON Schema describing the parameters this action accepts.
	 * Used to generate tool definitions for the AI.
	 * Must be a valid JSON Schema object with "type": "object", "properties": {...}, "required": [...]
	 */
	virtual TSharedPtr<FJsonObject> GetParameterSchema() const = 0;

	/**
	 * Execute the action with the given parameters.
	 * Called on the game thread.
	 * @param Params  Parsed JSON parameters from the AI tool call
	 * @return Result with success/failure and optional data
	 */
	virtual FBlueprintActionResult Execute(const TSharedPtr<FJsonObject>& Params) = 0;
};
