// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#pragma once

#include "Actions/IBlueprintAction.h"

class FActionRegistry;

/**
 * Action: add_node
 * Adds a node to a Blueprint's graph. Supports function calls, events, branches,
 * loops, variable access, and many more node types.
 *
 * Returns the node's tracking ID and a list of all its pins (name, direction, type)
 * so the AI can make connections.
 */
class FAddNodeAction : public IBlueprintAction
{
public:
	FAddNodeAction(FActionRegistry* InRegistry) : Registry(InRegistry) {}

	virtual FString GetName() const override { return TEXT("add_node"); }

	virtual FString GetDescription() const override
	{
		return TEXT("Adds a node to a Blueprint graph. Returns a node_id and all pin names/types for connecting later.\n\n"
			"node_type values:\n"
			"- 'CallFunction': Call any static or member function. Requires function_class and function_name.\n"
			"- 'Branch': If/else branch. Has 'Condition' (bool input), 'True' and 'False' exec outputs.\n"
			"- 'Sequence': Execution sequence. Has 'then_0', 'then_1', etc. exec outputs.\n"
			"- 'ForLoop': For loop. Has 'FirstIndex', 'LastIndex' int inputs, 'LoopBody' and 'Completed' exec outputs.\n"
			"- 'ForEachLoop': Iterates over an array.\n"
			"- 'WhileLoop': While loop.\n"
			"- 'CustomEvent': Creates a custom event. Requires custom_event_name.\n"
			"- 'Event': Built-in event override (BeginPlay, Tick, etc.). Requires event_name.\n"
			"- 'VariableGet': Get a Blueprint variable. Requires variable_name.\n"
			"- 'VariableSet': Set a Blueprint variable. Requires variable_name.\n"
			"- 'SpawnActor': Spawn actor from class.\n"
			"- 'Delay': Delay node (Latent). Has 'Duration' float input.\n"
			"- 'PrintString': Print string to screen/log.\n"
			"- 'SetTimer': Set timer by function name. Has 'FunctionName' (string), 'Time' (float), 'bLooping' (bool) pins.\n"
			"- 'SetTimerByEvent': Set timer by event/delegate. Has 'Delegate', 'Time' (float), 'bLooping' (bool) pins.\n"
			"- 'GetPlayerController': Get player controller.\n"
			"- 'MakeArray': Construct array.\n"
			"- 'Cast': Cast to a class. Requires cast_to_class.\n"
			"- 'Self': Get reference to self.\n\n"
			"Common function_class values for CallFunction:\n"
			"- 'KismetSystemLibrary' (PrintString, Delay, SetTimerByFunctionName, IsValid, etc.)\n"
			"- 'KismetMathLibrary' (Add_FloatFloat, Multiply_FloatFloat, RandomFloatInRange, etc.)\n"
			"- 'GameplayStatics' (GetPlayerController, SpawnActor, GetAllActorsOfClass, PlaySound2D, etc.)\n"
			"- 'KismetStringLibrary' (Concat_StrStr, Contains, etc.)\n"
			"- 'KismetArrayLibrary' (Array_Add, Array_Length, etc.)\n"
			"- Or any class name for member functions.");
	}

	virtual TSharedPtr<FJsonObject> GetParameterSchema() const override;
	virtual FBlueprintActionResult Execute(const TSharedPtr<FJsonObject>& Params) override;

private:
	/** Get pin info as JSON array for the result */
	TArray<TSharedPtr<FJsonValue>> SerializePins(UEdGraphNode* Node) const;

	FActionRegistry* Registry = nullptr;
};
