// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#pragma once

#include "CoreMinimal.h"
#include "Actions/IBlueprintAction.h"

class IAIProvider;
class UEdGraphNode;

/**
 * Registry of all available Blueprint actions (tools).
 * Also maintains a node tracking map for referencing created nodes by short ID.
 */
class FActionRegistry : public TSharedFromThis<FActionRegistry>
{
public:
	FActionRegistry();

	/** Register a new action. Call during plugin startup. */
	void RegisterAction(TSharedPtr<IBlueprintAction> Action);

	/** Find an action by name */
	TSharedPtr<IBlueprintAction> FindAction(const FString& Name) const;

	/** Get all registered actions */
	const TArray<TSharedPtr<IBlueprintAction>>& GetAllActions() const { return Actions; }

	/**
	 * Build tool definition JSON array for a specific AI provider.
	 * Uses the provider's FormatToolDefinition() to produce the right format.
	 */
	TArray<TSharedPtr<FJsonObject>> BuildToolDefinitions(const TSharedPtr<IAIProvider>& Provider) const;

	/** Generate the system prompt section listing all tools */
	FString GenerateToolDocumentation() const;

	// ── Node tracking ──────────────────────────────────────────────────────

	/** Register a node and return a short ID (e.g. "N1") */
	FString TrackNode(UEdGraphNode* Node);

	/** Look up a tracked node by its short ID */
	UEdGraphNode* FindTrackedNode(const FString& NodeId) const;

	// ── Blueprint tracking ────────────────────────────────────────────────

	/** Track a created blueprint (name → full path) */
	void TrackCreatedBlueprint(const FString& Name, const FString& Path);

	/** Get all created blueprints this session */
	const TArray<TPair<FString, FString>>& GetCreatedBlueprints() const { return CreatedBlueprints; }

	/** Build a compact summary of all tracked nodes grouped by owning blueprint */
	FString GetTrackedNodesInfo() const;

	/** Clear all session state (tracked nodes + created blueprints) */
	void ClearSessionState();

private:
	void RegisterBuiltInActions();

	TArray<TSharedPtr<IBlueprintAction>> Actions;
	TMap<FString, TSharedPtr<IBlueprintAction>> ActionMap;

	// Node tracking
	int32 NextNodeId = 1;
	TMap<FString, TWeakObjectPtr<UEdGraphNode>> TrackedNodes;

	// Blueprint tracking
	TArray<TPair<FString, FString>> CreatedBlueprints;
};
