// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#pragma once

#include "AI/AIProviderBase.h"

/**
 * AI provider implementation for the OpenAI API.
 * Supports both Chat Completions (/v1/chat/completions) and Responses (/v1/responses) endpoints.
 * Auto-detects which format to use based on the configured endpoint URL.
 * Supports function calling via the tools[] parameter.
 */
class FOpenAIProvider : public FAIProviderBase
{
public:
	FOpenAIProvider(const FString& InApiKey, const FString& InModel, const FString& InEndpoint = TEXT(""));

	virtual FString GetProviderName() const override { return TEXT("OpenAI"); }
	virtual bool SupportsToolCalling() const override { return true; }

	virtual void SendRequest(
		const FAICompletionRequest& Request,
		FOnAIResponse OnResponse,
		FOnAIError OnError
	) override;

	virtual TSharedPtr<FJsonObject> FormatToolDefinition(
		const FString& Name,
		const FString& Description,
		const TSharedPtr<FJsonObject>& ParamSchema
	) const override;

protected:
	virtual FAICompletionResponse ParseResponse(const FString& ResponseBody) override;
	virtual FString GetModelsEndpoint() const override;
	virtual TMap<FString, FString> GetAuthHeaders() const override;

	/** Build messages array for Chat Completions API */
	TArray<TSharedPtr<FJsonValue>> BuildMessages(const TArray<FAIMessage>& Messages) const;

	/** Build input array for Responses API */
	TArray<TSharedPtr<FJsonValue>> BuildResponsesInput(const TArray<FAIMessage>& Messages, FString& OutInstructions) const;

	/** Parse response from Responses API format */
	FAICompletionResponse ParseResponsesAPIResponse(const TSharedPtr<FJsonObject>& Json) const;

	FString ApiKey;
	FString Model;
	FString Endpoint;

	/** True when the endpoint URL contains '/responses' — uses Responses API format */
	bool bUseResponsesAPI = false;
};
