// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#pragma once

#include "CoreMinimal.h"
#include "Dom/JsonObject.h"

// ─── Data Structures ────────────────────────────────────────────────────────

/** A single tool call requested by the AI */
struct FAIToolCall
{
	/** Provider-assigned ID (used when returning results) */
	FString Id;

	/** Tool / action name (matches IBlueprintAction::GetName) */
	FString Name;

	/** JSON parameters to pass to the action */
	TSharedPtr<FJsonObject> Parameters;
};

/** Role in the conversation */
enum class EAIMessageRole : uint8
{
	System,
	User,
	Assistant,
	ToolResult,
};

/** One message in the conversation history */
struct FAIMessage
{
	EAIMessageRole Role = EAIMessageRole::User;

	/** Text content (can be empty for tool-only messages) */
	FString Content;

	/** For ToolResult: the ID of the tool call this is responding to */
	FString ToolCallId;

	/** For ToolResult: the name of the tool */
	FString ToolName;

	/** For Assistant messages: any tool calls the assistant made */
	TArray<FAIToolCall> ToolCalls;

	// Convenience factories
	static FAIMessage MakeSystem(const FString& Text)
	{
		FAIMessage Msg;
		Msg.Role = EAIMessageRole::System;
		Msg.Content = Text;
		return Msg;
	}

	static FAIMessage MakeUser(const FString& Text)
	{
		FAIMessage Msg;
		Msg.Role = EAIMessageRole::User;
		Msg.Content = Text;
		return Msg;
	}

	static FAIMessage MakeAssistant(const FString& Text, const TArray<FAIToolCall>& InToolCalls = {})
	{
		FAIMessage Msg;
		Msg.Role = EAIMessageRole::Assistant;
		Msg.Content = Text;
		Msg.ToolCalls = InToolCalls;
		return Msg;
	}

	static FAIMessage MakeToolResult(const FString& CallId, const FString& ToolName, const FString& ResultJson)
	{
		FAIMessage Msg;
		Msg.Role = EAIMessageRole::ToolResult;
		Msg.ToolCallId = CallId;
		Msg.ToolName = ToolName;
		Msg.Content = ResultJson;
		return Msg;
	}
};

/** Request payload sent to the AI provider */
struct FAICompletionRequest
{
	TArray<FAIMessage> Messages;
	TArray<TSharedPtr<FJsonObject>> Tools;   // Tool definitions in provider-native format
	float Temperature = 0.3f;
	int32 MaxTokens = 8192;
	FString Model;
};

/** Parsed response from the AI */
struct FAICompletionResponse
{
	bool bSuccess = false;
	FString ErrorMessage;

	/** Assistant text content (may be empty if only tool calls) */
	FString TextContent;

	/** Tool calls requested by the assistant */
	TArray<FAIToolCall> ToolCalls;

	/** Raw JSON for debugging */
	FString RawJson;

	/** Token usage */
	int32 InputTokens = 0;
	int32 OutputTokens = 0;
	int32 TotalTokens = 0;
};

// ─── Delegates ──────────────────────────────────────────────────────────────

/** Info about a model returned from the provider's model list */
struct FModelInfo
{
	/** Model ID (what gets sent in the API request) */
	FString Id;

	/** Human-readable display name (may be same as Id) */
	FString DisplayName;

	/** Price per 1M input tokens in USD. -1 = unknown */
	double InputPricePer1M = -1.0;

	/** Price per 1M output tokens in USD. -1 = unknown */
	double OutputPricePer1M = -1.0;

	/** Whether this model is free */
	bool bIsFree = false;

	/** Context window in tokens. 0 = unknown */
	int32 ContextLength = 0;

	/** Build a display string like "gpt-4o [FREE]" or "gpt-4o ($2.50/$10.00)" */
	FString GetDisplayString() const
	{
		FString Result = Id;
		if (bIsFree)
		{
			Result += TEXT("  [FREE]");
		}
		else if (InputPricePer1M >= 0 && OutputPricePer1M >= 0)
		{
			Result += FString::Printf(TEXT("  ($%.2f/$%.2f)"), InputPricePer1M, OutputPricePer1M);
		}
		return Result;
	}
};

DECLARE_DELEGATE_OneParam(FOnAIResponse, const FAICompletionResponse& /*Response*/);
DECLARE_DELEGATE_OneParam(FOnAIError, const FString& /*Error*/);
DECLARE_DELEGATE_OneParam(FOnModelsReceived, const TArray<FModelInfo>& /*Models*/);

// ─── Provider Interface ─────────────────────────────────────────────────────

/**
 * Abstract interface for AI API providers.
 * Each provider converts our unified structures into provider-specific JSON.
 */
class IAIProvider : public TSharedFromThis<IAIProvider>
{
public:
	virtual ~IAIProvider() = default;

	/** Human-readable provider name */
	virtual FString GetProviderName() const = 0;

	/** Send a completion request (async — response via delegate) */
	virtual void SendRequest(
		const FAICompletionRequest& Request,
		FOnAIResponse OnResponse,
		FOnAIError OnError
	) = 0;

	/** Cancel any in-flight request */
	virtual void CancelRequest() = 0;

	/** Does this provider support tool / function calling? */
	virtual bool SupportsToolCalling() const = 0;

	/**
	 * Convert a unified tool definition into this provider's JSON format.
	 * @param Name          Tool name
	 * @param Description   Tool description
	 * @param ParamSchema   JSON Schema for parameters
	 */
	virtual TSharedPtr<FJsonObject> FormatToolDefinition(
		const FString& Name,
		const FString& Description,
		const TSharedPtr<FJsonObject>& ParamSchema
	) const = 0;

	/**
	 * Fetch available model IDs from the provider's API.
	 * Results are returned asynchronously via delegate.
	 */
	virtual void FetchAvailableModels(FOnModelsReceived OnSuccess, FOnAIError OnError) = 0;
};
