// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#pragma once

#include "CoreMinimal.h"
#include "AI/IAIProvider.h"

class FActionRegistry;
class IAIProvider;

/** Delegate broadcast when a new chat message should be displayed in the UI */
DECLARE_MULTICAST_DELEGATE_TwoParams(FOnChatMessage, const FString& /*Sender*/, const FString& /*Message*/);

/** Delegate broadcast when a tool is being executed */
DECLARE_MULTICAST_DELEGATE_TwoParams(FOnToolExecution, const FString& /*ToolName*/, const FString& /*Status*/);

/** Delegate broadcast when the AI is thinking / idle */
DECLARE_MULTICAST_DELEGATE_OneParam(FOnBusyStateChanged, bool /*bIsBusy*/);

/** Delegate broadcast when context usage changes */
DECLARE_MULTICAST_DELEGATE_ThreeParams(FOnContextUsageUpdated, int32 /*InputUsed*/, int32 /*OutputUsed*/, int32 /*ContextWindow*/);

/**
 * Manages conversation state and the agentic tool-calling loop.
 *
 * Flow:
 *   1. User sends message → AddUserMessage()
 *   2. Build API request (system prompt + history + tools)
 *   3. Send to current AI provider
 *   4. On response:
 *      a. Text with <tool_call> tags → parse, execute, LOOP back to step 2
 *      b. Structured tool_calls → execute each, append results, LOOP back to step 2
 *      c. Plain text only → display to user, end turn
 *   5. Repeat until AI responds with text only (no tool calls) or max iterations hit
 */
class FConversationManager : public TSharedFromThis<FConversationManager>
{
public:
	FConversationManager(TSharedPtr<FActionRegistry> InRegistry);

	/** Send a user message and kick off the agent loop */
	void SendUserMessage(const FString& UserMessage);

	/** Stop the current generation / agent loop */
	void StopGeneration();

	/** Clear conversation history and node tracker */
	void ClearConversation();

	/** Set the AI provider to use */
	void SetProvider(TSharedPtr<IAIProvider> InProvider);

	// ── Delegates for the UI ───────────────────────────────────────────────
	FOnChatMessage OnChatMessage;
	FOnToolExecution OnToolExecution;
	FOnBusyStateChanged OnBusyStateChanged;
	FOnContextUsageUpdated OnContextUsageUpdated;

	bool IsBusy() const { return bIsBusy; }

	/** Set the context window size for the current model (in tokens) */
	void SetModelContextWindow(int32 ContextTokens);

private:
	/** Build the system prompt describing available tools and Blueprint concepts */
	FString BuildSystemPrompt() const;

	/** Build a project snapshot (list of assets, folder structure) for the system prompt */
	FString BuildProjectSnapshot() const;

	/** Build working state section (created BPs, tracked nodes) for the system prompt */
	FString BuildWorkingState() const;

	/** Get the effective max payload chars (model context if known, else settings) */
	int32 GetEffectiveMaxPayloadChars() const;

	/** Summarize the conversation to free context space */
	void SummarizeConversation();

	/** Send the current conversation to the AI provider */
	void SendToProvider();

	/** Handle AI response */
	void HandleResponse(const FAICompletionResponse& Response);

	/** Handle AI error */
	void HandleError(const FString& Error);

	/** Execute a batch of tool calls and return results (structured tool role) */
	void ExecuteToolCalls(const TArray<FAIToolCall>& ToolCalls);

	/**
	 * Execute tool calls from text-based parsing.
	 * Results are aggregated into a single user-role message instead of
	 * individual "tool" role messages, for compatibility with bridges/proxies
	 * that don't support the tool role.
	 */
	void ExecuteTextToolCalls(const TArray<FAIToolCall>& ToolCalls);

	/**
	 * Parse text-based tool calls from AI text content.
	 * Looks for <tool_call>{"name":"...","parameters":{...}}</tool_call> blocks.
	 * Returns extracted tool calls and sets OutCleanText to the text with tool calls removed.
	 */
	TArray<FAIToolCall> ParseTextBasedToolCalls(const FString& TextContent, FString& OutCleanText) const;

	TSharedPtr<FActionRegistry> ActionRegistry;
	TSharedPtr<IAIProvider> Provider;

	TArray<FAIMessage> ConversationHistory;

	/** Cache of recent tool call results keyed by "name|params_json" to detect duplicates */
	TMap<FString, FString> ToolCallCache;

	bool bIsBusy = false;
	int32 CurrentIteration = 0;
	int32 MaxIterations = 25;
	mutable int32 TextToolCallCounter = 0;
	int32 ConsecutiveDuplicateBatches = 0;

	// ── Context tracking ─────────────────────────────────────────────────
	int32 TotalInputTokensUsed = 0;
	int32 TotalOutputTokensUsed = 0;
	int32 ModelContextWindow = 0;
	int32 LastRequestInputTokens = 0;
	int32 LastRequestPayloadChars = 0;
	bool bNeedsSummarization = false;

	// ── Narration nudge detector ─────────────────────────────────────────
	int32 NudgeCount = 0;

	// ── Wiring deferral tracking ─────────────────────────────────────────
	/** True when the previous iteration deferred connect_nodes/set_pin_value calls */
	bool bHadDeferredWiring = false;
	/** Count of wiring nudges sent when AI fails to follow up on deferred calls */
	int32 WiringNudgeCount = 0;

	// ── Error loop detection ─────────────────────────────────────────────
	TMap<FString, int32> RecentErrorCounts;
	int32 ConsecutiveFailedIterations = 0;

	// ── Success loop detection (identical batch repetition) ─────────────
	/** Signature of the previous batch (sorted tool names + params) */
	FString LastBatchSignature;
	/** How many consecutive iterations used the exact same batch signature */
	int32 ConsecutiveIdenticalBatches = 0;

	/** Compute a deterministic signature for a batch of tool calls */
	static FString ComputeBatchSignature(const TArray<FAIToolCall>& ToolCalls);

	/** Check for and handle success loop (identical batch repeated too many times) */
	void CheckSuccessLoop(const TArray<FAIToolCall>& ToolCalls);

	/** Build a cache key for a tool call (name + serialized params) */
	static FString MakeToolCacheKey(const FString& Name, const TSharedPtr<FJsonObject>& Params);

	/** Remove cached results for observation tools (compile, get_blueprint_info, etc.)
	 *  after a state-changing action modifies the Blueprint. */
	void InvalidateStaleCacheEntries();

	/**
	 * Compact old observation tool results to short summaries.
	 * Replaces full JSON blobs from get_blueprint_info, get_class_functions, etc.
	 * with one-liner summaries, preserving only state-changing tool results intact.
	 * This keeps old context lean while the AI already processed the full data.
	 */
	static FString CompactOldToolResults(const FString& AggregatedToolResults);
};
