// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#pragma once

#include "CoreMinimal.h"
#include "AI/IAIProvider.h"
#include "Interfaces/IHttpRequest.h"

/**
 * Shared base for HTTP-based AI providers.
 * Handles HTTP lifecycle, JSON serialization/parsing, and common headers.
 */
class FAIProviderBase : public IAIProvider
{
public:
	virtual ~FAIProviderBase() override;

	virtual void CancelRequest() override;

	virtual void FetchAvailableModels(FOnModelsReceived OnSuccess, FOnAIError OnError) override;

protected:
	/** Build and send an HTTP GET request (for model listing, etc.) */
	void SendHttpGetRequest(
		const FString& Url,
		const TMap<FString, FString>& Headers,
		TFunction<void(const FString& /*ResponseBody*/)> OnSuccess,
		FOnAIError OnError
	);

	/** Return the models endpoint URL. Override per provider. */
	virtual FString GetModelsEndpoint() const { return FString(); }

	/** Return the auth headers needed for requests. Override per provider. */
	virtual TMap<FString, FString> GetAuthHeaders() const { return {}; }

	/** Parse a model list response body into an array of model ID strings. Override per provider. */
	virtual TArray<FModelInfo> ParseModelListResponse(const FString& ResponseBody) const;

	/** Build and send an HTTP POST request with JSON body */
	void SendHttpRequest(
		const FString& Url,
		const TMap<FString, FString>& Headers,
		const TSharedRef<FJsonObject>& Body,
		FOnAIResponse OnResponse,
		FOnAIError OnError
	);

	/** Override in subclass to parse the raw JSON response into FAICompletionResponse */
	virtual FAICompletionResponse ParseResponse(const FString& ResponseBody) = 0;

	/** JSON helpers */
	static FString JsonToString(const TSharedRef<FJsonObject>& Obj);
	static TSharedPtr<FJsonObject> StringToJson(const FString& Str);

	/** Shared helper: create a JSON Schema property */
	static TSharedPtr<FJsonObject> MakeStringProp(const FString& Description);
	static TSharedPtr<FJsonObject> MakeIntProp(const FString& Description);
	static TSharedPtr<FJsonObject> MakeBoolProp(const FString& Description);
	static TSharedPtr<FJsonObject> MakeArrayProp(const FString& Description, TSharedPtr<FJsonObject> ItemsSchema);

private:
	TSharedPtr<class IHttpRequest> ActiveRequest;
	bool bIsRetrying = false;
};
