// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#include "UI/SBlueprintAIChat.h"
#include "AI/ConversationManager.h"
#include "AI/IAIProvider.h"
#include "Settings/BlueprintAISettings.h"
#include "BlueprintAIAssistantModule.h"
#include "ISettingsModule.h"
#include "Async/Async.h"

#include "Widgets/Layout/SBox.h"
#include "Widgets/Layout/SScrollBox.h"
#include "Widgets/Layout/SSeparator.h"
#include "Widgets/Layout/SSpacer.h"
#include "Widgets/Input/SButton.h"
#include "Widgets/Input/SComboBox.h"
#include "Widgets/Input/SEditableText.h"
#include "Widgets/Input/SMultiLineEditableTextBox.h"
#include "Widgets/Text/STextBlock.h"
#include "Widgets/Text/SRichTextBlock.h"
#include "Widgets/Views/SListView.h"
#include "Styling/AppStyle.h"
#include "Framework/Application/SlateApplication.h"
#include "HAL/PlatformProcess.h"
#include "HAL/PlatformMisc.h"
#include "Misc/FileHelper.h"
#include "Misc/Paths.h"

#define LOCTEXT_NAMESPACE "BlueprintAIChat"

// ─── Row widget for a single chat message ────────────────────────────────────

class SChatMessageRow : public SMultiColumnTableRow<TSharedPtr<FChatMessageEntry>>
{
public:
	SLATE_BEGIN_ARGS(SChatMessageRow) {}
		SLATE_ARGUMENT(TSharedPtr<FChatMessageEntry>, MessageEntry)
	SLATE_END_ARGS()

	void Construct(const FArguments& InArgs, const TSharedRef<STableViewBase>& InOwnerTableView)
	{
		Entry = InArgs._MessageEntry;
		SMultiColumnTableRow<TSharedPtr<FChatMessageEntry>>::Construct(
			FSuperRowType::FArguments(), InOwnerTableView);
	}

	virtual TSharedRef<SWidget> GenerateWidgetForColumn(const FName& ColumnName) override
	{
		if (!Entry.IsValid()) return SNullWidget::NullWidget;

		// Determine color based on sender
		FSlateColor SenderColor = FSlateColor(FLinearColor::White);
		if (Entry->Sender == TEXT("User"))
		{
			SenderColor = FSlateColor(FLinearColor(0.3f, 0.7f, 1.0f));
		}
		else if (Entry->Sender == TEXT("AI"))
		{
			SenderColor = FSlateColor(FLinearColor(0.3f, 1.0f, 0.5f));
		}
		else if (Entry->Sender == TEXT("Error"))
		{
			SenderColor = FSlateColor(FLinearColor(1.0f, 0.3f, 0.3f));
		}
		else if (Entry->Sender == TEXT("System"))
		{
			SenderColor = FSlateColor(FLinearColor(0.7f, 0.7f, 0.7f));
		}
		else // Tool action
		{
			SenderColor = FSlateColor(FLinearColor(1.0f, 0.8f, 0.2f));
		}

		if (ColumnName == TEXT("Message"))
		{
			return SNew(SHorizontalBox)
				+ SHorizontalBox::Slot()
				.AutoWidth()
				.Padding(4, 2)
				[
					SNew(STextBlock)
					.Text(FText::FromString(FString::Printf(TEXT("[%s]"), *Entry->Sender)))
					.ColorAndOpacity(SenderColor)
					.Font(FCoreStyle::GetDefaultFontStyle("Bold", 9))
				]
				+ SHorizontalBox::Slot()
				.FillWidth(1.0f)
				.Padding(4, 2)
				[
					SNew(SEditableText)
					.Text(FText::FromString(Entry->Message))
					.IsReadOnly(true)
					.Font(Entry->bIsToolAction
						? FCoreStyle::GetDefaultFontStyle("Italic", 8)
						: FCoreStyle::GetDefaultFontStyle("Regular", 9))
				];
		}

		return SNullWidget::NullWidget;
	}

private:
	TSharedPtr<FChatMessageEntry> Entry;
};

// ─── SBlueprintAIChat ───────────────────────────────────────────────────────

void SBlueprintAIChat::Construct(const FArguments& InArgs)
{
	// Provider options
	ProviderOptions.Add(MakeShared<FString>(TEXT("Claude")));
	ProviderOptions.Add(MakeShared<FString>(TEXT("OpenAI")));
	ProviderOptions.Add(MakeShared<FString>(TEXT("OpenRouter")));
	ProviderOptions.Add(MakeShared<FString>(TEXT("Custom")));

	// Initialize model display from settings
	{
		UBlueprintAISettings* S = UBlueprintAISettings::Get();
		switch (S->ActiveProvider)
		{
		case EAIProviderType::Claude:     CurrentModelDisplay = S->ClaudeModel; break;
		case EAIProviderType::OpenAI:     CurrentModelDisplay = S->OpenAIModel; break;
		case EAIProviderType::OpenRouter:  CurrentModelDisplay = S->OpenRouterModel; break;
		case EAIProviderType::Custom:      CurrentModelDisplay = S->CustomModel; break;
		}
		if (CurrentModelDisplay.IsEmpty()) CurrentModelDisplay = TEXT("(none)");
		{
			TSharedPtr<FModelInfo> InitModel = MakeShared<FModelInfo>();
			InitModel->Id = CurrentModelDisplay;
			ModelOptions.Add(InitModel);
		}
	}

	ChildSlot
	[
		SNew(SVerticalBox)

		// ── Header bar ─────────────────────────────────────────────────
		+ SVerticalBox::Slot()
		.AutoHeight()
		.Padding(4)
		[
			SNew(SHorizontalBox)

			+ SHorizontalBox::Slot()
			.AutoWidth()
			.VAlign(VAlign_Center)
			.Padding(4, 0)
			[
				SNew(STextBlock)
				.Text(LOCTEXT("ProviderLabel", "Provider:"))
				.Font(FCoreStyle::GetDefaultFontStyle("Bold", 9))
			]

			+ SHorizontalBox::Slot()
			.AutoWidth()
			.Padding(4, 0)
			[
				SNew(SComboBox<TSharedPtr<FString>>)
				.OptionsSource(&ProviderOptions)
				.OnSelectionChanged(this, &SBlueprintAIChat::OnProviderChanged)
				.OnGenerateWidget_Lambda([](TSharedPtr<FString> Item)
				{
					return SNew(STextBlock).Text(FText::FromString(*Item));
				})
				[
					SNew(STextBlock)
					.Text_Lambda([this]()
					{
						UBlueprintAISettings* Settings = UBlueprintAISettings::Get();
						switch (Settings->ActiveProvider)
						{
						case EAIProviderType::Claude: return FText::FromString(TEXT("Claude"));
						case EAIProviderType::OpenAI: return FText::FromString(TEXT("OpenAI"));
						case EAIProviderType::OpenRouter: return FText::FromString(TEXT("OpenRouter"));
						case EAIProviderType::Custom: return FText::FromString(TEXT("Custom"));
						default: return FText::FromString(TEXT("Claude"));
						}
					})
				]
			]

			// ── Model selector ──
			+ SHorizontalBox::Slot()
			.AutoWidth()
			.VAlign(VAlign_Center)
			.Padding(12, 0, 4, 0)
			[
				SNew(STextBlock)
				.Text(LOCTEXT("ModelLabel", "Model:"))
				.Font(FCoreStyle::GetDefaultFontStyle("Bold", 9))
			]

			+ SHorizontalBox::Slot()
			.AutoWidth()
			.Padding(4, 0)
			[
				SAssignNew(ModelComboBox, SComboBox<TSharedPtr<FModelInfo>>)
				.OptionsSource(&ModelOptions)
				.OnSelectionChanged(this, &SBlueprintAIChat::OnModelChanged)
				.OnGenerateWidget_Lambda([](TSharedPtr<FModelInfo> Item)
				{
					FString Label = Item->GetDisplayString();
					FSlateColor TextColor = FSlateColor(FLinearColor::White);
					if (Item->bIsFree)
					{
						TextColor = FSlateColor(FLinearColor::Green);
					}
					return SNew(STextBlock)
						.Text(FText::FromString(Label))
						.ColorAndOpacity(TextColor);
				})
				[
					SNew(STextBlock)
					.Text_Lambda([this]()
					{
						return FText::FromString(CurrentModelDisplay);
					})
				]
			]

			+ SHorizontalBox::Slot()
			.AutoWidth()
			.VAlign(VAlign_Center)
			.Padding(2, 0)
			[
				SNew(SButton)
				.ToolTipText(LOCTEXT("RefreshModelsTooltip", "Refresh available models from provider"))
				.IsEnabled_Lambda([this]() { return !bFetchingModels; })
				.OnClicked_Lambda([this]()
				{
					RefreshModelList();
					return FReply::Handled();
				})
				.ContentPadding(FMargin(4, 2))
				[
					SNew(STextBlock)
					.Text(LOCTEXT("RefreshIcon", "\u21BB")) // ↻ circular arrow
					.Font(FCoreStyle::GetDefaultFontStyle("Regular", 12))
				]
			]

			+ SHorizontalBox::Slot()
			.FillWidth(1.0f)
			[
				SNullWidget::NullWidget
			]

			+ SHorizontalBox::Slot()
			.AutoWidth()
			.Padding(4, 0)
			[
				SNew(SButton)
				.Text(LOCTEXT("ClearBtn", "Clear"))
				.OnClicked_Lambda([this]()
				{
					OnClearConversation();
					return FReply::Handled();
				})
			]

			+ SHorizontalBox::Slot()
			.AutoWidth()
			.Padding(4, 0)
			[
				SNew(SButton)
				.Text(LOCTEXT("StopBtn", "Stop"))
				.ToolTipText(LOCTEXT("StopTooltip", "Stop the current AI generation"))
				.IsEnabled_Lambda([this]() { return bIsBusy; })
				.OnClicked_Lambda([this]()
				{
					OnStopGeneration();
					return FReply::Handled();
				})
			]

			+ SHorizontalBox::Slot()
			.AutoWidth()
			.Padding(4, 0)
			[
				SNew(SButton)
				.Text(LOCTEXT("DownloadBtn", "Save Chat"))
				.ToolTipText(LOCTEXT("DownloadTooltip", "Save entire chat log as .txt to your Desktop"))
				.OnClicked_Lambda([this]()
				{
					OnDownloadChat();
					return FReply::Handled();
				})
			]

			+ SHorizontalBox::Slot()
			.AutoWidth()
			.Padding(4, 0)
			[
				SNew(SButton)
				.Text(LOCTEXT("SettingsBtn", "Settings"))
				.OnClicked_Lambda([]()
				{
					// Open settings
					FModuleManager::LoadModuleChecked<ISettingsModule>("Settings")
						.ShowViewer("Project", "Plugins", "Blueprint AI Assistant");
					return FReply::Handled();
				})
			]
		]

		+ SVerticalBox::Slot()
		.AutoHeight()
		[
			SNew(SSeparator)
		]

		// ── Message list ───────────────────────────────────────────────
		+ SVerticalBox::Slot()
		.FillHeight(1.0f)
		.Padding(2)
		[
			SAssignNew(MessageListView, SListView<TSharedPtr<FChatMessageEntry>>)
			.ListItemsSource(&Messages)
			.OnGenerateRow(this, &SBlueprintAIChat::OnGenerateRow)
			.SelectionMode(ESelectionMode::None)
			.HeaderRow(
				SNew(SHeaderRow)
				+ SHeaderRow::Column(TEXT("Message"))
				.DefaultLabel(LOCTEXT("MessageCol", "Conversation"))
				.FillWidth(1.0f)
			)
		]

		+ SVerticalBox::Slot()
		.AutoHeight()
		[
			SNew(SSeparator)
		]

		// ── Input area ─────────────────────────────────────────────────
		+ SVerticalBox::Slot()
		.AutoHeight()
		.Padding(4)
		[
			SNew(SHorizontalBox)

			+ SHorizontalBox::Slot()
			.FillWidth(1.0f)
			.Padding(0, 0, 4, 0)
			[
				SNew(SBox)
				.MinDesiredHeight(40)
				.MaxDesiredHeight(120)
				[
					SAssignNew(InputTextBox, SMultiLineEditableTextBox)
					.HintText(LOCTEXT("InputHint", "Describe what Blueprint to create or modify..."))
					.OnKeyDownHandler_Lambda([this](const FGeometry&, const FKeyEvent& KeyEvent) -> FReply
					{
						// Enter (without Shift) sends the message
						if (KeyEvent.GetKey() == EKeys::Enter && !KeyEvent.IsShiftDown())
						{
							OnSendMessage();
							return FReply::Handled();
						}
						return FReply::Unhandled();
					})
				]
			]

			+ SHorizontalBox::Slot()
			.AutoWidth()
			.VAlign(VAlign_Bottom)
			[
				SAssignNew(SendButton, SButton)
				.Text(LOCTEXT("SendBtn", "Send"))
				.IsEnabled_Lambda([this]() { return !bIsBusy; })
				.OnClicked_Lambda([this]()
				{
					OnSendMessage();
					return FReply::Handled();
				})
			]
		]

		// ── Status bar ──────────────────────────────────────────────────
		+ SVerticalBox::Slot()
		.AutoHeight()
		.Padding(4, 2)
		[
			SNew(STextBlock)
			.Text_Lambda([this]()
			{
				FString Status;
				if (bIsBusy)
				{
					Status = TEXT("AI is thinking...");
				}
				else
				{
					Status = TEXT("Ready");
				}

				if (CurrentContextWindow > 0)
				{
					int32 TotalUsed = CurrentInputTokens + CurrentOutputTokens;
					float Pct = (float)TotalUsed / (float)CurrentContextWindow * 100.f;
					FString UsedStr, WindowStr;
					if (TotalUsed >= 1000) UsedStr = FString::Printf(TEXT("%.1fK"), TotalUsed / 1000.f);
					else UsedStr = FString::Printf(TEXT("%d"), TotalUsed);
					if (CurrentContextWindow >= 1000) WindowStr = FString::Printf(TEXT("%.0fK"), CurrentContextWindow / 1000.f);
					else WindowStr = FString::Printf(TEXT("%d"), CurrentContextWindow);
					Status += FString::Printf(TEXT(" | Context: %s / %s tokens (%.1f%%)"), *UsedStr, *WindowStr, Pct);
				}

				return FText::FromString(Status);
			})
			.Font(FCoreStyle::GetDefaultFontStyle("Italic", 8))
			.ColorAndOpacity_Lambda([this]() -> FSlateColor
			{
				if (CurrentContextWindow > 0)
				{
					int32 TotalUsed = CurrentInputTokens + CurrentOutputTokens;
					float Pct = (float)TotalUsed / (float)CurrentContextWindow;
					if (Pct > 0.8f) return FSlateColor(FLinearColor(0.9f, 0.2f, 0.2f)); // Red
					if (Pct > 0.5f) return FSlateColor(FLinearColor(0.9f, 0.7f, 0.1f)); // Yellow
					return FSlateColor(FLinearColor(0.3f, 0.8f, 0.3f)); // Green
				}
				return FSlateColor(FLinearColor(0.5f, 0.5f, 0.5f));
			})
		]

		// ── Disclaimer ──────────────────────────────────────────────────
		+ SVerticalBox::Slot()
		.AutoHeight()
		.Padding(4, 1)
		.HAlign(HAlign_Center)
		[
			SNew(STextBlock)
			.Text(FText::FromString(TEXT("AI can make mistakes. Always verify results and use the latest models.")))
			.Font(FCoreStyle::GetDefaultFontStyle("Italic", 7))
			.ColorAndOpacity(FSlateColor(FLinearColor(0.45f, 0.45f, 0.45f)))
			.Justification(ETextJustify::Center)
		]
	];

	// Add welcome message
	TSharedPtr<FChatMessageEntry> Welcome = MakeShared<FChatMessageEntry>();
	Welcome->Sender = TEXT("System");
	Welcome->Message = TEXT("Blueprint AI Assistant ready. Describe what you'd like to create or modify. "
		"Make sure your AI provider API key is set in Project Settings → Plugins → Blueprint AI Assistant.");
	Welcome->Timestamp = FDateTime::Now();
	Messages.Add(Welcome);
}

void SBlueprintAIChat::SetConversationManager(TSharedPtr<FConversationManager> InManager)
{
	ConversationManager = InManager;

	if (ConversationManager.IsValid())
	{
		ConversationManager->OnChatMessage.AddSP(this, &SBlueprintAIChat::OnNewChatMessage);
		ConversationManager->OnToolExecution.AddSP(this, &SBlueprintAIChat::OnToolExecutionUpdate);
		ConversationManager->OnBusyStateChanged.AddSP(this, &SBlueprintAIChat::OnBusyStateChanged);
		ConversationManager->OnContextUsageUpdated.AddSP(this, &SBlueprintAIChat::OnContextUsageUpdated);
	}
}

TSharedRef<ITableRow> SBlueprintAIChat::OnGenerateRow(
	TSharedPtr<FChatMessageEntry> Item,
	const TSharedRef<STableViewBase>& OwnerTable)
{
	return SNew(SChatMessageRow, OwnerTable)
		.MessageEntry(Item);
}

void SBlueprintAIChat::OnSendMessage()
{
	if (!InputTextBox.IsValid()) return;

	FString Text = InputTextBox->GetText().ToString().TrimStartAndEnd();
	if (Text.IsEmpty()) return;

	if (!ConversationManager.IsValid())
	{
		OnNewChatMessage(TEXT("Error"), TEXT("Conversation manager not initialized"));
		return;
	}

	InputTextBox->SetText(FText::GetEmpty());
	ConversationManager->SendUserMessage(Text);
}

void SBlueprintAIChat::OnClearConversation()
{
	Messages.Empty();

	if (ConversationManager.IsValid())
	{
		ConversationManager->ClearConversation();
	}

	if (MessageListView.IsValid())
	{
		MessageListView->RequestListRefresh();
	}
}

void SBlueprintAIChat::OnStopGeneration()
{
	if (ConversationManager.IsValid())
	{
		ConversationManager->StopGeneration();
	}
}

void SBlueprintAIChat::OnDownloadChat()
{
	if (Messages.Num() == 0)
	{
		OnNewChatMessage(TEXT("System"), TEXT("Nothing to save — chat is empty."));
		return;
	}

	// Build text content
	FString ChatText;
	ChatText += TEXT("=== Blueprint AI Assistant Chat Log ===\n");
	ChatText += FString::Printf(TEXT("Exported: %s\n"), *FDateTime::Now().ToString());
	ChatText += TEXT("========================================\n\n");

	for (const TSharedPtr<FChatMessageEntry>& Entry : Messages)
	{
		if (!Entry.IsValid()) continue;
		ChatText += FString::Printf(TEXT("[%s] [%s] %s\n"),
			*Entry->Timestamp.ToString(TEXT("%H:%M:%S")),
			*Entry->Sender,
			*Entry->Message);
	}

	// Save to Desktop
	FString UserHome = FPlatformMisc::GetEnvironmentVariable(TEXT("USERPROFILE"));
	if (UserHome.IsEmpty())
	{
		UserHome = FPlatformProcess::UserDir();
	}
	FString DesktopPath = FPaths::Combine(UserHome, TEXT("Desktop"));

	FString Timestamp = FDateTime::Now().ToString(TEXT("%Y%m%d_%H%M%S"));
	FString FileName = FString::Printf(TEXT("BlueprintAI_ChatLog_%s.txt"), *Timestamp);
	FString FullPath = FPaths::Combine(DesktopPath, FileName);

	if (FFileHelper::SaveStringToFile(ChatText, *FullPath, FFileHelper::EEncodingOptions::ForceUTF8WithoutBOM))
	{
		OnNewChatMessage(TEXT("System"), FString::Printf(TEXT("Chat saved to: %s"), *FullPath));
	}
	else
	{
		OnNewChatMessage(TEXT("Error"), FString::Printf(TEXT("Failed to save chat to: %s"), *FullPath));
	}
}

void SBlueprintAIChat::OnNewChatMessage(const FString& Sender, const FString& Message)
{
	TSharedPtr<FChatMessageEntry> Entry = MakeShared<FChatMessageEntry>();
	Entry->Sender = Sender;
	Entry->Message = Message;
	Entry->Timestamp = FDateTime::Now();
	Entry->bIsToolAction = false;

	Messages.Add(Entry);

	if (MessageListView.IsValid())
	{
		MessageListView->RequestListRefresh();
		ScrollToBottom();
	}
}

void SBlueprintAIChat::OnToolExecutionUpdate(const FString& ToolName, const FString& Status)
{
	TSharedPtr<FChatMessageEntry> Entry = MakeShared<FChatMessageEntry>();
	Entry->Sender = ToolName;
	Entry->Message = Status;
	Entry->Timestamp = FDateTime::Now();
	Entry->bIsToolAction = true;

	Messages.Add(Entry);

	if (MessageListView.IsValid())
	{
		MessageListView->RequestListRefresh();
		ScrollToBottom();
	}
}

void SBlueprintAIChat::OnBusyStateChanged(bool bNewBusy)
{
	bIsBusy = bNewBusy;
}

void SBlueprintAIChat::OnContextUsageUpdated(int32 InputUsed, int32 OutputUsed, int32 ContextWindow)
{
	CurrentInputTokens = InputUsed;
	CurrentOutputTokens = OutputUsed;
	CurrentContextWindow = ContextWindow;
}

void SBlueprintAIChat::OnProviderChanged(TSharedPtr<FString> NewValue, ESelectInfo::Type SelectionType)
{
	if (!NewValue.IsValid()) return;

	UBlueprintAISettings* Settings = UBlueprintAISettings::Get();

	if (*NewValue == TEXT("Claude")) Settings->ActiveProvider = EAIProviderType::Claude;
	else if (*NewValue == TEXT("OpenAI")) Settings->ActiveProvider = EAIProviderType::OpenAI;
	else if (*NewValue == TEXT("OpenRouter")) Settings->ActiveProvider = EAIProviderType::OpenRouter;
	else if (*NewValue == TEXT("Custom")) Settings->ActiveProvider = EAIProviderType::Custom;

	Settings->SaveConfig();

	// Refresh the provider
	FBlueprintAIAssistantModule::Get().RefreshAIProvider();

	// Update model display to match new provider
	switch (Settings->ActiveProvider)
	{
	case EAIProviderType::Claude:     CurrentModelDisplay = Settings->ClaudeModel; break;
	case EAIProviderType::OpenAI:     CurrentModelDisplay = Settings->OpenAIModel; break;
	case EAIProviderType::OpenRouter:  CurrentModelDisplay = Settings->OpenRouterModel; break;
	case EAIProviderType::Custom:      CurrentModelDisplay = Settings->CustomModel; break;
	}
	if (CurrentModelDisplay.IsEmpty()) CurrentModelDisplay = TEXT("(none)");

	// Reset model options to just the current model
	ModelOptions.Empty();
	{
		TSharedPtr<FModelInfo> InitModel = MakeShared<FModelInfo>();
		InitModel->Id = CurrentModelDisplay;
		ModelOptions.Add(InitModel);
	}
	if (ModelComboBox.IsValid())
	{
		ModelComboBox->RefreshOptions();
	}
}

void SBlueprintAIChat::OnModelChanged(TSharedPtr<FModelInfo> NewValue, ESelectInfo::Type SelectionType)
{
	if (!NewValue.IsValid()) return;

	UBlueprintAISettings* Settings = UBlueprintAISettings::Get();
	const FString& SelectedModel = NewValue->Id;

	// Update the model setting for the active provider
	switch (Settings->ActiveProvider)
	{
	case EAIProviderType::Claude:     Settings->ClaudeModel = SelectedModel; break;
	case EAIProviderType::OpenAI:     Settings->OpenAIModel = SelectedModel; break;
	case EAIProviderType::OpenRouter:  Settings->OpenRouterModel = SelectedModel; break;
	case EAIProviderType::Custom:      Settings->CustomModel = SelectedModel; break;
	}

	CurrentModelDisplay = NewValue->GetDisplayString();
	Settings->SaveConfig();

	// Pass context window to conversation manager
	if (ConversationManager.IsValid() && NewValue->ContextLength > 0)
	{
		ConversationManager->SetModelContextWindow(NewValue->ContextLength);
	}

	// Refresh the provider so it picks up the new model
	FBlueprintAIAssistantModule::Get().RefreshAIProvider();
}

void SBlueprintAIChat::RefreshModelList()
{
	TSharedPtr<IAIProvider> Provider = FBlueprintAIAssistantModule::Get().GetCurrentProvider();
	if (!Provider.IsValid())
	{
		OnNewChatMessage(TEXT("Error"), TEXT("No AI provider configured. Set your API key in Settings."));
		return;
	}

	bFetchingModels = true;
	OnNewChatMessage(TEXT("System"), TEXT("Fetching available models..."));

	TWeakPtr<SBlueprintAIChat> WeakSelf(SharedThis(this));

	FOnModelsReceived OnSuccess;
	OnSuccess.BindLambda([WeakSelf](const TArray<FModelInfo>& Models)
	{
		// Must run on game thread for Slate updates
		AsyncTask(ENamedThreads::GameThread, [WeakSelf, Models]()
		{
			TSharedPtr<SBlueprintAIChat> This = WeakSelf.Pin();
			if (!This.IsValid()) return;

			This->bFetchingModels = false;
			This->ModelOptions.Empty();

			if (Models.Num() == 0)
			{
				FModelInfo Placeholder;
				Placeholder.Id = This->CurrentModelDisplay;
				Placeholder.DisplayName = This->CurrentModelDisplay;
				This->ModelOptions.Add(MakeShared<FModelInfo>(Placeholder));
				This->OnNewChatMessage(TEXT("System"), TEXT("No models returned by provider."));
			}
			else
			{
				int32 FreeCount = 0;
				for (const FModelInfo& M : Models)
				{
					This->ModelOptions.Add(MakeShared<FModelInfo>(M));
					if (M.bIsFree) FreeCount++;
				}
				This->OnNewChatMessage(TEXT("System"),
					FString::Printf(TEXT("Found %d models (%d free). Select one from the dropdown."), Models.Num(), FreeCount));
			}

			if (This->ModelComboBox.IsValid())
			{
				This->ModelComboBox->RefreshOptions();
			}
		});
	});

	FOnAIError OnError;
	OnError.BindLambda([WeakSelf](const FString& Error)
	{
		AsyncTask(ENamedThreads::GameThread, [WeakSelf, Error]()
		{
			TSharedPtr<SBlueprintAIChat> This = WeakSelf.Pin();
			if (!This.IsValid()) return;

			This->bFetchingModels = false;
			This->OnNewChatMessage(TEXT("Error"),
				FString::Printf(TEXT("Failed to fetch models: %s"), *Error.Left(300)));
		});
	});

	Provider->FetchAvailableModels(OnSuccess, OnError);
}

void SBlueprintAIChat::ScrollToBottom()
{
	if (MessageListView.IsValid() && Messages.Num() > 0)
	{
		MessageListView->RequestScrollIntoView(Messages.Last());
	}
}

#undef LOCTEXT_NAMESPACE
