// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#include "Memory/BlueprintAIMemory.h"
#include "Misc/FileHelper.h"
#include "Misc/Paths.h"
#include "HAL/PlatformFileManager.h"
#include "Serialization/JsonSerializer.h"
#include "Serialization/JsonWriter.h"
#include "Dom/JsonObject.h"
#include "Dom/JsonValue.h"

FBlueprintAIMemory::FBlueprintAIMemory()
{
}

FBlueprintAIMemory& FBlueprintAIMemory::Get()
{
	static FBlueprintAIMemory Instance;
	if (!Instance.bLoaded)
	{
		Instance.Load();
	}
	return Instance;
}

FString FBlueprintAIMemory::GetMemoryFilePath() const
{
	return FPaths::Combine(FPaths::ProjectSavedDir(), TEXT("BlueprintAI"), TEXT("memories.json"));
}

void FBlueprintAIMemory::Load()
{
	bLoaded = true;
	Memories.Empty();
	NextMemoryId = 1;

	FString FilePath = GetMemoryFilePath();
	FString JsonStr;

	if (!FFileHelper::LoadFileToString(JsonStr, *FilePath))
	{
		// No file yet — that's fine, start fresh
		return;
	}

	TSharedRef<TJsonReader<>> Reader = TJsonReaderFactory<>::Create(JsonStr);
	TSharedPtr<FJsonObject> Root;
	if (!FJsonSerializer::Deserialize(Reader, Root) || !Root.IsValid())
	{
		UE_LOG(LogTemp, Warning, TEXT("BlueprintAIMemory: Failed to parse memories.json"));
		return;
	}

	const TArray<TSharedPtr<FJsonValue>>* EntriesArr;
	if (Root->TryGetArrayField(TEXT("memories"), EntriesArr))
	{
		for (const TSharedPtr<FJsonValue>& EntryVal : *EntriesArr)
		{
			const TSharedPtr<FJsonObject>* EntryObj;
			if (EntryVal->TryGetObject(EntryObj))
			{
				FMemoryEntry Entry;
				Entry.Id = (*EntryObj)->GetStringField(TEXT("id"));
				Entry.Content = (*EntryObj)->GetStringField(TEXT("content"));

				FString DateStr;
				if ((*EntryObj)->TryGetStringField(TEXT("created_at"), DateStr))
				{
					FDateTime::ParseIso8601(*DateStr, Entry.CreatedAt);
				}
				else
				{
					Entry.CreatedAt = FDateTime::Now();
				}

				Memories.Add(Entry);

				// Track the highest numeric ID to avoid collisions
				FString IdNum = Entry.Id;
				IdNum.RemoveFromStart(TEXT("M"));
				int32 Num = FCString::Atoi(*IdNum);
				if (Num >= NextMemoryId)
				{
					NextMemoryId = Num + 1;
				}
			}
		}
	}

	UE_LOG(LogTemp, Log, TEXT("BlueprintAIMemory: Loaded %d memories from %s"), Memories.Num(), *FilePath);
}

void FBlueprintAIMemory::Save()
{
	FString FilePath = GetMemoryFilePath();

	// Ensure directory exists
	FString Dir = FPaths::GetPath(FilePath);
	IPlatformFile& PlatformFile = FPlatformFileManager::Get().GetPlatformFile();
	if (!PlatformFile.DirectoryExists(*Dir))
	{
		PlatformFile.CreateDirectoryTree(*Dir);
	}

	// Build JSON
	TSharedPtr<FJsonObject> Root = MakeShared<FJsonObject>();
	TArray<TSharedPtr<FJsonValue>> EntriesArr;

	for (const FMemoryEntry& Entry : Memories)
	{
		TSharedPtr<FJsonObject> EntryObj = MakeShared<FJsonObject>();
		EntryObj->SetStringField(TEXT("id"), Entry.Id);
		EntryObj->SetStringField(TEXT("content"), Entry.Content);
		EntryObj->SetStringField(TEXT("created_at"), Entry.CreatedAt.ToIso8601());
		EntriesArr.Add(MakeShared<FJsonValueObject>(EntryObj));
	}

	Root->SetArrayField(TEXT("memories"), EntriesArr);

	FString JsonStr;
	TSharedRef<TJsonWriter<>> Writer = TJsonWriterFactory<>::Create(&JsonStr);
	FJsonSerializer::Serialize(Root.ToSharedRef(), Writer);

	if (FFileHelper::SaveStringToFile(JsonStr, *FilePath))
	{
		UE_LOG(LogTemp, Log, TEXT("BlueprintAIMemory: Saved %d memories to %s"), Memories.Num(), *FilePath);
	}
	else
	{
		UE_LOG(LogTemp, Error, TEXT("BlueprintAIMemory: Failed to save memories to %s"), *FilePath);
	}
}

FString FBlueprintAIMemory::AddMemory(const FString& Content)
{
	FMemoryEntry NewEntry;
	NewEntry.Id = FString::Printf(TEXT("M%d"), NextMemoryId++);
	NewEntry.Content = Content;
	NewEntry.CreatedAt = FDateTime::Now();

	Memories.Add(NewEntry);
	Save();

	UE_LOG(LogTemp, Log, TEXT("BlueprintAIMemory: Added memory %s: %s"), *NewEntry.Id, *Content.Left(100));
	return NewEntry.Id;
}

bool FBlueprintAIMemory::RemoveMemory(const FString& Id)
{
	for (int32 i = 0; i < Memories.Num(); i++)
	{
		if (Memories[i].Id.Equals(Id, ESearchCase::IgnoreCase))
		{
			UE_LOG(LogTemp, Log, TEXT("BlueprintAIMemory: Removed memory %s"), *Id);
			Memories.RemoveAt(i);
			Save();
			return true;
		}
	}
	return false;
}

FString FBlueprintAIMemory::BuildMemoryPromptSection() const
{
	if (Memories.Num() == 0)
	{
		return FString();
	}

	FString Section;
	Section += TEXT("=== AI MEMORY (persistent across sessions) ===\n");
	for (const FMemoryEntry& Entry : Memories)
	{
		Section += FString::Printf(TEXT("[%s] %s\n"), *Entry.Id, *Entry.Content);
	}
	Section += TEXT("\nYou can save new memories with save_memory or remove old ones with remove_memory.\n\n");
	return Section;
}
