// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#include "Logging/BlueprintAILogger.h"
#include "Settings/BlueprintAISettings.h"
#include "Misc/DateTime.h"
#include "Misc/Paths.h"
#include "Misc/FileHelper.h"
#include "HAL/PlatformProcess.h"
#include "HAL/FileManager.h"

FBlueprintAILogger& FBlueprintAILogger::Get()
{
	static FBlueprintAILogger Instance;
	return Instance;
}

FBlueprintAILogger::~FBlueprintAILogger()
{
	Shutdown();
}

void FBlueprintAILogger::Initialize()
{
	if (bInitialized) return;

	const UBlueprintAISettings* Settings = UBlueprintAISettings::Get();
	if (!Settings || !Settings->bEnableFileLogging)
	{
		return;
	}

	// Build path: user Desktop + timestamped filename
	// FPlatformProcess::UserDir() returns something like C:\Users\name\Documents/
	// We want the Desktop folder which is a sibling to Documents.
	FString DesktopPath = FPlatformProcess::UserDir();
	// Remove trailing slashes and go up one directory from Documents
	FPaths::NormalizeDirectoryName(DesktopPath);
	DesktopPath = FPaths::GetPath(DesktopPath) / TEXT("Desktop");

	// Fallback: if Desktop doesn't exist, use project Saved folder
	if (!FPaths::DirectoryExists(DesktopPath))
	{
		DesktopPath = FPaths::ProjectSavedDir() / TEXT("Logs");
	}

	const FString Timestamp = FDateTime::Now().ToString(TEXT("%Y%m%d_%H%M%S"));
	LogFilePath = DesktopPath / FString::Printf(TEXT("BlueprintAI_FullLog_%s.txt"), *Timestamp);

	// Ensure directory exists
	IFileManager::Get().MakeDirectory(*FPaths::GetPath(LogFilePath), true);

	FileWriter.Reset(IFileManager::Get().CreateFileWriter(*LogFilePath, FILEWRITE_Append));
	if (!FileWriter.IsValid())
	{
		UE_LOG(LogTemp, Error, TEXT("BlueprintAI Logger: Failed to create log file: %s"), *LogFilePath);
		return;
	}

	bInitialized = true;

	// Write header
	Log(TEXT("INIT"), FString::Printf(TEXT("=== Blueprint AI Assistant Full Debug Log ===")));
	Log(TEXT("INIT"), FString::Printf(TEXT("Session started: %s"), *FDateTime::Now().ToString()));
	Log(TEXT("INIT"), FString::Printf(TEXT("Log file: %s"), *LogFilePath));
	Log(TEXT("INIT"), FString::Printf(TEXT("Engine version: %s"), *FEngineVersion::Current().ToString()));
	Log(TEXT("INIT"), FString::Printf(TEXT("Project: %s"), *FPaths::GetProjectFilePath()));

	if (Settings)
	{
		FString ProviderName;
		switch (Settings->ActiveProvider)
		{
		case EAIProviderType::Claude:     ProviderName = TEXT("Claude"); break;
		case EAIProviderType::OpenAI:     ProviderName = TEXT("OpenAI"); break;
		case EAIProviderType::OpenRouter:  ProviderName = TEXT("OpenRouter"); break;
		case EAIProviderType::Custom:      ProviderName = TEXT("Custom"); break;
		}
		Log(TEXT("INIT"), FString::Printf(TEXT("Active provider: %s"), *ProviderName));
		Log(TEXT("INIT"), FString::Printf(TEXT("Temperature: %.2f | MaxTokens: %d | MaxIterations: %d"),
			Settings->Temperature, Settings->MaxTokens, Settings->MaxToolIterations));
		Log(TEXT("INIT"), FString::Printf(TEXT("MaxMessageChars: %d | MaxTotalPayloadChars: %d"),
			Settings->MaxMessageChars, Settings->MaxTotalPayloadChars));
		Log(TEXT("INIT"), FString::Printf(TEXT("HttpRequestTimeoutSeconds: %d"),
			Settings->HttpRequestTimeoutSeconds));
	}
	Log(TEXT("INIT"), TEXT("========================================================"));
	Log(TEXT("INIT"), TEXT(""));

	UE_LOG(LogTemp, Log, TEXT("BlueprintAI Logger: Logging to %s"), *LogFilePath);
}

void FBlueprintAILogger::Shutdown()
{
	if (!bInitialized) return;

	Log(TEXT("SHUTDOWN"), TEXT("Session ended."));
	Log(TEXT("SHUTDOWN"), FString::Printf(TEXT("=== End of log: %s ==="), *FDateTime::Now().ToString()));

	FScopeLock Lock(&WriteLock);
	if (FileWriter.IsValid())
	{
		FileWriter->Flush();
		FileWriter->Close();
		FileWriter.Reset();
	}

	bInitialized = false;
}

void FBlueprintAILogger::Log(const FString& Category, const FString& Message)
{
	if (!bInitialized || !FileWriter.IsValid()) return;

	const FString Line = FString::Printf(TEXT("[%s] [%s] %s\r\n"),
		*GetTimestamp(), *Category, *Message);

	// Convert to UTF-8 for file writing
	FTCHARToUTF8 Utf8Line(*Line);

	FScopeLock Lock(&WriteLock);
	FileWriter->Serialize(const_cast<char*>(Utf8Line.Get()), Utf8Line.Length());
	FileWriter->Flush(); // Flush after every write so we don't lose data on crash
}

void FBlueprintAILogger::Logf(const FString& Category, const TCHAR* Fmt, ...)
{
	if (!bInitialized) return;

	TCHAR Buffer[4096];
	va_list Args;
	va_start(Args, Fmt);
	FCString::GetVarArgs(Buffer, UE_ARRAY_COUNT(Buffer), Fmt, Args);
	va_end(Args);

	Log(Category, FString(Buffer));
}

bool FBlueprintAILogger::IsEnabled() const
{
	return bInitialized && FileWriter.IsValid();
}

FString FBlueprintAILogger::GetTimestamp() const
{
	return FDateTime::Now().ToString(TEXT("%H:%M:%S.%s"));
}
