// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#include "BlueprintAIAssistantModule.h"

#include "AI/ConversationManager.h"
#include "AI/ClaudeProvider.h"
#include "AI/OpenAIProvider.h"
#include "AI/OpenRouterProvider.h"
#include "Logging/BlueprintAILogger.h"
#include "AI/CustomProvider.h"
#include "Actions/ActionRegistry.h"
#include "Settings/BlueprintAISettings.h"
#include "UI/SBlueprintAIChat.h"

#include "Framework/Docking/TabManager.h"
#include "Widgets/Docking/SDockTab.h"
#include "ToolMenus.h"
#include "WorkspaceMenuStructure.h"
#include "WorkspaceMenuStructureModule.h"
#include "Styling/AppStyle.h"

#define LOCTEXT_NAMESPACE "BlueprintAIAssistant"

const FName FBlueprintAIAssistantModule::AIAssistantTabId(TEXT("BlueprintAIAssistantTab"));

// ─── Module Lifecycle ────────────────────────────────────────────────────────

void FBlueprintAIAssistantModule::StartupModule()
{
	// Initialize the debug file logger (creates timestamped log on Desktop)
	FBlueprintAILogger::Get().Initialize();

	// Initialize core systems
	ActionRegistry = MakeShared<FActionRegistry>();
	ConversationManager = MakeShared<FConversationManager>(ActionRegistry);

	// Create the default AI provider based on settings
	RefreshAIProvider();

	// Register the editor tab
	RegisterTab();

	// Register the toolbar/menu buttons (deferred to allow ToolMenus module to initialize)
	UToolMenus::RegisterStartupCallback(FSimpleMulticastDelegate::FDelegate::CreateRaw(this, &FBlueprintAIAssistantModule::RegisterMenus));
}

void FBlueprintAIAssistantModule::ShutdownModule()
{
	UToolMenus::UnRegisterStartupCallback(this);
	UToolMenus::UnregisterOwner(this);

	UnregisterTab();

	ConversationManager.Reset();
	ActionRegistry.Reset();
	CurrentProvider.Reset();

	// Shut down the file logger last
	FBlueprintAILogger::Get().Shutdown();
}

FBlueprintAIAssistantModule& FBlueprintAIAssistantModule::Get()
{
	return FModuleManager::GetModuleChecked<FBlueprintAIAssistantModule>(TEXT("BlueprintAIAssistant"));
}

// ─── AI Provider Management ─────────────────────────────────────────────────

void FBlueprintAIAssistantModule::RefreshAIProvider()
{
	const UBlueprintAISettings* Settings = UBlueprintAISettings::Get();

	switch (Settings->ActiveProvider)
	{
	case EAIProviderType::Claude:
		CurrentProvider = MakeShared<FClaudeProvider>(
			Settings->ClaudeApiKey,
			Settings->ClaudeModel,
			Settings->ClaudeApiVersion
		);
		break;

	case EAIProviderType::OpenAI:
		CurrentProvider = MakeShared<FOpenAIProvider>(
			Settings->OpenAIApiKey,
			Settings->OpenAIModel,
			Settings->OpenAIEndpoint
		);
		break;

	case EAIProviderType::OpenRouter:
		CurrentProvider = MakeShared<FOpenRouterProvider>(
			Settings->OpenRouterApiKey,
			Settings->OpenRouterModel
		);
		break;

	case EAIProviderType::Custom:
		CurrentProvider = MakeShared<FCustomProvider>(
			Settings->CustomEndpoint,
			Settings->CustomApiKey,
			Settings->CustomModel,
			Settings->CustomApiFormat
		);
		break;
	}

	if (ConversationManager.IsValid())
	{
		ConversationManager->SetProvider(CurrentProvider);
	}
}

// ─── Tab Registration ────────────────────────────────────────────────────────

void FBlueprintAIAssistantModule::RegisterTab()
{
	FGlobalTabmanager::Get()->RegisterNomadTabSpawner(
		AIAssistantTabId,
		FOnSpawnTab::CreateRaw(this, &FBlueprintAIAssistantModule::SpawnTab))
		.SetDisplayName(LOCTEXT("TabTitle", "Blueprint AI Assistant"))
		.SetTooltipText(LOCTEXT("TabTooltip", "AI-powered Blueprint creation and manipulation"))
		.SetGroup(WorkspaceMenu::GetMenuStructure().GetToolsCategory())
		.SetIcon(FSlateIcon(FAppStyle::GetAppStyleSetName(), "Icons.Help"));
}

void FBlueprintAIAssistantModule::UnregisterTab()
{
	FGlobalTabmanager::Get()->UnregisterNomadTabSpawner(AIAssistantTabId);
}

TSharedRef<SDockTab> FBlueprintAIAssistantModule::SpawnTab(const FSpawnTabArgs& Args)
{
	TSharedRef<SBlueprintAIChat> ChatWidget = SNew(SBlueprintAIChat);
	ChatWidget->SetConversationManager(ConversationManager);

	return SNew(SDockTab)
		.TabRole(ETabRole::NomadTab)
		.Label(LOCTEXT("TabLabel", "AI Blueprint Assistant"))
		[
			ChatWidget
		];
}

// ─── Menu/Toolbar Registration ───────────────────────────────────────────────

void FBlueprintAIAssistantModule::RegisterMenus()
{
	FToolMenuOwnerScoped OwnerScoped(this);

	// Add to the Tools menu in the Level Editor
	UToolMenu* ToolsMenu = UToolMenus::Get()->ExtendMenu("LevelEditor.MainMenu.Tools");
	FToolMenuSection& Section = ToolsMenu->FindOrAddSection("BlueprintAI");

	Section.AddMenuEntry(
		"OpenBlueprintAIAssistant",
		LOCTEXT("MenuLabel", "Blueprint AI Assistant"),
		LOCTEXT("MenuTooltip", "Open the AI-powered Blueprint creation assistant"),
		FSlateIcon(FAppStyle::GetAppStyleSetName(), "Icons.Help"),
		FUIAction(FExecuteAction::CreateLambda([]()
		{
			FGlobalTabmanager::Get()->TryInvokeTab(AIAssistantTabId);
		}))
	);
}

#undef LOCTEXT_NAMESPACE

IMPLEMENT_MODULE(FBlueprintAIAssistantModule, BlueprintAIAssistant)
