// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#include "Actions/SetVariableDefaultsAction.h"

#include "Engine/Blueprint.h"
#include "Kismet2/BlueprintEditorUtils.h"
#include "AssetRegistry/AssetRegistryModule.h"

static UBlueprint* FindBPForVarDefaults(const FString& NameOrPath)
{
	if (NameOrPath.StartsWith(TEXT("/")))
	{
		return LoadObject<UBlueprint>(nullptr, *NameOrPath);
	}
	FAssetRegistryModule& AssetReg = FModuleManager::LoadModuleChecked<FAssetRegistryModule>("AssetRegistry");
	TArray<FAssetData> Assets;
	AssetReg.Get().GetAssetsByClass(UBlueprint::StaticClass()->GetClassPathName(), Assets);
	for (const FAssetData& Asset : Assets)
	{
		if (Asset.AssetName.ToString().Equals(NameOrPath, ESearchCase::IgnoreCase))
		{
			return Cast<UBlueprint>(Asset.GetAsset());
		}
	}
	return nullptr;
}

TSharedPtr<FJsonObject> FSetVariableDefaultsAction::GetParameterSchema() const
{
	auto MakeStr = [](const FString& Desc) -> TSharedPtr<FJsonObject> {
		TSharedPtr<FJsonObject> P = MakeShared<FJsonObject>();
		P->SetStringField(TEXT("type"), TEXT("string"));
		P->SetStringField(TEXT("description"), Desc);
		return P;
	};
	auto MakeBool = [](const FString& Desc) -> TSharedPtr<FJsonObject> {
		TSharedPtr<FJsonObject> P = MakeShared<FJsonObject>();
		P->SetStringField(TEXT("type"), TEXT("boolean"));
		P->SetStringField(TEXT("description"), Desc);
		return P;
	};

	TSharedPtr<FJsonObject> Schema = MakeShared<FJsonObject>();
	Schema->SetStringField(TEXT("type"), TEXT("object"));

	TSharedPtr<FJsonObject> Props = MakeShared<FJsonObject>();
	Props->SetObjectField(TEXT("blueprint"), MakeStr(TEXT("Blueprint name or content path")));
	Props->SetObjectField(TEXT("variable_name"), MakeStr(TEXT("Variable name")));
	Props->SetObjectField(TEXT("default_value"), MakeStr(TEXT("Default value as string (e.g. '100', 'true', 'Hello')")));
	Props->SetObjectField(TEXT("expose_on_spawn"), MakeBool(TEXT("If true, expose this variable so it can be set when spawning the actor")));
	Props->SetObjectField(TEXT("read_only"), MakeBool(TEXT("If true, mark the variable as Blueprint Read Only")));

	Schema->SetObjectField(TEXT("properties"), Props);

	TArray<TSharedPtr<FJsonValue>> Required;
	Required.Add(MakeShared<FJsonValueString>(TEXT("blueprint")));
	Required.Add(MakeShared<FJsonValueString>(TEXT("variable_name")));
	Schema->SetArrayField(TEXT("required"), Required);

	return Schema;
}

FBlueprintActionResult FSetVariableDefaultsAction::Execute(const TSharedPtr<FJsonObject>& Params)
{
	const FString BPName = Params->GetStringField(TEXT("blueprint"));
	const FString VarName = Params->GetStringField(TEXT("variable_name"));

	UBlueprint* BP = FindBPForVarDefaults(BPName);
	if (!BP)
	{
		return FBlueprintActionResult::Failure(FString::Printf(TEXT("Blueprint '%s' not found"), *BPName));
	}

	// Find the variable
	FBPVariableDescription* FoundVar = nullptr;
	for (FBPVariableDescription& Var : BP->NewVariables)
	{
		if (Var.VarName == FName(*VarName))
		{
			FoundVar = &Var;
			break;
		}
	}

	if (!FoundVar)
	{
		return FBlueprintActionResult::Failure(
			FString::Printf(TEXT("Variable '%s' not found in Blueprint '%s'"), *VarName, *BPName));
	}

	TSharedPtr<FJsonObject> Data = MakeShared<FJsonObject>();
	Data->SetStringField(TEXT("blueprint"), BPName);
	Data->SetStringField(TEXT("variable_name"), VarName);

	// Set default value
	if (Params->HasField(TEXT("default_value")))
	{
		FString DefaultVal = Params->GetStringField(TEXT("default_value"));
		FoundVar->DefaultValue = DefaultVal;
		Data->SetStringField(TEXT("default_value"), DefaultVal);
	}

	// Expose on spawn
	if (Params->HasField(TEXT("expose_on_spawn")))
	{
		bool bExposeOnSpawn = Params->GetBoolField(TEXT("expose_on_spawn"));
		if (bExposeOnSpawn)
		{
			FoundVar->PropertyFlags |= CPF_ExposeOnSpawn;
		}
		else
		{
			FoundVar->PropertyFlags &= ~CPF_ExposeOnSpawn;
		}
		Data->SetBoolField(TEXT("expose_on_spawn"), bExposeOnSpawn);
	}

	// Read only
	if (Params->HasField(TEXT("read_only")))
	{
		bool bReadOnly = Params->GetBoolField(TEXT("read_only"));
		if (bReadOnly)
		{
			FoundVar->PropertyFlags |= CPF_BlueprintReadOnly;
		}
		else
		{
			FoundVar->PropertyFlags &= ~CPF_BlueprintReadOnly;
			FoundVar->PropertyFlags |= CPF_BlueprintVisible;
		}
		Data->SetBoolField(TEXT("read_only"), bReadOnly);
	}

	FBlueprintEditorUtils::MarkBlueprintAsStructurallyModified(BP);

	return FBlueprintActionResult::Success(
		FString::Printf(TEXT("Updated variable '%s' in '%s'"), *VarName, *BPName),
		Data);
}
