// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#include "Actions/SetPinValueAction.h"
#include "Actions/ActionRegistry.h"

#include "EdGraph/EdGraphNode.h"
#include "EdGraph/EdGraphPin.h"
#include "EdGraphSchema_K2.h"

TSharedPtr<FJsonObject> FSetPinValueAction::GetParameterSchema() const
{
	auto MakeStr = [](const FString& Desc) -> TSharedPtr<FJsonObject> {
		TSharedPtr<FJsonObject> P = MakeShared<FJsonObject>();
		P->SetStringField(TEXT("type"), TEXT("string"));
		P->SetStringField(TEXT("description"), Desc);
		return P;
	};

	TSharedPtr<FJsonObject> Schema = MakeShared<FJsonObject>();
	Schema->SetStringField(TEXT("type"), TEXT("object"));

	TSharedPtr<FJsonObject> Props = MakeShared<FJsonObject>();
	Props->SetObjectField(TEXT("node_id"), MakeStr(TEXT("The tracked node ID (e.g. 'N1')")));
	Props->SetObjectField(TEXT("pin_name"), MakeStr(TEXT("Exact name of the pin to set")));
	Props->SetObjectField(TEXT("value"), MakeStr(TEXT("The default value as a string (e.g. '42', 'true', 'Hello World', '1.0,2.0,3.0' for vectors)")));

	Schema->SetObjectField(TEXT("properties"), Props);

	TArray<TSharedPtr<FJsonValue>> Required;
	Required.Add(MakeShared<FJsonValueString>(TEXT("node_id")));
	Required.Add(MakeShared<FJsonValueString>(TEXT("pin_name")));
	Required.Add(MakeShared<FJsonValueString>(TEXT("value")));
	Schema->SetArrayField(TEXT("required"), Required);

	return Schema;
}

FBlueprintActionResult FSetPinValueAction::Execute(const TSharedPtr<FJsonObject>& Params)
{
	const FString NodeId = Params->GetStringField(TEXT("node_id"));
	const FString PinName = Params->GetStringField(TEXT("pin_name"));
	const FString Value = Params->GetStringField(TEXT("value"));

	UEdGraphNode* Node = Registry->FindTrackedNode(NodeId);
	if (!Node)
	{
		return FBlueprintActionResult::Failure(
			FString::Printf(TEXT("Node '%s' not found or no longer valid"), *NodeId));
	}

	// Find the pin (with comprehensive fuzzy matching)
	UEdGraphPin* TargetPin = nullptr;

	// 1. Exact match (case-insensitive)
	for (UEdGraphPin* Pin : Node->Pins)
	{
		if (Pin->Direction == EGPD_Input && !Pin->bHidden &&
			Pin->PinName.ToString().Equals(PinName, ESearchCase::IgnoreCase))
		{
			TargetPin = Pin;
			break;
		}
	}

	// 2. Normalized: strip spaces and underscores
	if (!TargetPin)
	{
		FString Normalized = PinName.ToLower().Replace(TEXT(" "), TEXT("")).Replace(TEXT("_"), TEXT(""));
		for (UEdGraphPin* Pin : Node->Pins)
		{
			if (Pin->Direction == EGPD_Input && !Pin->bHidden)
			{
				FString CandidateNorm = Pin->PinName.ToString().ToLower().Replace(TEXT(" "), TEXT("")).Replace(TEXT("_"), TEXT(""));
				if (CandidateNorm == Normalized)
				{
					TargetPin = Pin;
					break;
				}
			}
		}
	}

	// 3. Boolean prefix fallback: 'Looping' matches 'bLooping', etc.
	if (!TargetPin)
	{
		for (UEdGraphPin* Pin : Node->Pins)
		{
			if (Pin->Direction == EGPD_Input && !Pin->bHidden)
			{
				FString CandidateName = Pin->PinName.ToString();
				if (CandidateName.Equals(TEXT("b") + PinName, ESearchCase::IgnoreCase))
				{
					TargetPin = Pin;
					break;
				}
				if (PinName.StartsWith(TEXT("b"), ESearchCase::CaseSensitive) && PinName.Len() > 1)
				{
					if (CandidateName.Equals(PinName.Mid(1), ESearchCase::IgnoreCase))
					{
						TargetPin = Pin;
						break;
					}
				}
			}
		}
	}

	// 4. Single non-exec input pin fallback
	if (!TargetPin)
	{
		UEdGraphPin* OnlyCandidate = nullptr;
		int32 Count = 0;
		for (UEdGraphPin* Pin : Node->Pins)
		{
			if (Pin->Direction == EGPD_Input && !Pin->bHidden &&
				Pin->PinType.PinCategory != UEdGraphSchema_K2::PC_Exec)
			{
				OnlyCandidate = Pin;
				Count++;
			}
		}
		if (Count == 1 && OnlyCandidate)
		{
			TargetPin = OnlyCandidate;
		}
	}

	if (!TargetPin)
	{
		// List available pins for error message — include node title so AI knows what node this ID points to
		FString AvailablePins;
		for (UEdGraphPin* Pin : Node->Pins)
		{
			if (Pin->Direction == EGPD_Input && !Pin->bHidden)
			{
				if (!AvailablePins.IsEmpty()) AvailablePins += TEXT(", ");
				AvailablePins += Pin->PinName.ToString();
			}
		}
		FString NodeTitle = Node->GetNodeTitle(ENodeTitleType::FullTitle).ToString();
		return FBlueprintActionResult::Failure(
			FString::Printf(TEXT("Pin '%s' not found on node '%s' (\"%s\"). Available input pins: %s"),
				*PinName, *NodeId, *NodeTitle, *AvailablePins));
	}

	// Set the default value
	const UEdGraphSchema_K2* Schema = GetDefault<UEdGraphSchema_K2>();
	Schema->TrySetDefaultValue(*TargetPin, Value);

	TSharedPtr<FJsonObject> Data = MakeShared<FJsonObject>();
	Data->SetStringField(TEXT("node_id"), NodeId);
	Data->SetStringField(TEXT("pin_name"), PinName);
	Data->SetStringField(TEXT("value_set"), Value);
	Data->SetStringField(TEXT("actual_default"), TargetPin->DefaultValue);

	return FBlueprintActionResult::Success(
		FString::Printf(TEXT("Set pin '%s' on node '%s' to '%s'"), *PinName, *NodeId, *Value),
		Data);
}
