// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#include "Actions/SetNodeCommentAction.h"
#include "Actions/ActionRegistry.h"

#include "EdGraph/EdGraphNode.h"
#include "Kismet2/BlueprintEditorUtils.h"
#include "Engine/Blueprint.h"

TSharedPtr<FJsonObject> FSetNodeCommentAction::GetParameterSchema() const
{
	auto MakeStr = [](const FString& Desc) -> TSharedPtr<FJsonObject> {
		TSharedPtr<FJsonObject> P = MakeShared<FJsonObject>();
		P->SetStringField(TEXT("type"), TEXT("string"));
		P->SetStringField(TEXT("description"), Desc);
		return P;
	};

	TSharedPtr<FJsonObject> Schema = MakeShared<FJsonObject>();
	Schema->SetStringField(TEXT("type"), TEXT("object"));

	TSharedPtr<FJsonObject> Props = MakeShared<FJsonObject>();
	Props->SetObjectField(TEXT("node_id"), MakeStr(TEXT("The tracked node ID (e.g. 'N1')")));
	Props->SetObjectField(TEXT("comment"), MakeStr(TEXT("Comment text to set on the node. Pass empty string to clear.")));

	Schema->SetObjectField(TEXT("properties"), Props);

	TArray<TSharedPtr<FJsonValue>> Required;
	Required.Add(MakeShared<FJsonValueString>(TEXT("node_id")));
	Required.Add(MakeShared<FJsonValueString>(TEXT("comment")));
	Schema->SetArrayField(TEXT("required"), Required);

	return Schema;
}

FBlueprintActionResult FSetNodeCommentAction::Execute(const TSharedPtr<FJsonObject>& Params)
{
	const FString NodeId = Params->GetStringField(TEXT("node_id"));
	const FString Comment = Params->GetStringField(TEXT("comment"));

	if (NodeId.IsEmpty())
	{
		return FBlueprintActionResult::Failure(TEXT("node_id is required"));
	}

	UEdGraphNode* Node = Registry->FindTrackedNode(NodeId);
	if (!Node)
	{
		return FBlueprintActionResult::Failure(
			FString::Printf(TEXT("Node '%s' not found. Use get_blueprint_info or get_selected_nodes to discover node IDs."), *NodeId));
	}

	// Set the comment
	Node->NodeComment = Comment;
	Node->bCommentBubblePinned = !Comment.IsEmpty();
	Node->bCommentBubbleVisible = !Comment.IsEmpty();

	// Mark the owning Blueprint as modified
	if (UEdGraph* Graph = Node->GetGraph())
	{
		if (UBlueprint* BP = FBlueprintEditorUtils::FindBlueprintForGraph(Graph))
		{
			FBlueprintEditorUtils::MarkBlueprintAsModified(BP);
		}
		Graph->NotifyGraphChanged();
	}

	TSharedPtr<FJsonObject> Data = MakeShared<FJsonObject>();
	Data->SetStringField(TEXT("node_id"), NodeId);
	Data->SetStringField(TEXT("node_title"), Node->GetNodeTitle(ENodeTitleType::FullTitle).ToString());
	Data->SetStringField(TEXT("comment"), Comment);

	if (Comment.IsEmpty())
	{
		return FBlueprintActionResult::Success(
			FString::Printf(TEXT("Cleared comment on node '%s' (%s)"), *NodeId, *Node->GetNodeTitle(ENodeTitleType::FullTitle).ToString()),
			Data);
	}

	return FBlueprintActionResult::Success(
		FString::Printf(TEXT("Set comment on node '%s' (%s): \"%s\""), *NodeId, *Node->GetNodeTitle(ENodeTitleType::FullTitle).ToString(), *Comment),
		Data);
}
