// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#include "Actions/SetComponentPropertyAction.h"

#include "Engine/Blueprint.h"
#include "Engine/SCS_Node.h"
#include "Engine/SimpleConstructionScript.h"
#include "Engine/StaticMesh.h"
#include "Materials/MaterialInterface.h"
#include "Kismet2/BlueprintEditorUtils.h"
#include "AssetRegistry/AssetRegistryModule.h"
#include "UObject/UnrealType.h"
#include "UObject/EnumProperty.h"
#include "UObject/TextProperty.h"
#include "Components/PrimitiveComponent.h"
#include "GameFramework/Actor.h"

// ── Helpers ──────────────────────────────────────────────────────────────────

static UBlueprint* FindBPForCompProp(const FString& NameOrPath)
{
	if (NameOrPath.StartsWith(TEXT("/")))
	{
		return LoadObject<UBlueprint>(nullptr, *NameOrPath);
	}
	FAssetRegistryModule& AssetReg = FModuleManager::LoadModuleChecked<FAssetRegistryModule>("AssetRegistry");
	TArray<FAssetData> Assets;
	AssetReg.Get().GetAssetsByClass(UBlueprint::StaticClass()->GetClassPathName(), Assets);
	for (const FAssetData& Asset : Assets)
	{
		if (Asset.AssetName.ToString().Equals(NameOrPath, ESearchCase::IgnoreCase))
		{
			return Cast<UBlueprint>(Asset.GetAsset());
		}
	}
	return nullptr;
}

static USCS_Node* FindSCSNode(USimpleConstructionScript* SCS, const FString& CompName)
{
	for (USCS_Node* Node : SCS->GetAllNodes())
	{
		if (Node->GetVariableName().ToString().Equals(CompName, ESearchCase::IgnoreCase))
		{
			return Node;
		}
		// Also match against the component template name
		if (Node->ComponentTemplate &&
			Node->ComponentTemplate->GetName().Equals(CompName, ESearchCase::IgnoreCase))
		{
			return Node;
		}
	}
	return nullptr;
}

static UActorComponent* FindInheritedComponent(UBlueprint* BP, const FString& CompName)
{
	if (!BP || !BP->GeneratedClass)
	{
		return nullptr;
	}

	AActor* CDO = Cast<AActor>(BP->GeneratedClass->GetDefaultObject());
	if (!CDO)
	{
		return nullptr;
	}

	// Try matching by class member property name (e.g., "CharacterMovement", "CapsuleComponent", "Mesh")
	for (TFieldIterator<FObjectPropertyBase> It(CDO->GetClass()); It; ++It)
	{
		FObjectPropertyBase* Prop = *It;
		if (Prop->PropertyClass && Prop->PropertyClass->IsChildOf(UActorComponent::StaticClass()))
		{
			if (Prop->GetName().Equals(CompName, ESearchCase::IgnoreCase))
			{
				UObject* Obj = Prop->GetObjectPropertyValue(Prop->ContainerPtrToValuePtr<void>(CDO));
				if (UActorComponent* Comp = Cast<UActorComponent>(Obj))
				{
					return Comp;
				}
			}
		}
	}

	// Try matching by component instance name (subobject name, e.g., "CharMoveComp", "CollisionCylinder")
	TInlineComponentArray<UActorComponent*> Components;
	CDO->GetComponents(Components);
	for (UActorComponent* Comp : Components)
	{
		if (Comp->GetName().Equals(CompName, ESearchCase::IgnoreCase))
		{
			return Comp;
		}
	}

	// Try matching by class name (e.g., "CharacterMovementComponent" or "CharacterMovement")
	for (UActorComponent* Comp : Components)
	{
		FString ClassName = Comp->GetClass()->GetName();
		if (ClassName.Equals(CompName, ESearchCase::IgnoreCase))
		{
			return Comp;
		}
		FString ShortName = ClassName;
		ShortName.RemoveFromEnd(TEXT("Component"));
		if (!ShortName.IsEmpty() && ShortName.Equals(CompName, ESearchCase::IgnoreCase))
		{
			return Comp;
		}
	}

	return nullptr;
}

// ── Schema ───────────────────────────────────────────────────────────────────

TSharedPtr<FJsonObject> FSetComponentPropertyAction::GetParameterSchema() const
{
	auto MakeStr = [](const FString& Desc) -> TSharedPtr<FJsonObject> {
		TSharedPtr<FJsonObject> P = MakeShared<FJsonObject>();
		P->SetStringField(TEXT("type"), TEXT("string"));
		P->SetStringField(TEXT("description"), Desc);
		return P;
	};

	TSharedPtr<FJsonObject> Schema = MakeShared<FJsonObject>();
	Schema->SetStringField(TEXT("type"), TEXT("object"));

	TSharedPtr<FJsonObject> Props = MakeShared<FJsonObject>();
	Props->SetObjectField(TEXT("blueprint"), MakeStr(TEXT("Blueprint name or content path")));
	Props->SetObjectField(TEXT("component_name"), MakeStr(TEXT("Component name. For Blueprint-added: use name from add_component. For inherited from parent class (e.g. CharacterMovement, CapsuleComponent, Mesh): use the property name shown in components list.")));
	Props->SetObjectField(TEXT("property_name"), MakeStr(TEXT("Property name (e.g. 'StaticMesh', 'RelativeLocation', 'CastShadow', 'InitialSpeed')")));
	Props->SetObjectField(TEXT("property_value"), MakeStr(TEXT("Value as string. Objects: content path. Vectors: '(X=0,Y=0,Z=0)' with parens. Rotators: '(Pitch=0,Yaw=0,Roll=0)'. Bools: 'true'/'false'. Enums: enum value name.")));

	Schema->SetObjectField(TEXT("properties"), Props);

	TArray<TSharedPtr<FJsonValue>> Required;
	Required.Add(MakeShared<FJsonValueString>(TEXT("blueprint")));
	Required.Add(MakeShared<FJsonValueString>(TEXT("component_name")));
	Required.Add(MakeShared<FJsonValueString>(TEXT("property_name")));
	Required.Add(MakeShared<FJsonValueString>(TEXT("property_value")));
	Schema->SetArrayField(TEXT("required"), Required);

	return Schema;
}

// ── Execute ──────────────────────────────────────────────────────────────────

FBlueprintActionResult FSetComponentPropertyAction::Execute(const TSharedPtr<FJsonObject>& Params)
{
	const FString BPName = Params->GetStringField(TEXT("blueprint"));
	const FString CompName = Params->GetStringField(TEXT("component_name"));
	const FString PropName = Params->GetStringField(TEXT("property_name"));
	const FString PropValue = Params->GetStringField(TEXT("property_value"));

	// Find blueprint
	UBlueprint* BP = FindBPForCompProp(BPName);
	if (!BP)
	{
		return FBlueprintActionResult::Failure(FString::Printf(TEXT("Blueprint '%s' not found"), *BPName));
	}

	USimpleConstructionScript* SCS = BP->SimpleConstructionScript;

	// Find the component template - first try SCS (Blueprint-added), then inherited from parent
	UActorComponent* Template = nullptr;
	if (SCS)
	{
		USCS_Node* SCSNode = FindSCSNode(SCS, CompName);
		if (SCSNode)
		{
			Template = SCSNode->ComponentTemplate;
		}
	}

	// If not found in SCS, try inherited components from parent class (e.g., CharacterMovement, CapsuleComponent, Mesh)
	if (!Template)
	{
		Template = FindInheritedComponent(BP, CompName);
	}

	if (!Template)
	{
		// Build comprehensive list of available components (SCS + inherited)
		FString Available;
		TSet<FString> Listed;
		if (SCS)
		{
			for (USCS_Node* N : SCS->GetAllNodes())
			{
				FString Name = N->GetVariableName().ToString();
				Available += Name + TEXT(", ");
				Listed.Add(Name.ToLower());
			}
		}
		if (BP->GeneratedClass)
		{
			AActor* CDO = Cast<AActor>(BP->GeneratedClass->GetDefaultObject());
			if (CDO)
			{
				for (TFieldIterator<FObjectPropertyBase> It(CDO->GetClass()); It; ++It)
				{
					FObjectPropertyBase* Prop = *It;
					if (Prop->PropertyClass && Prop->PropertyClass->IsChildOf(UActorComponent::StaticClass()))
					{
						FString InheritedName = Prop->GetName();
						if (!Listed.Contains(InheritedName.ToLower()))
						{
							UObject* Obj = Prop->GetObjectPropertyValue(Prop->ContainerPtrToValuePtr<void>(CDO));
							if (Obj)
							{
								Available += InheritedName + TEXT(" (inherited), ");
								Listed.Add(InheritedName.ToLower());
							}
						}
					}
				}
			}
		}
		return FBlueprintActionResult::Failure(
			FString::Printf(TEXT("Component '%s' not found in '%s'. Available: [%s]"),
				*CompName, *BPName, *Available));
	}

	// ── Special handling for collision properties (nested in BodyInstance) ──
	UPrimitiveComponent* PrimComp = Cast<UPrimitiveComponent>(Template);
	if (PrimComp)
	{
		FString PropLower = PropName.ToLower();

		// CollisionProfileName
		if (PropLower == TEXT("collisionprofilename") || PropLower == TEXT("collision_profile") || PropLower == TEXT("collisionprofile"))
		{
			PrimComp->SetCollisionProfileName(FName(*PropValue));
			FBlueprintEditorUtils::MarkBlueprintAsStructurallyModified(BP);

			TSharedPtr<FJsonObject> Data = MakeShared<FJsonObject>();
			Data->SetStringField(TEXT("component_name"), CompName);
			Data->SetStringField(TEXT("property_name"), TEXT("CollisionProfileName"));
			Data->SetStringField(TEXT("property_value"), PropValue);
			return FBlueprintActionResult::Success(
				FString::Printf(TEXT("Set collision profile to '%s' on component '%s' in '%s'"),
					*PropValue, *CompName, *BPName),
				Data);
		}

		// CollisionEnabled
		if (PropLower == TEXT("collisionenabled") || PropLower == TEXT("collision_enabled"))
		{
			FString ValLower = PropValue.ToLower();
			ECollisionEnabled::Type CollType = ECollisionEnabled::QueryAndPhysics; // default
			if (ValLower.Contains(TEXT("nocollision")) || ValLower == TEXT("none") || ValLower == TEXT("false") || ValLower == TEXT("disabled"))
			{
				CollType = ECollisionEnabled::NoCollision;
			}
			else if (ValLower.Contains(TEXT("queryonly")) || ValLower.Contains(TEXT("query_only")))
			{
				CollType = ECollisionEnabled::QueryOnly;
			}
			else if (ValLower.Contains(TEXT("physicsonly")) || ValLower.Contains(TEXT("physics_only")))
			{
				CollType = ECollisionEnabled::PhysicsOnly;
			}
			PrimComp->SetCollisionEnabled(CollType);
			FBlueprintEditorUtils::MarkBlueprintAsStructurallyModified(BP);

			TSharedPtr<FJsonObject> Data = MakeShared<FJsonObject>();
			Data->SetStringField(TEXT("component_name"), CompName);
			Data->SetStringField(TEXT("property_name"), TEXT("CollisionEnabled"));
			Data->SetStringField(TEXT("property_value"), PropValue);
			return FBlueprintActionResult::Success(
				FString::Printf(TEXT("Set collision enabled to '%s' on component '%s' in '%s'"),
					*PropValue, *CompName, *BPName),
				Data);
		}

		// bSimulatePhysics
		if (PropLower == TEXT("bsimulatephysics") || PropLower == TEXT("simulatephysics") || PropLower == TEXT("simulate_physics"))
		{
			FString ValLower = PropValue.ToLower();
			bool bSimulate = (ValLower == TEXT("true") || ValLower == TEXT("1") || ValLower == TEXT("yes"));
			PrimComp->SetSimulatePhysics(bSimulate);
			FBlueprintEditorUtils::MarkBlueprintAsStructurallyModified(BP);

			TSharedPtr<FJsonObject> Data = MakeShared<FJsonObject>();
			Data->SetStringField(TEXT("component_name"), CompName);
			Data->SetStringField(TEXT("property_name"), TEXT("bSimulatePhysics"));
			Data->SetStringField(TEXT("property_value"), PropValue);
			return FBlueprintActionResult::Success(
				FString::Printf(TEXT("Set simulate physics to '%s' on component '%s' in '%s'"),
					*PropValue, *CompName, *BPName),
				Data);
		}

		// bGenerateOverlapEvents / GenerateOverlapEvents
		if (PropLower == TEXT("bgenerateoverlapevents") || PropLower == TEXT("generateoverlapevents") || PropLower == TEXT("generate_overlap_events"))
		{
			FString ValLower = PropValue.ToLower();
			bool bOverlap = (ValLower == TEXT("true") || ValLower == TEXT("1") || ValLower == TEXT("yes"));
			PrimComp->SetGenerateOverlapEvents(bOverlap);
			FBlueprintEditorUtils::MarkBlueprintAsStructurallyModified(BP);

			TSharedPtr<FJsonObject> Data = MakeShared<FJsonObject>();
			Data->SetStringField(TEXT("component_name"), CompName);
			Data->SetStringField(TEXT("property_name"), TEXT("bGenerateOverlapEvents"));
			Data->SetStringField(TEXT("property_value"), PropValue);
			return FBlueprintActionResult::Success(
				FString::Printf(TEXT("Set generate overlap events to '%s' on component '%s' in '%s'"),
					*PropValue, *CompName, *BPName),
				Data);
		}

		// CollisionResponseToAllChannels
		if (PropLower == TEXT("collisionresponsetoallchannels") || PropLower == TEXT("collision_response"))
		{
			FString ValLower = PropValue.ToLower();
			ECollisionResponse Response = ECR_Block;
			if (ValLower.Contains(TEXT("ignore"))) Response = ECR_Ignore;
			else if (ValLower.Contains(TEXT("overlap"))) Response = ECR_Overlap;
			PrimComp->SetCollisionResponseToAllChannels(Response);
			FBlueprintEditorUtils::MarkBlueprintAsStructurallyModified(BP);

			TSharedPtr<FJsonObject> Data = MakeShared<FJsonObject>();
			Data->SetStringField(TEXT("component_name"), CompName);
			Data->SetStringField(TEXT("property_name"), TEXT("CollisionResponseToAllChannels"));
			Data->SetStringField(TEXT("property_value"), PropValue);
			return FBlueprintActionResult::Success(
				FString::Printf(TEXT("Set collision response to all channels to '%s' on component '%s' in '%s'"),
					*PropValue, *CompName, *BPName),
				Data);
		}

		// bNotifyRigidBodyCollision (for Hit events)
		if (PropLower == TEXT("bnotifyrigidbodycollision") || PropLower == TEXT("notifyrigidbodycollision") || PropLower == TEXT("simulation_generates_hit_events"))
		{
			FString ValLower = PropValue.ToLower();
			bool bNotify = (ValLower == TEXT("true") || ValLower == TEXT("1") || ValLower == TEXT("yes"));
			PrimComp->SetNotifyRigidBodyCollision(bNotify);
			FBlueprintEditorUtils::MarkBlueprintAsStructurallyModified(BP);

			TSharedPtr<FJsonObject> Data = MakeShared<FJsonObject>();
			Data->SetStringField(TEXT("component_name"), CompName);
			Data->SetStringField(TEXT("property_name"), TEXT("bNotifyRigidBodyCollision"));
			Data->SetStringField(TEXT("property_value"), PropValue);
			return FBlueprintActionResult::Success(
				FString::Printf(TEXT("Set notify rigid body collision to '%s' on component '%s' in '%s'"),
					*PropValue, *CompName, *BPName),
				Data);
		}
	}

	// Try to find and set property via UE reflection
	FProperty* Property = Template->GetClass()->FindPropertyByName(FName(*PropName));
	if (!Property)
	{
		// Try case-insensitive search
		for (TFieldIterator<FProperty> It(Template->GetClass()); It; ++It)
		{
			if (It->GetName().Equals(PropName, ESearchCase::IgnoreCase))
			{
				Property = *It;
				break;
			}
		}
	}

	if (!Property)
	{
		// List some available properties
		FString Available;
		int32 Count = 0;
		for (TFieldIterator<FProperty> It(Template->GetClass()); It; ++It)
		{
			if (It->HasAnyPropertyFlags(CPF_Edit | CPF_BlueprintVisible))
			{
				Available += It->GetName() + TEXT(", ");
				if (++Count >= 20) { Available += TEXT("..."); break; }
			}
		}
		return FBlueprintActionResult::Failure(
			FString::Printf(TEXT("Property '%s' not found on '%s'. Available: [%s]"),
				*PropName, *Template->GetClass()->GetName(), *Available));
	}

	// Set the property value
	Template->Modify();

	void* ValuePtr = Property->ContainerPtrToValuePtr<void>(Template);
	bool bSet = false;

	// Handle object properties specially (static mesh, material, etc.)
	if (FObjectPropertyBase* ObjProp = CastField<FObjectPropertyBase>(Property))
	{
		UObject* LoadedObj = LoadObject<UObject>(nullptr, *PropValue);
		if (LoadedObj)
		{
			ObjProp->SetObjectPropertyValue(ValuePtr, LoadedObj);
			bSet = true;
		}
		else
		{
			return FBlueprintActionResult::Failure(
				FString::Printf(TEXT("Could not load object '%s' for property '%s'. Use full content path (e.g. '/Engine/BasicShapes/Sphere.Sphere')"),
					*PropValue, *PropName));
		}
	}
	// Bool
	else if (FBoolProperty* BoolProp = CastField<FBoolProperty>(Property))
	{
		FString Lower = PropValue.ToLower();
		bool bVal = (Lower == TEXT("true") || Lower == TEXT("1") || Lower == TEXT("yes"));
		BoolProp->SetPropertyValue(ValuePtr, bVal);
		bSet = true;
	}
	// Numeric types
	else if (CastField<FFloatProperty>(Property))
	{
		float Val = FCString::Atof(*PropValue);
		CastField<FFloatProperty>(Property)->SetPropertyValue(ValuePtr, Val);
		bSet = true;
	}
	else if (CastField<FDoubleProperty>(Property))
	{
		double Val = FCString::Atod(*PropValue);
		CastField<FDoubleProperty>(Property)->SetPropertyValue(ValuePtr, Val);
		bSet = true;
	}
	else if (FIntProperty* IntProp = CastField<FIntProperty>(Property))
	{
		int32 Val = FCString::Atoi(*PropValue);
		IntProp->SetPropertyValue(ValuePtr, Val);
		bSet = true;
	}
	// FName
	else if (FNameProperty* NameProp = CastField<FNameProperty>(Property))
	{
		NameProp->SetPropertyValue(ValuePtr, FName(*PropValue));
		bSet = true;
	}
	// FString
	else if (FStrProperty* StrProp = CastField<FStrProperty>(Property))
	{
		StrProp->SetPropertyValue(ValuePtr, PropValue);
		bSet = true;
	}
	// Enum (by name)
	else if (FEnumProperty* EnumProp = CastField<FEnumProperty>(Property))
	{
		UEnum* Enum = EnumProp->GetEnum();
		int64 EnumVal = Enum->GetValueByNameString(PropValue);
		if (EnumVal != INDEX_NONE)
		{
			EnumProp->GetUnderlyingProperty()->SetIntPropertyValue(ValuePtr, EnumVal);
			bSet = true;
		}
	}
	else if (FByteProperty* ByteProp = CastField<FByteProperty>(Property))
	{
		if (UEnum* Enum = ByteProp->Enum)
		{
			int64 EnumVal = Enum->GetValueByNameString(PropValue);
			if (EnumVal != INDEX_NONE)
			{
				ByteProp->SetPropertyValue(ValuePtr, (uint8)EnumVal);
				bSet = true;
			}
		}
		if (!bSet)
		{
			ByteProp->SetPropertyValue(ValuePtr, (uint8)FCString::Atoi(*PropValue));
			bSet = true;
		}
	}

	// Fallback: use ImportText for struct types (FVector, FRotator, FTransform, FColor, etc.)
	if (!bSet)
	{
		// Auto-wrap in parentheses if the value looks like a struct (has '=' signs)
		// UE's ImportText expects "(X=0,Y=0,Z=0)" not "X=0,Y=0,Z=0"
		FString ImportValue = PropValue;
		if (ImportValue.Contains(TEXT("=")) && !ImportValue.StartsWith(TEXT("(")))
		{
			ImportValue = TEXT("(") + ImportValue + TEXT(")");
		}
		const TCHAR* Result = Property->ImportText_Direct(*ImportValue, ValuePtr, Template, PPF_None);
		bSet = (Result != nullptr);
	}

	if (!bSet)
	{
		return FBlueprintActionResult::Failure(
			FString::Printf(TEXT("Failed to set property '%s' to '%s' on component '%s'"),
				*PropName, *PropValue, *CompName));
	}

	FBlueprintEditorUtils::MarkBlueprintAsStructurallyModified(BP);

	TSharedPtr<FJsonObject> Data = MakeShared<FJsonObject>();
	Data->SetStringField(TEXT("component_name"), CompName);
	Data->SetStringField(TEXT("property_name"), PropName);
	Data->SetStringField(TEXT("property_value"), PropValue);

	return FBlueprintActionResult::Success(
		FString::Printf(TEXT("Set '%s' = '%s' on component '%s' in '%s'"),
			*PropName, *PropValue, *CompName, *BPName),
		Data);
}
