// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#include "Actions/SearchBlueprintNodesAction.h"
#include "Actions/ActionRegistry.h"

#include "Engine/Blueprint.h"
#include "EdGraph/EdGraph.h"
#include "EdGraph/EdGraphNode.h"
#include "EdGraph/EdGraphPin.h"
#include "AssetRegistry/AssetRegistryModule.h"

static UBlueprint* FindBPForSearch(const FString& NameOrPath)
{
	if (NameOrPath.StartsWith(TEXT("/")))
	{
		return LoadObject<UBlueprint>(nullptr, *NameOrPath);
	}
	FAssetRegistryModule& AssetReg = FModuleManager::LoadModuleChecked<FAssetRegistryModule>("AssetRegistry");
	TArray<FAssetData> Assets;
	AssetReg.Get().GetAssetsByClass(UBlueprint::StaticClass()->GetClassPathName(), Assets);
	for (const FAssetData& Asset : Assets)
	{
		if (Asset.AssetName.ToString().Equals(NameOrPath, ESearchCase::IgnoreCase))
		{
			return Cast<UBlueprint>(Asset.GetAsset());
		}
	}
	return nullptr;
}

TSharedPtr<FJsonObject> FSearchBlueprintNodesAction::GetParameterSchema() const
{
	auto MakeStr = [](const FString& Desc) -> TSharedPtr<FJsonObject> {
		TSharedPtr<FJsonObject> P = MakeShared<FJsonObject>();
		P->SetStringField(TEXT("type"), TEXT("string"));
		P->SetStringField(TEXT("description"), Desc);
		return P;
	};
	auto MakeInt = [](const FString& Desc, int32 Default) -> TSharedPtr<FJsonObject> {
		TSharedPtr<FJsonObject> P = MakeShared<FJsonObject>();
		P->SetStringField(TEXT("type"), TEXT("integer"));
		P->SetStringField(TEXT("description"), Desc);
		P->SetNumberField(TEXT("default"), Default);
		return P;
	};

	TSharedPtr<FJsonObject> Schema = MakeShared<FJsonObject>();
	Schema->SetStringField(TEXT("type"), TEXT("object"));

	TSharedPtr<FJsonObject> Props = MakeShared<FJsonObject>();
	Props->SetObjectField(TEXT("blueprint"), MakeStr(TEXT("Blueprint name or content path")));
	Props->SetObjectField(TEXT("query"), MakeStr(TEXT("Search keyword to match against node titles, class names, and pin names. Case-insensitive. Examples: 'Jump', 'BeginPlay', 'SetActorLocation', 'Branch'.")));
	Props->SetObjectField(TEXT("graph_name"), MakeStr(TEXT("Optional: only search in this graph (e.g. 'EventGraph'). If omitted, searches all graphs.")));
	Props->SetObjectField(TEXT("max_results"), MakeInt(TEXT("Maximum number of matching nodes to return (default: 10)"), 10));

	Schema->SetObjectField(TEXT("properties"), Props);

	TArray<TSharedPtr<FJsonValue>> Required;
	Required.Add(MakeShared<FJsonValueString>(TEXT("blueprint")));
	Required.Add(MakeShared<FJsonValueString>(TEXT("query")));
	Schema->SetArrayField(TEXT("required"), Required);

	return Schema;
}

FBlueprintActionResult FSearchBlueprintNodesAction::Execute(const TSharedPtr<FJsonObject>& Params)
{
	const FString BPName = Params->GetStringField(TEXT("blueprint"));
	const FString Query = Params->GetStringField(TEXT("query"));
	const FString FilterGraph = Params->HasField(TEXT("graph_name")) ? Params->GetStringField(TEXT("graph_name")) : TEXT("");
	const int32 MaxResults = Params->HasField(TEXT("max_results")) ? static_cast<int32>(Params->GetNumberField(TEXT("max_results"))) : 10;

	if (Query.IsEmpty())
	{
		return FBlueprintActionResult::Failure(TEXT("'query' parameter is required and cannot be empty."));
	}

	UBlueprint* BP = FindBPForSearch(BPName);
	if (!BP)
	{
		return FBlueprintActionResult::Failure(FString::Printf(TEXT("Blueprint '%s' not found"), *BPName));
	}

	TArray<UEdGraph*> AllGraphs;
	BP->GetAllGraphs(AllGraphs);

	TArray<TSharedPtr<FJsonValue>> MatchesArray;
	int32 TotalSearched = 0;
	int32 TotalMatched = 0;

	for (UEdGraph* Graph : AllGraphs)
	{
		if (!FilterGraph.IsEmpty() && !Graph->GetName().Equals(FilterGraph, ESearchCase::IgnoreCase))
		{
			continue;
		}

		for (UEdGraphNode* Node : Graph->Nodes)
		{
			TotalSearched++;

			// Check if query matches node title, class name, or any pin name
			FString Title = Node->GetNodeTitle(ENodeTitleType::FullTitle).ToString();
			FString ClassName = Node->GetClass()->GetName();
			bool bMatch = Title.Contains(Query, ESearchCase::IgnoreCase) ||
				ClassName.Contains(Query, ESearchCase::IgnoreCase);

			// Also check pin names if not already matched
			if (!bMatch)
			{
				for (UEdGraphPin* Pin : Node->Pins)
				{
					if (Pin->PinName.ToString().Contains(Query, ESearchCase::IgnoreCase))
					{
						bMatch = true;
						break;
					}
				}
			}

			if (!bMatch) continue;

			TotalMatched++;
			if (MatchesArray.Num() >= MaxResults) continue; // Count but don't add

			TSharedPtr<FJsonObject> NodeObj = MakeShared<FJsonObject>();

			// Track node for ID assignment
			FString NodeId = Registry->TrackNode(Node);
			NodeObj->SetStringField(TEXT("node_id"), NodeId);
			NodeObj->SetStringField(TEXT("graph"), Graph->GetName());
			NodeObj->SetStringField(TEXT("title"), Title);
			NodeObj->SetStringField(TEXT("class"), ClassName);
			NodeObj->SetNumberField(TEXT("pos_x"), Node->NodePosX);
			NodeObj->SetNumberField(TEXT("pos_y"), Node->NodePosY);

			// Include pins with connections
			TArray<TSharedPtr<FJsonValue>> PinsArray;
			for (UEdGraphPin* Pin : Node->Pins)
			{
				if (Pin->bHidden) continue;

				TSharedPtr<FJsonObject> PinObj = MakeShared<FJsonObject>();
				PinObj->SetStringField(TEXT("name"), Pin->PinName.ToString());
				PinObj->SetStringField(TEXT("dir"), Pin->Direction == EGPD_Input ? TEXT("In") : TEXT("Out"));

				if (Pin->LinkedTo.Num() > 0)
				{
					TArray<TSharedPtr<FJsonValue>> Links;
					for (UEdGraphPin* Link : Pin->LinkedTo)
					{
						FString LinkedNodeId = Registry->TrackNode(Link->GetOwningNode());
						Links.Add(MakeShared<FJsonValueString>(
							FString::Printf(TEXT("%s.%s"), *LinkedNodeId, *Link->PinName.ToString())));
					}
					PinObj->SetArrayField(TEXT("linked"), Links);
				}

				PinsArray.Add(MakeShared<FJsonValueObject>(PinObj));
			}
			NodeObj->SetArrayField(TEXT("pins"), PinsArray);

			MatchesArray.Add(MakeShared<FJsonValueObject>(NodeObj));
		}
	}

	TSharedPtr<FJsonObject> Data = MakeShared<FJsonObject>();
	Data->SetStringField(TEXT("blueprint"), BP->GetName());
	Data->SetStringField(TEXT("query"), Query);
	Data->SetNumberField(TEXT("total_searched"), TotalSearched);
	Data->SetNumberField(TEXT("total_matched"), TotalMatched);
	Data->SetNumberField(TEXT("returned"), MatchesArray.Num());
	Data->SetArrayField(TEXT("matches"), MatchesArray);

	FString Message = FString::Printf(TEXT("Found %d node(s) matching '%s' in '%s' (searched %d nodes, showing %d)"),
		TotalMatched, *Query, *BP->GetName(), TotalSearched, MatchesArray.Num());

	return FBlueprintActionResult::Success(Message, Data);
}
