// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#include "Actions/RenameVariableAction.h"

#include "Engine/Blueprint.h"
#include "Kismet2/BlueprintEditorUtils.h"
#include "AssetRegistry/AssetRegistryModule.h"

static UBlueprint* FindBPForRename(const FString& NameOrPath)
{
	if (NameOrPath.StartsWith(TEXT("/")))
	{
		return LoadObject<UBlueprint>(nullptr, *NameOrPath);
	}
	FAssetRegistryModule& AssetReg = FModuleManager::LoadModuleChecked<FAssetRegistryModule>("AssetRegistry");
	TArray<FAssetData> Assets;
	AssetReg.Get().GetAssetsByClass(UBlueprint::StaticClass()->GetClassPathName(), Assets);
	for (const FAssetData& Asset : Assets)
	{
		if (Asset.AssetName.ToString().Equals(NameOrPath, ESearchCase::IgnoreCase))
		{
			return Cast<UBlueprint>(Asset.GetAsset());
		}
	}
	return nullptr;
}

TSharedPtr<FJsonObject> FRenameVariableAction::GetParameterSchema() const
{
	auto MakeStr = [](const FString& Desc) -> TSharedPtr<FJsonObject> {
		TSharedPtr<FJsonObject> P = MakeShared<FJsonObject>();
		P->SetStringField(TEXT("type"), TEXT("string"));
		P->SetStringField(TEXT("description"), Desc);
		return P;
	};

	TSharedPtr<FJsonObject> Schema = MakeShared<FJsonObject>();
	Schema->SetStringField(TEXT("type"), TEXT("object"));

	TSharedPtr<FJsonObject> Props = MakeShared<FJsonObject>();
	Props->SetObjectField(TEXT("blueprint"), MakeStr(TEXT("Blueprint name or content path")));
	Props->SetObjectField(TEXT("old_name"), MakeStr(TEXT("Current variable name")));
	Props->SetObjectField(TEXT("new_name"), MakeStr(TEXT("New variable name")));

	Schema->SetObjectField(TEXT("properties"), Props);

	TArray<TSharedPtr<FJsonValue>> Required;
	Required.Add(MakeShared<FJsonValueString>(TEXT("blueprint")));
	Required.Add(MakeShared<FJsonValueString>(TEXT("old_name")));
	Required.Add(MakeShared<FJsonValueString>(TEXT("new_name")));
	Schema->SetArrayField(TEXT("required"), Required);

	return Schema;
}

FBlueprintActionResult FRenameVariableAction::Execute(const TSharedPtr<FJsonObject>& Params)
{
	const FString BPName = Params->GetStringField(TEXT("blueprint"));
	const FString OldName = Params->GetStringField(TEXT("old_name"));
	const FString NewName = Params->GetStringField(TEXT("new_name"));

	UBlueprint* BP = FindBPForRename(BPName);
	if (!BP)
	{
		return FBlueprintActionResult::Failure(FString::Printf(TEXT("Blueprint '%s' not found"), *BPName));
	}

	// Check that the old variable exists
	bool bFound = false;
	for (const FBPVariableDescription& Var : BP->NewVariables)
	{
		if (Var.VarName == FName(*OldName))
		{
			bFound = true;
			break;
		}
	}

	if (!bFound)
	{
		return FBlueprintActionResult::Failure(
			FString::Printf(TEXT("Variable '%s' not found in Blueprint '%s'"), *OldName, *BPName));
	}

	FBlueprintEditorUtils::RenameMemberVariable(BP, FName(*OldName), FName(*NewName));

	TSharedPtr<FJsonObject> Data = MakeShared<FJsonObject>();
	Data->SetStringField(TEXT("blueprint"), BPName);
	Data->SetStringField(TEXT("old_name"), OldName);
	Data->SetStringField(TEXT("new_name"), NewName);

	return FBlueprintActionResult::Success(
		FString::Printf(TEXT("Renamed variable '%s' to '%s' in '%s'"), *OldName, *NewName, *BPName),
		Data);
}
