// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#include "Actions/RemoveMemoryAction.h"
#include "Memory/BlueprintAIMemory.h"

TSharedPtr<FJsonObject> FRemoveMemoryAction::GetParameterSchema() const
{
	TSharedPtr<FJsonObject> Schema = MakeShared<FJsonObject>();
	Schema->SetStringField(TEXT("type"), TEXT("object"));

	TSharedPtr<FJsonObject> Props = MakeShared<FJsonObject>();

	TSharedPtr<FJsonObject> IdProp = MakeShared<FJsonObject>();
	IdProp->SetStringField(TEXT("type"), TEXT("string"));
	IdProp->SetStringField(TEXT("description"),
		TEXT("The memory ID to remove (e.g. 'M1', 'M2'). See the memory list in the system context."));
	Props->SetObjectField(TEXT("memory_id"), IdProp);

	Schema->SetObjectField(TEXT("properties"), Props);

	TArray<TSharedPtr<FJsonValue>> Required;
	Required.Add(MakeShared<FJsonValueString>(TEXT("memory_id")));
	Schema->SetArrayField(TEXT("required"), Required);

	return Schema;
}

FBlueprintActionResult FRemoveMemoryAction::Execute(const TSharedPtr<FJsonObject>& Params)
{
	FString MemoryId = Params->GetStringField(TEXT("memory_id"));
	if (MemoryId.IsEmpty())
	{
		return FBlueprintActionResult::Failure(TEXT("memory_id is required"));
	}

	if (FBlueprintAIMemory::Get().RemoveMemory(MemoryId))
	{
		return FBlueprintActionResult::Success(
			FString::Printf(TEXT("Memory %s removed successfully."), *MemoryId)
		);
	}
	else
	{
		return FBlueprintActionResult::Failure(
			FString::Printf(TEXT("Memory '%s' not found. Check the memory list for valid IDs."), *MemoryId)
		);
	}
}
