// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#include "Actions/RemoveComponentAction.h"

#include "Engine/Blueprint.h"
#include "Engine/SCS_Node.h"
#include "Engine/SimpleConstructionScript.h"
#include "Kismet2/BlueprintEditorUtils.h"
#include "AssetRegistry/AssetRegistryModule.h"

// ── Helpers ──────────────────────────────────────────────────────────────────

static UBlueprint* FindBPForRemoveComp(const FString& NameOrPath)
{
	if (NameOrPath.StartsWith(TEXT("/")))
	{
		return LoadObject<UBlueprint>(nullptr, *NameOrPath);
	}
	FAssetRegistryModule& AssetReg = FModuleManager::LoadModuleChecked<FAssetRegistryModule>("AssetRegistry");
	TArray<FAssetData> Assets;
	AssetReg.Get().GetAssetsByClass(UBlueprint::StaticClass()->GetClassPathName(), Assets);
	for (const FAssetData& Asset : Assets)
	{
		if (Asset.AssetName.ToString().Equals(NameOrPath, ESearchCase::IgnoreCase))
		{
			return Cast<UBlueprint>(Asset.GetAsset());
		}
	}
	return nullptr;
}

// ── Schema ───────────────────────────────────────────────────────────────────

TSharedPtr<FJsonObject> FRemoveComponentAction::GetParameterSchema() const
{
	auto MakeStr = [](const FString& Desc) -> TSharedPtr<FJsonObject> {
		TSharedPtr<FJsonObject> P = MakeShared<FJsonObject>();
		P->SetStringField(TEXT("type"), TEXT("string"));
		P->SetStringField(TEXT("description"), Desc);
		return P;
	};

	TSharedPtr<FJsonObject> Schema = MakeShared<FJsonObject>();
	Schema->SetStringField(TEXT("type"), TEXT("object"));

	TSharedPtr<FJsonObject> Props = MakeShared<FJsonObject>();
	Props->SetObjectField(TEXT("blueprint"), MakeStr(TEXT("Blueprint name or content path")));
	Props->SetObjectField(TEXT("component_name"), MakeStr(TEXT("Component variable name to remove (as returned by add_component)")));

	Schema->SetObjectField(TEXT("properties"), Props);

	TArray<TSharedPtr<FJsonValue>> Required;
	Required.Add(MakeShared<FJsonValueString>(TEXT("blueprint")));
	Required.Add(MakeShared<FJsonValueString>(TEXT("component_name")));
	Schema->SetArrayField(TEXT("required"), Required);

	return Schema;
}

// ── Execute ──────────────────────────────────────────────────────────────────

FBlueprintActionResult FRemoveComponentAction::Execute(const TSharedPtr<FJsonObject>& Params)
{
	const FString BPName = Params->GetStringField(TEXT("blueprint"));
	const FString CompName = Params->GetStringField(TEXT("component_name"));

	UBlueprint* BP = FindBPForRemoveComp(BPName);
	if (!BP)
	{
		return FBlueprintActionResult::Failure(FString::Printf(TEXT("Blueprint '%s' not found"), *BPName));
	}

	USimpleConstructionScript* SCS = BP->SimpleConstructionScript;
	if (!SCS)
	{
		return FBlueprintActionResult::Failure(
			FString::Printf(TEXT("Blueprint '%s' does not support components"), *BPName));
	}

	// Find the SCS node
	USCS_Node* TargetNode = nullptr;
	for (USCS_Node* Node : SCS->GetAllNodes())
	{
		if (Node->GetVariableName().ToString().Equals(CompName, ESearchCase::IgnoreCase))
		{
			TargetNode = Node;
			break;
		}
	}

	if (!TargetNode)
	{
		FString Available;
		for (USCS_Node* N : SCS->GetAllNodes())
		{
			Available += N->GetVariableName().ToString() + TEXT(", ");
		}
		return FBlueprintActionResult::Failure(
			FString::Printf(TEXT("Component '%s' not found in '%s'. Available: [%s]"),
				*CompName, *BPName, *Available));
	}

	FString CompClassName = TargetNode->ComponentTemplate
		? TargetNode->ComponentTemplate->GetClass()->GetName()
		: TEXT("Unknown");

	SCS->RemoveNode(TargetNode);

	FBlueprintEditorUtils::MarkBlueprintAsStructurallyModified(BP);

	TSharedPtr<FJsonObject> Data = MakeShared<FJsonObject>();
	Data->SetStringField(TEXT("removed_component"), CompName);
	Data->SetStringField(TEXT("component_class"), CompClassName);

	return FBlueprintActionResult::Success(
		FString::Printf(TEXT("Removed component '%s' (%s) from '%s'"),
			*CompName, *CompClassName, *BPName),
		Data);
}
