// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#include "Actions/ListAssetsAction.h"

#include "AssetRegistry/AssetRegistryModule.h"
#include "AssetRegistry/IAssetRegistry.h"
#include "Engine/Blueprint.h"

TSharedPtr<FJsonObject> FListAssetsAction::GetParameterSchema() const
{
	auto MakeStr = [](const FString& Desc) -> TSharedPtr<FJsonObject> {
		TSharedPtr<FJsonObject> P = MakeShared<FJsonObject>();
		P->SetStringField(TEXT("type"), TEXT("string"));
		P->SetStringField(TEXT("description"), Desc);
		return P;
	};
	auto MakeInt = [](const FString& Desc) -> TSharedPtr<FJsonObject> {
		TSharedPtr<FJsonObject> P = MakeShared<FJsonObject>();
		P->SetStringField(TEXT("type"), TEXT("integer"));
		P->SetStringField(TEXT("description"), Desc);
		return P;
	};

	TSharedPtr<FJsonObject> Schema = MakeShared<FJsonObject>();
	Schema->SetStringField(TEXT("type"), TEXT("object"));

	TSharedPtr<FJsonObject> Props = MakeShared<FJsonObject>();
	Props->SetObjectField(TEXT("path"), MakeStr(TEXT("Content path to search in (e.g. '/Game', '/Game/Blueprints'). Default: '/Game'")));
	Props->SetObjectField(TEXT("type_filter"), MakeStr(TEXT("Asset type to filter by (e.g. 'Blueprint', 'Material', 'StaticMesh', 'SkeletalMesh', 'Texture2D', 'SoundWave'). Leave empty for all types.")));
	Props->SetObjectField(TEXT("name_filter"), MakeStr(TEXT("Substring to match in the asset name (case-insensitive)")));
	Props->SetObjectField(TEXT("max_results"), MakeInt(TEXT("Maximum number of results to return. Default: 50")));

	Schema->SetObjectField(TEXT("properties"), Props);

	// No required fields — all optional
	TArray<TSharedPtr<FJsonValue>> Required;
	Schema->SetArrayField(TEXT("required"), Required);

	return Schema;
}

FBlueprintActionResult FListAssetsAction::Execute(const TSharedPtr<FJsonObject>& Params)
{
	const FString Path = Params->HasField(TEXT("path")) ? Params->GetStringField(TEXT("path")) : TEXT("/Game");
	const FString TypeFilter = Params->HasField(TEXT("type_filter")) ? Params->GetStringField(TEXT("type_filter")) : TEXT("");
	const FString NameFilter = Params->HasField(TEXT("name_filter")) ? Params->GetStringField(TEXT("name_filter")) : TEXT("");
	const int32 MaxResults = Params->HasField(TEXT("max_results")) ? (int32)Params->GetNumberField(TEXT("max_results")) : 50;

	FAssetRegistryModule& AssetReg = FModuleManager::LoadModuleChecked<FAssetRegistryModule>("AssetRegistry");
	IAssetRegistry& Registry = AssetReg.Get();

	TArray<FAssetData> AllAssets;
	Registry.GetAssetsByPath(FName(*Path), AllAssets, /*bRecursive=*/true);

	TArray<TSharedPtr<FJsonValue>> ResultArray;
	int32 Count = 0;

	for (const FAssetData& Asset : AllAssets)
	{
		if (Count >= MaxResults) break;

		// Type filter
		if (!TypeFilter.IsEmpty())
		{
			FString AssetClassName = Asset.AssetClassPath.GetAssetName().ToString();
			if (!AssetClassName.Equals(TypeFilter, ESearchCase::IgnoreCase) &&
				!AssetClassName.Equals(TEXT("U") + TypeFilter, ESearchCase::IgnoreCase))
			{
				continue;
			}
		}

		// Name filter
		if (!NameFilter.IsEmpty())
		{
			if (!Asset.AssetName.ToString().Contains(NameFilter))
			{
				continue;
			}
		}

		TSharedPtr<FJsonObject> AssetObj = MakeShared<FJsonObject>();
		AssetObj->SetStringField(TEXT("name"), Asset.AssetName.ToString());
		AssetObj->SetStringField(TEXT("path"), Asset.GetObjectPathString());
		AssetObj->SetStringField(TEXT("type"), Asset.AssetClassPath.GetAssetName().ToString());
		AssetObj->SetStringField(TEXT("package"), Asset.PackageName.ToString());

		ResultArray.Add(MakeShared<FJsonValueObject>(AssetObj));
		Count++;
	}

	TSharedPtr<FJsonObject> Data = MakeShared<FJsonObject>();
	Data->SetNumberField(TEXT("count"), ResultArray.Num());
	Data->SetNumberField(TEXT("total_found"), AllAssets.Num());
	Data->SetArrayField(TEXT("assets"), ResultArray);

	return FBlueprintActionResult::Success(
		FString::Printf(TEXT("Found %d assets (showing %d)"), AllAssets.Num(), ResultArray.Num()),
		Data);
}
