// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#include "Actions/GetSelectedNodesAction.h"
#include "Actions/ActionRegistry.h"

#include "Engine/Blueprint.h"
#include "EdGraph/EdGraphNode.h"
#include "EdGraphSchema_K2.h"
#include "Subsystems/AssetEditorSubsystem.h"
#include "Editor.h"
#include "BlueprintEditor.h"

// ── Lightweight pin serializer ────────────────────────────────────────────────
static FString SelPinTypeToStr(const FEdGraphPinType& PinType)
{
	FString Base = PinType.PinCategory.ToString();
	if (PinType.PinSubCategoryObject.IsValid())
	{
		Base += TEXT(":") + PinType.PinSubCategoryObject->GetName();
	}
	if (PinType.ContainerType == EPinContainerType::Array) Base = TEXT("Array<") + Base + TEXT(">");
	if (PinType.ContainerType == EPinContainerType::Set) Base = TEXT("Set<") + Base + TEXT(">");
	if (PinType.ContainerType == EPinContainerType::Map) Base = TEXT("Map<") + Base + TEXT(">");
	return Base;
}

static TArray<TSharedPtr<FJsonValue>> SerializeSelectedNodePins(UEdGraphNode* Node)
{
	TArray<TSharedPtr<FJsonValue>> PinArray;
	if (!Node) return PinArray;

	for (UEdGraphPin* Pin : Node->Pins)
	{
		if (Pin->bHidden) continue;

		TSharedPtr<FJsonObject> PinObj = MakeShared<FJsonObject>();
		PinObj->SetStringField(TEXT("name"), Pin->PinName.ToString());
		PinObj->SetStringField(TEXT("direction"), Pin->Direction == EGPD_Input ? TEXT("Input") : TEXT("Output"));
		PinObj->SetStringField(TEXT("type"), SelPinTypeToStr(Pin->PinType));

		// Show connected node IDs if any
		if (Pin->LinkedTo.Num() > 0)
		{
			PinObj->SetBoolField(TEXT("connected"), true);
		}

		if (!Pin->DefaultValue.IsEmpty())
		{
			PinObj->SetStringField(TEXT("default"), Pin->DefaultValue);
		}

		PinArray.Add(MakeShared<FJsonValueObject>(PinObj));
	}

	return PinArray;
}

TSharedPtr<FJsonObject> FGetSelectedNodesAction::GetParameterSchema() const
{
	TSharedPtr<FJsonObject> Schema = MakeShared<FJsonObject>();
	Schema->SetStringField(TEXT("type"), TEXT("object"));
	Schema->SetObjectField(TEXT("properties"), MakeShared<FJsonObject>());
	// No required parameters
	return Schema;
}

FBlueprintActionResult FGetSelectedNodesAction::Execute(const TSharedPtr<FJsonObject>& Params)
{
	if (!GEditor)
	{
		return FBlueprintActionResult::Failure(TEXT("Editor not available"));
	}

	UAssetEditorSubsystem* AssetEditorSubsystem = GEditor->GetEditorSubsystem<UAssetEditorSubsystem>();
	if (!AssetEditorSubsystem)
	{
		return FBlueprintActionResult::Failure(TEXT("AssetEditorSubsystem not available"));
	}

	// Find the active Blueprint editor
	TArray<UObject*> EditedAssets = AssetEditorSubsystem->GetAllEditedAssets();

	FBlueprintEditor* BlueprintEditor = nullptr;
	UBlueprint* ActiveBP = nullptr;

	for (UObject* Asset : EditedAssets)
	{
		UBlueprint* BP = Cast<UBlueprint>(Asset);
		if (!BP) continue;

		IAssetEditorInstance* EditorInstance = AssetEditorSubsystem->FindEditorForAsset(BP, /*bFocusIfOpen=*/false);
		if (!EditorInstance) continue;

		BlueprintEditor = static_cast<FBlueprintEditor*>(EditorInstance);
		ActiveBP = BP;
		break; // Use the first open Blueprint editor
	}

	if (!BlueprintEditor || !ActiveBP)
	{
		return FBlueprintActionResult::Failure(TEXT("No Blueprint editor is currently open. Open a Blueprint first using open_blueprint."));
	}

	// Get selected nodes
	const FGraphPanelSelectionSet& SelectedNodes = BlueprintEditor->GetSelectedNodes();

	if (SelectedNodes.Num() == 0)
	{
		TSharedPtr<FJsonObject> Data = MakeShared<FJsonObject>();
		Data->SetStringField(TEXT("blueprint"), ActiveBP->GetName());
		Data->SetNumberField(TEXT("count"), 0);
		return FBlueprintActionResult::Success(
			FString::Printf(TEXT("No nodes are currently selected in '%s'. The user needs to select nodes in the graph editor first."), *ActiveBP->GetName()),
			Data);
	}

	// Track and serialize each selected node
	TArray<TSharedPtr<FJsonValue>> NodeArray;
	FString NodeNames;

	for (UObject* Obj : SelectedNodes)
	{
		UEdGraphNode* Node = Cast<UEdGraphNode>(Obj);
		if (!Node) continue;

		FString NodeId = Registry->TrackNode(Node);
		FString NodeTitle = Node->GetNodeTitle(ENodeTitleType::FullTitle).ToString();

		TSharedPtr<FJsonObject> NodeObj = MakeShared<FJsonObject>();
		NodeObj->SetStringField(TEXT("node_id"), NodeId);
		NodeObj->SetStringField(TEXT("title"), NodeTitle);
		NodeObj->SetStringField(TEXT("class"), Node->GetClass()->GetName());

		// Include the graph this node belongs to
		if (UEdGraph* Graph = Node->GetGraph())
		{
			NodeObj->SetStringField(TEXT("graph"), Graph->GetName());
		}

		// Include position
		NodeObj->SetNumberField(TEXT("position_x"), Node->NodePosX);
		NodeObj->SetNumberField(TEXT("position_y"), Node->NodePosY);

		// Include pins
		NodeObj->SetArrayField(TEXT("pins"), SerializeSelectedNodePins(Node));

		// Include comment if node has one
		if (!Node->NodeComment.IsEmpty())
		{
			NodeObj->SetStringField(TEXT("comment"), Node->NodeComment);
		}

		NodeArray.Add(MakeShared<FJsonValueObject>(NodeObj));

		if (!NodeNames.IsEmpty()) NodeNames += TEXT(", ");
		NodeNames += FString::Printf(TEXT("%s (%s)"), *NodeTitle, *NodeId);
	}

	TSharedPtr<FJsonObject> Data = MakeShared<FJsonObject>();
	Data->SetStringField(TEXT("blueprint"), ActiveBP->GetName());
	Data->SetNumberField(TEXT("count"), NodeArray.Num());
	Data->SetArrayField(TEXT("selected_nodes"), NodeArray);

	return FBlueprintActionResult::Success(
		FString::Printf(TEXT("Found %d selected node(s) in '%s': %s"), NodeArray.Num(), *ActiveBP->GetName(), *NodeNames),
		Data);
}
