// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#include "Actions/GetProjectStructureAction.h"

#include "AssetRegistry/AssetRegistryModule.h"
#include "AssetRegistry/IAssetRegistry.h"

TSharedPtr<FJsonObject> FGetProjectStructureAction::GetParameterSchema() const
{
	auto MakeStr = [](const FString& Desc) -> TSharedPtr<FJsonObject> {
		TSharedPtr<FJsonObject> P = MakeShared<FJsonObject>();
		P->SetStringField(TEXT("type"), TEXT("string"));
		P->SetStringField(TEXT("description"), Desc);
		return P;
	};

	TSharedPtr<FJsonObject> Schema = MakeShared<FJsonObject>();
	Schema->SetStringField(TEXT("type"), TEXT("object"));

	TSharedPtr<FJsonObject> Props = MakeShared<FJsonObject>();
	Props->SetObjectField(TEXT("path"), MakeStr(TEXT("Content path to scan (default: '/Game')")));

	Schema->SetObjectField(TEXT("properties"), Props);

	TArray<TSharedPtr<FJsonValue>> Required;
	Schema->SetArrayField(TEXT("required"), Required);

	return Schema;
}

FBlueprintActionResult FGetProjectStructureAction::Execute(const TSharedPtr<FJsonObject>& Params)
{
	const FString RootPath = Params->HasField(TEXT("path")) ? Params->GetStringField(TEXT("path")) : TEXT("/Game");

	FAssetRegistryModule& AssetReg = FModuleManager::LoadModuleChecked<FAssetRegistryModule>("AssetRegistry");
	IAssetRegistry& Registry = AssetReg.Get();

	TArray<FAssetData> AllAssets;
	Registry.GetAssetsByPath(FName(*RootPath), AllAssets, /*bRecursive=*/true);

	// Count by type
	TMap<FString, int32> TypeCounts;
	for (const FAssetData& Asset : AllAssets)
	{
		FString TypeName = Asset.AssetClassPath.GetAssetName().ToString();
		TypeCounts.FindOrAdd(TypeName)++;
	}

	// Build folder list
	TSet<FString> Folders;
	for (const FAssetData& Asset : AllAssets)
	{
		FString PackagePath = Asset.PackagePath.ToString();
		Folders.Add(PackagePath);

		// Also add parent paths
		TArray<FString> Parts;
		PackagePath.ParseIntoArray(Parts, TEXT("/"), true);
		FString BuildPath;
		for (const FString& Part : Parts)
		{
			BuildPath += TEXT("/") + Part;
			Folders.Add(BuildPath);
		}
	}

	// Build result JSON
	TSharedPtr<FJsonObject> Data = MakeShared<FJsonObject>();
	Data->SetNumberField(TEXT("total_assets"), AllAssets.Num());
	Data->SetStringField(TEXT("root_path"), RootPath);

	// Asset type counts
	TSharedPtr<FJsonObject> TypeCountsObj = MakeShared<FJsonObject>();
	TArray<FString> SortedTypes;
	TypeCounts.GetKeys(SortedTypes);
	SortedTypes.Sort();
	for (const FString& Type : SortedTypes)
	{
		TypeCountsObj->SetNumberField(Type, TypeCounts[Type]);
	}
	Data->SetObjectField(TEXT("asset_type_counts"), TypeCountsObj);

	// Folder list
	TArray<FString> SortedFolders = Folders.Array();
	SortedFolders.Sort();
	TArray<TSharedPtr<FJsonValue>> FoldersArray;
	for (const FString& Folder : SortedFolders)
	{
		FoldersArray.Add(MakeShared<FJsonValueString>(Folder));
	}
	Data->SetArrayField(TEXT("folders"), FoldersArray);

	return FBlueprintActionResult::Success(
		FString::Printf(TEXT("Project has %d assets across %d folders"), AllAssets.Num(), Folders.Num()),
		Data);
}
