// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#include "Actions/GetOpenBlueprintAction.h"

#include "Engine/Blueprint.h"
#include "Subsystems/AssetEditorSubsystem.h"
#include "Editor.h"

TSharedPtr<FJsonObject> FGetOpenBlueprintAction::GetParameterSchema() const
{
	TSharedPtr<FJsonObject> Schema = MakeShared<FJsonObject>();
	Schema->SetStringField(TEXT("type"), TEXT("object"));
	Schema->SetObjectField(TEXT("properties"), MakeShared<FJsonObject>());
	// No required parameters
	return Schema;
}

FBlueprintActionResult FGetOpenBlueprintAction::Execute(const TSharedPtr<FJsonObject>& Params)
{
	if (!GEditor)
	{
		return FBlueprintActionResult::Failure(TEXT("Editor not available"));
	}

	UAssetEditorSubsystem* AssetEditorSubsystem = GEditor->GetEditorSubsystem<UAssetEditorSubsystem>();
	if (!AssetEditorSubsystem)
	{
		return FBlueprintActionResult::Failure(TEXT("AssetEditorSubsystem not available"));
	}

	TArray<UObject*> EditedAssets = AssetEditorSubsystem->GetAllEditedAssets();

	TArray<TSharedPtr<FJsonValue>> BlueprintArray;
	for (UObject* Asset : EditedAssets)
	{
		UBlueprint* BP = Cast<UBlueprint>(Asset);
		if (!BP) continue;

		TSharedPtr<FJsonObject> BPObj = MakeShared<FJsonObject>();
		BPObj->SetStringField(TEXT("name"), BP->GetName());
		BPObj->SetStringField(TEXT("path"), BP->GetPathName());

		// Include parent class info
		if (BP->ParentClass)
		{
			BPObj->SetStringField(TEXT("parent_class"), BP->ParentClass->GetName());
		}

		BlueprintArray.Add(MakeShared<FJsonValueObject>(BPObj));
	}

	if (BlueprintArray.Num() == 0)
	{
		TSharedPtr<FJsonObject> Data = MakeShared<FJsonObject>();
		Data->SetNumberField(TEXT("count"), 0);
		return FBlueprintActionResult::Success(
			TEXT("No Blueprints currently open in the editor. Use open_blueprint to open one, or list_assets to find available Blueprints."),
			Data);
	}

	TSharedPtr<FJsonObject> Data = MakeShared<FJsonObject>();
	Data->SetNumberField(TEXT("count"), BlueprintArray.Num());
	Data->SetArrayField(TEXT("blueprints"), BlueprintArray);

	// Build a readable list for the message
	FString NameList;
	for (int32 i = 0; i < BlueprintArray.Num(); ++i)
	{
		if (i > 0) NameList += TEXT(", ");
		NameList += BlueprintArray[i]->AsObject()->GetStringField(TEXT("name"));
	}

	return FBlueprintActionResult::Success(
		FString::Printf(TEXT("Found %d open Blueprint(s): %s"), BlueprintArray.Num(), *NameList),
		Data);
}
