// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#include "Actions/GetComponentPropertiesAction.h"

#include "Engine/Blueprint.h"
#include "Engine/SCS_Node.h"
#include "Engine/SimpleConstructionScript.h"
#include "AssetRegistry/AssetRegistryModule.h"
#include "UObject/UnrealType.h"
#include "UObject/EnumProperty.h"
#include "GameFramework/Actor.h"
#include "Components/ActorComponent.h"

// ── Helpers ──────────────────────────────────────────────────────────────────

static UBlueprint* FindBPForCompProps(const FString& NameOrPath)
{
	if (NameOrPath.StartsWith(TEXT("/")))
	{
		return LoadObject<UBlueprint>(nullptr, *NameOrPath);
	}
	FAssetRegistryModule& AssetReg = FModuleManager::LoadModuleChecked<FAssetRegistryModule>("AssetRegistry");
	TArray<FAssetData> Assets;
	AssetReg.Get().GetAssetsByClass(UBlueprint::StaticClass()->GetClassPathName(), Assets);
	for (const FAssetData& Asset : Assets)
	{
		if (Asset.AssetName.ToString().Equals(NameOrPath, ESearchCase::IgnoreCase))
		{
			return Cast<UBlueprint>(Asset.GetAsset());
		}
	}
	return nullptr;
}

static UActorComponent* FindComponentForProps(UBlueprint* BP, const FString& CompName)
{
	// Try SCS (Blueprint-added components) first
	if (BP->SimpleConstructionScript)
	{
		for (USCS_Node* Node : BP->SimpleConstructionScript->GetAllNodes())
		{
			if (Node->GetVariableName().ToString().Equals(CompName, ESearchCase::IgnoreCase))
			{
				return Node->ComponentTemplate;
			}
			if (Node->ComponentTemplate &&
				Node->ComponentTemplate->GetName().Equals(CompName, ESearchCase::IgnoreCase))
			{
				return Node->ComponentTemplate;
			}
		}
	}

	// Try inherited components from parent class CDO
	if (BP->GeneratedClass)
	{
		AActor* CDO = Cast<AActor>(BP->GeneratedClass->GetDefaultObject());
		if (CDO)
		{
			// Match by property name (e.g., "CharacterMovement", "CapsuleComponent")
			for (TFieldIterator<FObjectPropertyBase> It(CDO->GetClass()); It; ++It)
			{
				FObjectPropertyBase* Prop = *It;
				if (Prop->PropertyClass && Prop->PropertyClass->IsChildOf(UActorComponent::StaticClass()))
				{
					if (Prop->GetName().Equals(CompName, ESearchCase::IgnoreCase))
					{
						UObject* Obj = Prop->GetObjectPropertyValue(Prop->ContainerPtrToValuePtr<void>(CDO));
						if (UActorComponent* Comp = Cast<UActorComponent>(Obj))
						{
							return Comp;
						}
					}
				}
			}

			// Match by instance name
			TInlineComponentArray<UActorComponent*> Components;
			CDO->GetComponents(Components);
			for (UActorComponent* Comp : Components)
			{
				if (Comp->GetName().Equals(CompName, ESearchCase::IgnoreCase))
				{
					return Comp;
				}
			}

			// Match by class name (with/without "Component" suffix)
			for (UActorComponent* Comp : Components)
			{
				FString ClassName = Comp->GetClass()->GetName();
				if (ClassName.Equals(CompName, ESearchCase::IgnoreCase))
				{
					return Comp;
				}
				FString ShortName = ClassName;
				ShortName.RemoveFromEnd(TEXT("Component"));
				if (!ShortName.IsEmpty() && ShortName.Equals(CompName, ESearchCase::IgnoreCase))
				{
					return Comp;
				}
			}
		}
	}

	return nullptr;
}

// ── Schema ───────────────────────────────────────────────────────────────────

TSharedPtr<FJsonObject> FGetComponentPropertiesAction::GetParameterSchema() const
{
	auto MakeStr = [](const FString& Desc) -> TSharedPtr<FJsonObject> {
		TSharedPtr<FJsonObject> P = MakeShared<FJsonObject>();
		P->SetStringField(TEXT("type"), TEXT("string"));
		P->SetStringField(TEXT("description"), Desc);
		return P;
	};
	auto MakeInt = [](const FString& Desc, int32 Default) -> TSharedPtr<FJsonObject> {
		TSharedPtr<FJsonObject> P = MakeShared<FJsonObject>();
		P->SetStringField(TEXT("type"), TEXT("integer"));
		P->SetStringField(TEXT("description"), Desc);
		P->SetNumberField(TEXT("default"), Default);
		return P;
	};

	TSharedPtr<FJsonObject> Schema = MakeShared<FJsonObject>();
	Schema->SetStringField(TEXT("type"), TEXT("object"));

	TSharedPtr<FJsonObject> Props = MakeShared<FJsonObject>();
	Props->SetObjectField(TEXT("blueprint"), MakeStr(TEXT("Blueprint name or content path")));
	Props->SetObjectField(TEXT("component_name"), MakeStr(TEXT("Component name (e.g., 'CharacterMovement', 'CapsuleComponent', 'Mesh', 'CameraBoom', 'FollowCamera')")));
	Props->SetObjectField(TEXT("name_filter"), MakeStr(TEXT("Optional: filter properties by name (case-insensitive substring match). E.g., 'Jump' returns JumpZVelocity, CanJump, etc.")));
	Props->SetObjectField(TEXT("max_results"), MakeInt(TEXT("Maximum number of properties to return (default: 20)"), 20));

	Schema->SetObjectField(TEXT("properties"), Props);

	TArray<TSharedPtr<FJsonValue>> Required;
	Required.Add(MakeShared<FJsonValueString>(TEXT("blueprint")));
	Required.Add(MakeShared<FJsonValueString>(TEXT("component_name")));
	Schema->SetArrayField(TEXT("required"), Required);

	return Schema;
}

// ── Execute ──────────────────────────────────────────────────────────────────

FBlueprintActionResult FGetComponentPropertiesAction::Execute(const TSharedPtr<FJsonObject>& Params)
{
	const FString BPName = Params->GetStringField(TEXT("blueprint"));
	const FString CompName = Params->GetStringField(TEXT("component_name"));
	const FString NameFilter = Params->HasField(TEXT("name_filter")) ? Params->GetStringField(TEXT("name_filter")) : TEXT("");
	const int32 MaxResults = Params->HasField(TEXT("max_results")) ? static_cast<int32>(Params->GetNumberField(TEXT("max_results"))) : 20;

	UBlueprint* BP = FindBPForCompProps(BPName);
	if (!BP)
	{
		return FBlueprintActionResult::Failure(FString::Printf(TEXT("Blueprint '%s' not found"), *BPName));
	}

	UActorComponent* Component = FindComponentForProps(BP, CompName);
	if (!Component)
	{
		// Build available list
		FString Available;
		TSet<FString> Listed;
		if (BP->SimpleConstructionScript)
		{
			for (USCS_Node* N : BP->SimpleConstructionScript->GetAllNodes())
			{
				FString Name = N->GetVariableName().ToString();
				Available += Name + TEXT(", ");
				Listed.Add(Name.ToLower());
			}
		}
		if (BP->GeneratedClass)
		{
			AActor* CDO = Cast<AActor>(BP->GeneratedClass->GetDefaultObject());
			if (CDO)
			{
				for (TFieldIterator<FObjectPropertyBase> It(CDO->GetClass()); It; ++It)
				{
					FObjectPropertyBase* Prop = *It;
					if (Prop->PropertyClass && Prop->PropertyClass->IsChildOf(UActorComponent::StaticClass()))
					{
						FString InheritedName = Prop->GetName();
						if (!Listed.Contains(InheritedName.ToLower()))
						{
							UObject* Obj = Prop->GetObjectPropertyValue(Prop->ContainerPtrToValuePtr<void>(CDO));
							if (Obj)
							{
								Available += InheritedName + TEXT(" (inherited), ");
								Listed.Add(InheritedName.ToLower());
							}
						}
					}
				}
			}
		}
		return FBlueprintActionResult::Failure(
			FString::Printf(TEXT("Component '%s' not found in '%s'. Available: [%s]"),
				*CompName, *BPName, *Available));
	}

	// Enumerate properties
	TArray<TSharedPtr<FJsonValue>> PropsArray;
	int32 TotalMatched = 0;

	for (TFieldIterator<FProperty> It(Component->GetClass()); It; ++It)
	{
		FProperty* Property = *It;

		// Only show editable/visible properties
		if (!Property->HasAnyPropertyFlags(CPF_Edit | CPF_BlueprintVisible))
		{
			continue;
		}

		FString PropName = Property->GetName();

		// Apply name filter
		if (!NameFilter.IsEmpty() && !PropName.Contains(NameFilter, ESearchCase::IgnoreCase))
		{
			continue;
		}

		TotalMatched++;
		if (PropsArray.Num() >= MaxResults)
		{
			continue; // Count total but don't add beyond limit
		}

		TSharedPtr<FJsonObject> PropObj = MakeShared<FJsonObject>();
		PropObj->SetStringField(TEXT("name"), PropName);

		// Get type name
		FString TypeName = Property->GetCPPType();
		if (TypeName.IsEmpty())
		{
			TypeName = Property->GetClass()->GetName();
		}
		PropObj->SetStringField(TEXT("type"), TypeName);

		// Try to get current value as string
		void* ValuePtr = Property->ContainerPtrToValuePtr<void>(Component);
		FString ValueStr;
		Property->ExportTextItem_Direct(ValueStr, ValuePtr, nullptr, Component, PPF_None);
		if (!ValueStr.IsEmpty() && ValueStr.Len() <= 200)
		{
			PropObj->SetStringField(TEXT("value"), ValueStr);
		}

		// Add category if available
		FString Category = Property->GetMetaData(TEXT("Category"));
		if (!Category.IsEmpty())
		{
			PropObj->SetStringField(TEXT("category"), Category);
		}

		PropsArray.Add(MakeShared<FJsonValueObject>(PropObj));
	}

	TSharedPtr<FJsonObject> Data = MakeShared<FJsonObject>();
	Data->SetStringField(TEXT("blueprint"), BP->GetName());
	Data->SetStringField(TEXT("component"), CompName);
	Data->SetStringField(TEXT("component_class"), Component->GetClass()->GetName());
	if (!NameFilter.IsEmpty())
	{
		Data->SetStringField(TEXT("filter"), NameFilter);
	}
	Data->SetNumberField(TEXT("total_matched"), TotalMatched);
	Data->SetNumberField(TEXT("returned"), PropsArray.Num());
	Data->SetArrayField(TEXT("properties"), PropsArray);

	FString Message = FString::Printf(TEXT("Found %d propert%s on '%s' (%s) in '%s'%s, showing %d"),
		TotalMatched,
		TotalMatched == 1 ? TEXT("y") : TEXT("ies"),
		*CompName,
		*Component->GetClass()->GetName(),
		*BP->GetName(),
		NameFilter.IsEmpty() ? TEXT("") : *FString::Printf(TEXT(" matching '%s'"), *NameFilter),
		PropsArray.Num());

	return FBlueprintActionResult::Success(Message, Data);
}
