// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#include "Actions/GetClassFunctionsAction.h"

#include "UObject/UObjectIterator.h"
#include "UObject/UnrealType.h"
#include "Engine/Blueprint.h"
#include "EdGraphSchema_K2.h"

// ── Helpers ──────────────────────────────────────────────────────────────────

static UClass* FindClassByName(const FString& ClassName)
{
	// Try exact match, then with common prefixes
	TArray<FString> Candidates;
	Candidates.Add(ClassName);
	Candidates.Add(TEXT("U") + ClassName);
	Candidates.Add(TEXT("A") + ClassName);

	for (TObjectIterator<UClass> It; It; ++It)
	{
		FString Name = (*It)->GetName();
		for (const FString& Candidate : Candidates)
		{
			if (Name.Equals(Candidate, ESearchCase::IgnoreCase))
			{
				return *It;
			}
		}
	}
	return nullptr;
}

static FString GetPinCategoryName(const FProperty* Prop)
{
	if (!Prop) return TEXT("unknown");

	if (CastField<FBoolProperty>(Prop)) return TEXT("bool");
	if (CastField<FIntProperty>(Prop)) return TEXT("int");
	if (CastField<FInt64Property>(Prop)) return TEXT("int64");
	if (CastField<FFloatProperty>(Prop)) return TEXT("float");
	if (CastField<FDoubleProperty>(Prop)) return TEXT("double");
	if (CastField<FStrProperty>(Prop)) return TEXT("string");
	if (CastField<FNameProperty>(Prop)) return TEXT("name");
	if (CastField<FTextProperty>(Prop)) return TEXT("text");
	if (CastField<FByteProperty>(Prop)) return TEXT("byte");

	if (const FStructProperty* StructProp = CastField<FStructProperty>(Prop))
	{
		return StructProp->Struct->GetName();
	}
	if (const FObjectPropertyBase* ObjProp = CastField<FObjectPropertyBase>(Prop))
	{
		return ObjProp->PropertyClass ? ObjProp->PropertyClass->GetName() : TEXT("object");
	}
	if (const FArrayProperty* ArrayProp = CastField<FArrayProperty>(Prop))
	{
		return FString::Printf(TEXT("array<%s>"), *GetPinCategoryName(ArrayProp->Inner));
	}
	if (CastField<FEnumProperty>(Prop)) return TEXT("enum");

	return TEXT("other");
}

// ── Schema ───────────────────────────────────────────────────────────────────

TSharedPtr<FJsonObject> FGetClassFunctionsAction::GetParameterSchema() const
{
	auto MakeStr = [](const FString& Desc) -> TSharedPtr<FJsonObject> {
		TSharedPtr<FJsonObject> P = MakeShared<FJsonObject>();
		P->SetStringField(TEXT("type"), TEXT("string"));
		P->SetStringField(TEXT("description"), Desc);
		return P;
	};
	auto MakeBool = [](const FString& Desc) -> TSharedPtr<FJsonObject> {
		TSharedPtr<FJsonObject> P = MakeShared<FJsonObject>();
		P->SetStringField(TEXT("type"), TEXT("boolean"));
		P->SetStringField(TEXT("description"), Desc);
		return P;
	};
	auto MakeInt = [](const FString& Desc) -> TSharedPtr<FJsonObject> {
		TSharedPtr<FJsonObject> P = MakeShared<FJsonObject>();
		P->SetStringField(TEXT("type"), TEXT("integer"));
		P->SetStringField(TEXT("description"), Desc);
		return P;
	};

	TSharedPtr<FJsonObject> Schema = MakeShared<FJsonObject>();
	Schema->SetStringField(TEXT("type"), TEXT("object"));

	TSharedPtr<FJsonObject> Props = MakeShared<FJsonObject>();
	Props->SetObjectField(TEXT("class_name"), MakeStr(TEXT("Class name (e.g. 'Actor', 'KismetMathLibrary', 'StaticMeshComponent', 'GameplayStatics')")));
	Props->SetObjectField(TEXT("include_inherited"), MakeBool(TEXT("If true, include functions from parent classes. Default: false (only functions defined on this class).")));
	Props->SetObjectField(TEXT("name_filter"), MakeStr(TEXT("Filter functions by name (case-insensitive substring match, e.g. 'Rotate', 'Location')")));
	Props->SetObjectField(TEXT("max_results"), MakeInt(TEXT("Maximum number of functions to return (default: 30)")));

	Schema->SetObjectField(TEXT("properties"), Props);

	TArray<TSharedPtr<FJsonValue>> Required;
	Required.Add(MakeShared<FJsonValueString>(TEXT("class_name")));
	Schema->SetArrayField(TEXT("required"), Required);

	return Schema;
}

// ── Execute ──────────────────────────────────────────────────────────────────

FBlueprintActionResult FGetClassFunctionsAction::Execute(const TSharedPtr<FJsonObject>& Params)
{
	const FString ClassName = Params->GetStringField(TEXT("class_name"));
	const bool bIncludeInherited = Params->HasField(TEXT("include_inherited")) && Params->GetBoolField(TEXT("include_inherited"));
	const FString NameFilter = Params->HasField(TEXT("name_filter"))
		? Params->GetStringField(TEXT("name_filter"))
		: TEXT("");
	const int32 MaxResults = Params->HasField(TEXT("max_results"))
		? (int32)Params->GetNumberField(TEXT("max_results"))
		: 30;

	UClass* Class = FindClassByName(ClassName);
	if (!Class)
	{
		return FBlueprintActionResult::Failure(
			FString::Printf(TEXT("Class '%s' not found. Try: Actor, Pawn, Character, GameplayStatics, "
				"KismetMathLibrary, KismetSystemLibrary, StaticMeshComponent"), *ClassName));
	}

	TArray<TSharedPtr<FJsonValue>> FuncsArray;
	int32 Count = 0;

	for (TFieldIterator<UFunction> It(Class, bIncludeInherited ? EFieldIteratorFlags::IncludeSuper : EFieldIteratorFlags::ExcludeSuper); It; ++It)
	{
		UFunction* Func = *It;
		if (!Func) continue;

		// Only include Blueprint-callable functions
		if (!Func->HasAnyFunctionFlags(FUNC_BlueprintCallable | FUNC_BlueprintPure))
			continue;

		// Skip editor-only, deprecated, and delegate functions
		if (Func->HasAnyFunctionFlags(FUNC_EditorOnly | FUNC_Delegate))
			continue;

		FString FuncName = Func->GetName();

		// Apply name filter
		if (!NameFilter.IsEmpty() && !FuncName.Contains(NameFilter, ESearchCase::IgnoreCase))
			continue;

		// Build function info
		TSharedPtr<FJsonObject> FuncObj = MakeShared<FJsonObject>();
		FuncObj->SetStringField(TEXT("name"), FuncName);
		FuncObj->SetStringField(TEXT("class"), Func->GetOwnerClass()->GetName());

		if (Func->HasAnyFunctionFlags(FUNC_Static))
			FuncObj->SetBoolField(TEXT("is_static"), true);
		if (Func->HasAnyFunctionFlags(FUNC_BlueprintPure))
			FuncObj->SetBoolField(TEXT("is_pure"), true);

		// Build parameters
		TArray<TSharedPtr<FJsonValue>> ParamsArray;
		FString ReturnType;

		for (TFieldIterator<FProperty> PropIt(Func); PropIt; ++PropIt)
		{
			FProperty* Prop = *PropIt;
			if (Prop->HasAnyPropertyFlags(CPF_ReturnParm))
			{
				ReturnType = GetPinCategoryName(Prop);
				continue;
			}

			TSharedPtr<FJsonObject> ParamObj = MakeShared<FJsonObject>();
			ParamObj->SetStringField(TEXT("name"), Prop->GetName());
			ParamObj->SetStringField(TEXT("type"), GetPinCategoryName(Prop));
			if (Prop->HasAnyPropertyFlags(CPF_OutParm))
				ParamObj->SetBoolField(TEXT("is_out"), true);

			ParamsArray.Add(MakeShared<FJsonValueObject>(ParamObj));
		}

		if (ParamsArray.Num() > 0)
			FuncObj->SetArrayField(TEXT("params"), ParamsArray);
		if (!ReturnType.IsEmpty())
			FuncObj->SetStringField(TEXT("return_type"), ReturnType);

		FuncsArray.Add(MakeShared<FJsonValueObject>(FuncObj));

		if (++Count >= MaxResults)
			break;
	}

	TSharedPtr<FJsonObject> Data = MakeShared<FJsonObject>();
	Data->SetStringField(TEXT("class"), Class->GetName());
	Data->SetNumberField(TEXT("count"), FuncsArray.Num());
	Data->SetArrayField(TEXT("functions"), FuncsArray);

	return FBlueprintActionResult::Success(
		FString::Printf(TEXT("Found %d callable functions on '%s'"), FuncsArray.Num(), *Class->GetName()),
		Data);
}
