// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#include "Actions/GetBlueprintInfoAction.h"
#include "Actions/ActionRegistry.h"

#include "Engine/Blueprint.h"
#include "EdGraph/EdGraph.h"
#include "EdGraph/EdGraphNode.h"
#include "EdGraph/EdGraphPin.h"
#include "EdGraphSchema_K2.h"
#include "Kismet2/BlueprintEditorUtils.h"
#include "AssetRegistry/AssetRegistryModule.h"
#include "Engine/BlueprintGeneratedClass.h"
#include "Engine/SCS_Node.h"
#include "Engine/SimpleConstructionScript.h"
#include "GameFramework/Actor.h"
#include "UObject/UnrealType.h"

static UBlueprint* FindBPForInfo(const FString& NameOrPath)
{
	if (NameOrPath.StartsWith(TEXT("/")))
	{
		return LoadObject<UBlueprint>(nullptr, *NameOrPath);
	}
	FAssetRegistryModule& AssetReg = FModuleManager::LoadModuleChecked<FAssetRegistryModule>("AssetRegistry");
	TArray<FAssetData> Assets;
	AssetReg.Get().GetAssetsByClass(UBlueprint::StaticClass()->GetClassPathName(), Assets);
	for (const FAssetData& Asset : Assets)
	{
		if (Asset.AssetName.ToString().Equals(NameOrPath, ESearchCase::IgnoreCase))
		{
			return Cast<UBlueprint>(Asset.GetAsset());
		}
	}
	return nullptr;
}

TSharedPtr<FJsonObject> FGetBlueprintInfoAction::GetParameterSchema() const
{
	auto MakeStr = [](const FString& Desc) -> TSharedPtr<FJsonObject> {
		TSharedPtr<FJsonObject> P = MakeShared<FJsonObject>();
		P->SetStringField(TEXT("type"), TEXT("string"));
		P->SetStringField(TEXT("description"), Desc);
		return P;
	};
	auto MakeBool = [](const FString& Desc) -> TSharedPtr<FJsonObject> {
		TSharedPtr<FJsonObject> P = MakeShared<FJsonObject>();
		P->SetStringField(TEXT("type"), TEXT("boolean"));
		P->SetStringField(TEXT("description"), Desc);
		return P;
	};

	TSharedPtr<FJsonObject> Schema = MakeShared<FJsonObject>();
	Schema->SetStringField(TEXT("type"), TEXT("object"));

	TSharedPtr<FJsonObject> Props = MakeShared<FJsonObject>();
	Props->SetObjectField(TEXT("blueprint"), MakeStr(TEXT("Blueprint name or content path")));
	Props->SetObjectField(TEXT("include_nodes"), MakeBool(TEXT("If true, includes all nodes and their connections for each graph. Default: false.")));
	Props->SetObjectField(TEXT("graph_name"), MakeStr(TEXT("If specified, only return info for this graph")));

	Schema->SetObjectField(TEXT("properties"), Props);

	TArray<TSharedPtr<FJsonValue>> Required;
	Required.Add(MakeShared<FJsonValueString>(TEXT("blueprint")));
	Schema->SetArrayField(TEXT("required"), Required);

	return Schema;
}

FBlueprintActionResult FGetBlueprintInfoAction::Execute(const TSharedPtr<FJsonObject>& Params)
{
	const FString BPName = Params->GetStringField(TEXT("blueprint"));
	const bool bIncludeNodes = Params->HasField(TEXT("include_nodes")) && Params->GetBoolField(TEXT("include_nodes"));
	const FString FilterGraph = Params->HasField(TEXT("graph_name")) ? Params->GetStringField(TEXT("graph_name")) : TEXT("");

	UBlueprint* BP = FindBPForInfo(BPName);
	if (!BP) return FBlueprintActionResult::Failure(FString::Printf(TEXT("Blueprint '%s' not found"), *BPName));

	TSharedPtr<FJsonObject> Data = MakeShared<FJsonObject>();
	Data->SetStringField(TEXT("name"), BP->GetName());
	Data->SetStringField(TEXT("path"), BP->GetPathName());
	Data->SetStringField(TEXT("parent_class"), BP->ParentClass ? BP->ParentClass->GetName() : TEXT("Unknown"));
	Data->SetStringField(TEXT("status"), BP->Status == BS_Error ? TEXT("Error") : TEXT("UpToDate"));

	// Only include variables when NOT requesting a specific graph
	// (the AI already has them from the initial blueprint info call)
	if (FilterGraph.IsEmpty())
	{
		TArray<TSharedPtr<FJsonValue>> VarsArray;
		for (const FBPVariableDescription& Var : BP->NewVariables)
		{
			TSharedPtr<FJsonObject> VarObj = MakeShared<FJsonObject>();
			VarObj->SetStringField(TEXT("name"), Var.VarName.ToString());
			VarObj->SetStringField(TEXT("type"), Var.VarType.PinCategory.ToString());
			if (!Var.DefaultValue.IsEmpty())
			{
				VarObj->SetStringField(TEXT("default"), Var.DefaultValue);
			}
			VarsArray.Add(MakeShared<FJsonValueObject>(VarObj));
		}
		Data->SetArrayField(TEXT("variables"), VarsArray);

		// Components (SCS + inherited from parent class)
		TArray<TSharedPtr<FJsonValue>> CompsArray;

		// SCS components (added in this Blueprint)
		if (BP->SimpleConstructionScript)
		{
			for (USCS_Node* Node : BP->SimpleConstructionScript->GetAllNodes())
			{
				TSharedPtr<FJsonObject> CompObj = MakeShared<FJsonObject>();
				CompObj->SetStringField(TEXT("name"), Node->GetVariableName().ToString());
				if (Node->ComponentTemplate)
				{
					CompObj->SetStringField(TEXT("class"), Node->ComponentTemplate->GetClass()->GetName());
				}
				CompObj->SetStringField(TEXT("source"), TEXT("blueprint"));
				CompsArray.Add(MakeShared<FJsonValueObject>(CompObj));
			}
		}

		// Inherited components from parent class (e.g., CharacterMovement, CapsuleComponent, Mesh)
		if (BP->GeneratedClass)
		{
			AActor* CDO = Cast<AActor>(BP->GeneratedClass->GetDefaultObject());
			if (CDO)
			{
				TSet<FString> SCSNames;
				if (BP->SimpleConstructionScript)
				{
					for (USCS_Node* Node : BP->SimpleConstructionScript->GetAllNodes())
					{
						SCSNames.Add(Node->GetVariableName().ToString().ToLower());
					}
				}

				for (TFieldIterator<FObjectPropertyBase> It(CDO->GetClass()); It; ++It)
				{
					FObjectPropertyBase* Prop = *It;
					if (!Prop->PropertyClass || !Prop->PropertyClass->IsChildOf(UActorComponent::StaticClass()))
					{
						continue;
					}

					FString CompPropName = Prop->GetName();
					if (SCSNames.Contains(CompPropName.ToLower()))
					{
						continue;
					}

					UObject* Obj = Prop->GetObjectPropertyValue(Prop->ContainerPtrToValuePtr<void>(CDO));
					UActorComponent* Comp = Cast<UActorComponent>(Obj);
					if (!Comp) continue;

					TSharedPtr<FJsonObject> CompObj = MakeShared<FJsonObject>();
					CompObj->SetStringField(TEXT("name"), CompPropName);
					CompObj->SetStringField(TEXT("class"), Comp->GetClass()->GetName());
					CompObj->SetStringField(TEXT("source"), TEXT("inherited"));
					CompsArray.Add(MakeShared<FJsonValueObject>(CompObj));
				}
			}
		}

		Data->SetArrayField(TEXT("components"), CompsArray);
	}

	// Graphs
	TArray<UEdGraph*> AllGraphs;
	BP->GetAllGraphs(AllGraphs);

	TArray<TSharedPtr<FJsonValue>> GraphsArray;
	for (UEdGraph* Graph : AllGraphs)
	{
		if (!FilterGraph.IsEmpty() && !Graph->GetName().Equals(FilterGraph, ESearchCase::IgnoreCase))
		{
			continue;
		}

		TSharedPtr<FJsonObject> GraphObj = MakeShared<FJsonObject>();
		GraphObj->SetStringField(TEXT("name"), Graph->GetName());
		GraphObj->SetNumberField(TEXT("node_count"), Graph->Nodes.Num());

		if (bIncludeNodes)
		{
			// Cap nodes to keep output manageable (49-node graph was producing ~16K chars)
			static constexpr int32 MaxNodesReturned = 20;
			TArray<TSharedPtr<FJsonValue>> NodesArray;
			int32 NodeCount = 0;
			for (UEdGraphNode* Node : Graph->Nodes)
			{
				if (NodeCount >= MaxNodesReturned)
				{
					TSharedPtr<FJsonObject> OverflowObj = MakeShared<FJsonObject>();
					OverflowObj->SetStringField(TEXT("note"),
						FString::Printf(TEXT("...and %d more nodes (omitted to save space)"), Graph->Nodes.Num() - MaxNodesReturned));
					NodesArray.Add(MakeShared<FJsonValueObject>(OverflowObj));
					break;
				}
				NodeCount++;

				TSharedPtr<FJsonObject> NodeObj = MakeShared<FJsonObject>();

				// Track the node and get an ID
				FString NodeId = Registry->TrackNode(Node);
				NodeObj->SetStringField(TEXT("node_id"), NodeId);
				NodeObj->SetStringField(TEXT("title"), Node->GetNodeTitle(ENodeTitleType::FullTitle).ToString());
				NodeObj->SetStringField(TEXT("class"), Node->GetClass()->GetName());
				NodeObj->SetNumberField(TEXT("pos_x"), Node->NodePosX);
				NodeObj->SetNumberField(TEXT("pos_y"), Node->NodePosY);

				// Pins — compact: only show non-hidden pins, skip SubCategory and full type details
				TArray<TSharedPtr<FJsonValue>> PinsArray;
				for (UEdGraphPin* Pin : Node->Pins)
				{
					if (Pin->bHidden) continue;

					TSharedPtr<FJsonObject> PinObj = MakeShared<FJsonObject>();
					PinObj->SetStringField(TEXT("name"), Pin->PinName.ToString());
					PinObj->SetStringField(TEXT("dir"), Pin->Direction == EGPD_Input ? TEXT("In") : TEXT("Out"));

					// Only include connections (saves space over listing empty arrays)
					if (Pin->LinkedTo.Num() > 0)
					{
						TArray<TSharedPtr<FJsonValue>> Links;
						for (UEdGraphPin* Link : Pin->LinkedTo)
						{
							FString LinkedNodeId = Registry->TrackNode(Link->GetOwningNode());
							Links.Add(MakeShared<FJsonValueString>(
								FString::Printf(TEXT("%s.%s"), *LinkedNodeId, *Link->PinName.ToString())));
						}
						PinObj->SetArrayField(TEXT("linked"), Links);
					}

					PinsArray.Add(MakeShared<FJsonValueObject>(PinObj));
				}
				NodeObj->SetArrayField(TEXT("pins"), PinsArray);

				NodesArray.Add(MakeShared<FJsonValueObject>(NodeObj));
			}
			GraphObj->SetArrayField(TEXT("nodes"), NodesArray);
		}

		GraphsArray.Add(MakeShared<FJsonValueObject>(GraphObj));
	}
	Data->SetArrayField(TEXT("graphs"), GraphsArray);

	// Function list
	TArray<TSharedPtr<FJsonValue>> FuncsArray;
	for (UEdGraph* FGraph : BP->FunctionGraphs)
	{
		FuncsArray.Add(MakeShared<FJsonValueString>(FGraph->GetName()));
	}
	Data->SetArrayField(TEXT("functions"), FuncsArray);

	return FBlueprintActionResult::Success(
		FString::Printf(TEXT("Blueprint '%s' info retrieved"), *BPName),
		Data);
}
