// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#include "Actions/DisconnectNodesAction.h"
#include "Actions/ActionRegistry.h"

#include "EdGraph/EdGraphNode.h"
#include "EdGraph/EdGraphPin.h"
#include "EdGraphSchema_K2.h"
#include "Kismet2/BlueprintEditorUtils.h"
#include "Engine/Blueprint.h"

// ── Pin fuzzy matching (same logic as ConnectNodesAction) ────────────────────

static FString NormalizePinNameDisc(const FString& Name)
{
	return Name.ToLower().Replace(TEXT(" "), TEXT("")).Replace(TEXT("_"), TEXT(""));
}

static UEdGraphPin* FindPinFuzzyDisconnect(UEdGraphNode* Node, const FString& PinName, EEdGraphPinDirection Direction)
{
	// 0. Common AI aliases
	static TMap<FString, FString> ExecAliases = {
		{TEXT("true"), TEXT("then")},
		{TEXT("false"), TEXT("else")},
		{TEXT("exec"), TEXT("execute")},
	};

	FString AliasLower = PinName.ToLower();
	if (const FString* Alias = ExecAliases.Find(AliasLower))
	{
		if (UEdGraphPin* Pin = Node->FindPin(FName(**Alias), Direction))
			return Pin;
	}

	// 1. Exact match
	if (UEdGraphPin* Pin = Node->FindPin(FName(*PinName), Direction))
		return Pin;

	// 1b. "execute" → empty-string exec input
	if (Direction == EGPD_Input && AliasLower == TEXT("execute"))
	{
		for (UEdGraphPin* Pin : Node->Pins)
		{
			if (Pin->Direction == Direction && !Pin->bHidden &&
				Pin->PinType.PinCategory == UEdGraphSchema_K2::PC_Exec &&
				Pin->PinName.IsNone())
				return Pin;
		}
	}

	// 2. Normalized comparison (strip spaces + underscores)
	FString Normalized = NormalizePinNameDisc(PinName);
	for (UEdGraphPin* Pin : Node->Pins)
	{
		if (Pin->Direction != Direction || Pin->bHidden) continue;
		if (NormalizePinNameDisc(Pin->PinName.ToString()) == Normalized) return Pin;
	}

	// 3. Boolean prefix matching
	for (UEdGraphPin* Pin : Node->Pins)
	{
		if (Pin->Direction != Direction || Pin->bHidden) continue;
		FString CandidateName = Pin->PinName.ToString();
		if (CandidateName.Equals(TEXT("b") + PinName, ESearchCase::IgnoreCase))
			return Pin;
		if (PinName.StartsWith(TEXT("b"), ESearchCase::CaseSensitive) && PinName.Len() > 1)
		{
			if (CandidateName.Equals(PinName.Mid(1), ESearchCase::IgnoreCase))
				return Pin;
		}
	}

	// 4. Single non-exec pin fallback
	if (AliasLower != TEXT("execute") && AliasLower != TEXT("then") && AliasLower != TEXT("else"))
	{
		UEdGraphPin* OnlyCandidate = nullptr;
		int32 Count = 0;
		for (UEdGraphPin* Pin : Node->Pins)
		{
			if (Pin->Direction != Direction || Pin->bHidden) continue;
			if (Pin->PinType.PinCategory == UEdGraphSchema_K2::PC_Exec) continue;
			OnlyCandidate = Pin;
			Count++;
		}
		if (Count == 1 && OnlyCandidate) return OnlyCandidate;
	}

	return nullptr;
}

// ── Schema ───────────────────────────────────────────────────────────────────

TSharedPtr<FJsonObject> FDisconnectNodesAction::GetParameterSchema() const
{
	auto MakeStr = [](const FString& Desc) -> TSharedPtr<FJsonObject> {
		TSharedPtr<FJsonObject> P = MakeShared<FJsonObject>();
		P->SetStringField(TEXT("type"), TEXT("string"));
		P->SetStringField(TEXT("description"), Desc);
		return P;
	};

	TSharedPtr<FJsonObject> Schema = MakeShared<FJsonObject>();
	Schema->SetStringField(TEXT("type"), TEXT("object"));

	TSharedPtr<FJsonObject> Props = MakeShared<FJsonObject>();
	Props->SetObjectField(TEXT("source_node"), MakeStr(TEXT("Source node ID (e.g. 'N1')")));
	Props->SetObjectField(TEXT("source_pin"), MakeStr(TEXT("Name of the output pin on the source node")));
	Props->SetObjectField(TEXT("target_node"), MakeStr(TEXT("Target node ID (e.g. 'N2')")));
	Props->SetObjectField(TEXT("target_pin"), MakeStr(TEXT("Name of the input pin on the target node")));

	Schema->SetObjectField(TEXT("properties"), Props);

	TArray<TSharedPtr<FJsonValue>> Required;
	Required.Add(MakeShared<FJsonValueString>(TEXT("source_node")));
	Required.Add(MakeShared<FJsonValueString>(TEXT("source_pin")));
	Required.Add(MakeShared<FJsonValueString>(TEXT("target_node")));
	Required.Add(MakeShared<FJsonValueString>(TEXT("target_pin")));
	Schema->SetArrayField(TEXT("required"), Required);

	return Schema;
}

// ── Execute ──────────────────────────────────────────────────────────────────

FBlueprintActionResult FDisconnectNodesAction::Execute(const TSharedPtr<FJsonObject>& Params)
{
	const FString SourceNodeId = Params->GetStringField(TEXT("source_node"));
	const FString SourcePinName = Params->GetStringField(TEXT("source_pin"));
	const FString TargetNodeId = Params->GetStringField(TEXT("target_node"));
	const FString TargetPinName = Params->GetStringField(TEXT("target_pin"));

	// Find nodes
	UEdGraphNode* SourceNode = Registry->FindTrackedNode(SourceNodeId);
	if (!SourceNode)
	{
		return FBlueprintActionResult::Failure(
			FString::Printf(TEXT("Source node '%s' not found"), *SourceNodeId));
	}

	UEdGraphNode* TargetNode = Registry->FindTrackedNode(TargetNodeId);
	if (!TargetNode)
	{
		return FBlueprintActionResult::Failure(
			FString::Printf(TEXT("Target node '%s' not found"), *TargetNodeId));
	}

	// Find pins
	UEdGraphPin* SourcePin = FindPinFuzzyDisconnect(SourceNode, SourcePinName, EGPD_Output);
	if (!SourcePin)
	{
		FString Available;
		for (UEdGraphPin* Pin : SourceNode->Pins)
		{
			if (Pin->Direction == EGPD_Output && !Pin->bHidden)
				Available += Pin->PinName.ToString() + TEXT(", ");
		}
		return FBlueprintActionResult::Failure(
			FString::Printf(TEXT("Output pin '%s' not found on node %s. Available: [%s]"),
				*SourcePinName, *SourceNodeId, *Available));
	}

	UEdGraphPin* TargetPin = FindPinFuzzyDisconnect(TargetNode, TargetPinName, EGPD_Input);
	if (!TargetPin)
	{
		FString Available;
		for (UEdGraphPin* Pin : TargetNode->Pins)
		{
			if (Pin->Direction == EGPD_Input && !Pin->bHidden)
				Available += Pin->PinName.ToString() + TEXT(", ");
		}
		return FBlueprintActionResult::Failure(
			FString::Printf(TEXT("Input pin '%s' not found on node %s. Available: [%s]"),
				*TargetPinName, *TargetNodeId, *Available));
	}

	// Check if they're actually connected
	if (!SourcePin->LinkedTo.Contains(TargetPin))
	{
		return FBlueprintActionResult::Failure(
			FString::Printf(TEXT("Pins %s.%s and %s.%s are not connected"),
				*SourceNodeId, *SourcePinName, *TargetNodeId, *TargetPinName));
	}

	// Break the specific connection
	SourcePin->BreakLinkTo(TargetPin);

	// Mark modified
	UEdGraph* Graph = SourceNode->GetGraph();
	if (Graph)
	{
		Graph->NotifyGraphChanged();
		UBlueprint* BP = FBlueprintEditorUtils::FindBlueprintForGraph(Graph);
		if (BP)
		{
			FBlueprintEditorUtils::MarkBlueprintAsModified(BP);
		}
	}

	TSharedPtr<FJsonObject> Data = MakeShared<FJsonObject>();
	Data->SetStringField(TEXT("source"), FString::Printf(TEXT("%s.%s"), *SourceNodeId, *SourcePinName));
	Data->SetStringField(TEXT("target"), FString::Printf(TEXT("%s.%s"), *TargetNodeId, *TargetPinName));

	return FBlueprintActionResult::Success(
		FString::Printf(TEXT("Disconnected %s.%s from %s.%s"),
			*SourceNodeId, *SourcePinName, *TargetNodeId, *TargetPinName),
		Data);
}
