// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#include "Actions/DeleteNodeAction.h"
#include "Actions/ActionRegistry.h"

#include "EdGraph/EdGraph.h"
#include "EdGraph/EdGraphNode.h"

TSharedPtr<FJsonObject> FDeleteNodeAction::GetParameterSchema() const
{
	TSharedPtr<FJsonObject> Schema = MakeShared<FJsonObject>();
	Schema->SetStringField(TEXT("type"), TEXT("object"));

	TSharedPtr<FJsonObject> Props = MakeShared<FJsonObject>();

	TSharedPtr<FJsonObject> NodeIdProp = MakeShared<FJsonObject>();
	NodeIdProp->SetStringField(TEXT("type"), TEXT("string"));
	NodeIdProp->SetStringField(TEXT("description"), TEXT("The tracked node ID to delete (e.g. 'N1', 'N2')"));
	Props->SetObjectField(TEXT("node_id"), NodeIdProp);

	Schema->SetObjectField(TEXT("properties"), Props);

	TArray<TSharedPtr<FJsonValue>> Required;
	Required.Add(MakeShared<FJsonValueString>(TEXT("node_id")));
	Schema->SetArrayField(TEXT("required"), Required);

	return Schema;
}

FBlueprintActionResult FDeleteNodeAction::Execute(const TSharedPtr<FJsonObject>& Params)
{
	const FString NodeId = Params->GetStringField(TEXT("node_id"));

	UEdGraphNode* Node = Registry->FindTrackedNode(NodeId);
	if (!Node)
	{
		return FBlueprintActionResult::Failure(
			FString::Printf(TEXT("Node '%s' not found or no longer valid"), *NodeId));
	}

	UEdGraph* Graph = Node->GetGraph();
	if (!Graph)
	{
		return FBlueprintActionResult::Failure(TEXT("Node's graph is not valid"));
	}

	FString NodeTitle = Node->GetNodeTitle(ENodeTitleType::FullTitle).ToString();

	// Break all pin connections first
	for (UEdGraphPin* Pin : Node->Pins)
	{
		Pin->BreakAllPinLinks();
	}

	// Remove the node
	Graph->RemoveNode(Node);

	TSharedPtr<FJsonObject> Data = MakeShared<FJsonObject>();
	Data->SetStringField(TEXT("deleted_node_id"), NodeId);
	Data->SetStringField(TEXT("deleted_node_title"), NodeTitle);

	return FBlueprintActionResult::Success(
		FString::Printf(TEXT("Deleted node '%s' (%s)"), *NodeId, *NodeTitle),
		Data);
}
