// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#include "Actions/CreateBlueprintAction.h"
#include "Actions/ActionRegistry.h"

#include "Kismet2/KismetEditorUtilities.h"
#include "Kismet2/BlueprintEditorUtils.h"
#include "Engine/Blueprint.h"
#include "Engine/BlueprintGeneratedClass.h"
#include "AssetRegistry/AssetRegistryModule.h"
#include "AssetToolsModule.h"
#include "Factories/BlueprintFactory.h"
#include "UObject/SavePackage.h"
#include "GameFramework/Actor.h"
#include "GameFramework/Pawn.h"
#include "GameFramework/Character.h"
#include "GameFramework/GameModeBase.h"
#include "GameFramework/PlayerController.h"
#include "GameFramework/HUD.h"
#include "Components/ActorComponent.h"
#include "Components/SceneComponent.h"

TSharedPtr<FJsonObject> FCreateBlueprintAction::GetParameterSchema() const
{
	TSharedPtr<FJsonObject> Schema = MakeShared<FJsonObject>();
	Schema->SetStringField(TEXT("type"), TEXT("object"));

	TSharedPtr<FJsonObject> Props = MakeShared<FJsonObject>();

	// asset_name
	TSharedPtr<FJsonObject> NameProp = MakeShared<FJsonObject>();
	NameProp->SetStringField(TEXT("type"), TEXT("string"));
	NameProp->SetStringField(TEXT("description"), TEXT("Name of the Blueprint asset (e.g. 'BP_HealthSystem')"));
	Props->SetObjectField(TEXT("asset_name"), NameProp);

	// asset_path
	TSharedPtr<FJsonObject> PathProp = MakeShared<FJsonObject>();
	PathProp->SetStringField(TEXT("type"), TEXT("string"));
	PathProp->SetStringField(TEXT("description"), TEXT("Content path where to create (e.g. '/Game/Blueprints'). Defaults to '/Game/Blueprints'."));
	Props->SetObjectField(TEXT("asset_path"), PathProp);

	// parent_class
	TSharedPtr<FJsonObject> ParentProp = MakeShared<FJsonObject>();
	ParentProp->SetStringField(TEXT("type"), TEXT("string"));
	ParentProp->SetStringField(TEXT("description"),
		TEXT("Parent class name. Common values: 'Actor', 'Pawn', 'Character', 'ActorComponent', "
			"'SceneComponent', 'GameModeBase', 'PlayerController', 'HUD', 'Object'. Defaults to 'Actor'."));
	Props->SetObjectField(TEXT("parent_class"), ParentProp);

	Schema->SetObjectField(TEXT("properties"), Props);

	TArray<TSharedPtr<FJsonValue>> Required;
	Required.Add(MakeShared<FJsonValueString>(TEXT("asset_name")));
	Schema->SetArrayField(TEXT("required"), Required);

	return Schema;
}

FBlueprintActionResult FCreateBlueprintAction::Execute(const TSharedPtr<FJsonObject>& Params)
{
	FString AssetName = Params->GetStringField(TEXT("asset_name"));
	FString AssetPath = Params->HasField(TEXT("asset_path"))
		? Params->GetStringField(TEXT("asset_path"))
		: TEXT("/Game/Blueprints");
	FString ParentClassName = Params->HasField(TEXT("parent_class"))
		? Params->GetStringField(TEXT("parent_class"))
		: TEXT("Actor");

	if (AssetName.IsEmpty())
	{
		return FBlueprintActionResult::Failure(TEXT("asset_name is required"));
	}

	// Resolve parent class
	UClass* ParentClass = ResolveParentClass(ParentClassName);
	if (!ParentClass)
	{
		return FBlueprintActionResult::Failure(
			FString::Printf(TEXT("Could not resolve parent class '%s'"), *ParentClassName));
	}

	// Ensure path ends properly
	FString FullPath = AssetPath / AssetName;

	// ── Idempotency check: if a Blueprint with this name already exists, return it ──
	{
		FAssetRegistryModule& ARModule = FModuleManager::LoadModuleChecked<FAssetRegistryModule>("AssetRegistry");
		IAssetRegistry& AR = ARModule.Get();

		FString PackagePath = AssetPath / AssetName;
		// AssetRegistry uses package paths like /Game/Blueprints/BP_Turret
		FAssetData ExistingAsset = AR.GetAssetByObjectPath(FSoftObjectPath(PackagePath + TEXT(".") + AssetName));
		if (!ExistingAsset.IsValid())
		{
			// Try alternate lookup by scanning the folder
			TArray<FAssetData> AssetsInPath;
			AR.GetAssetsByPath(FName(*AssetPath), AssetsInPath, /*bRecursive=*/false);
			for (const FAssetData& Asset : AssetsInPath)
			{
				if (Asset.AssetName.ToString().Equals(AssetName, ESearchCase::IgnoreCase))
				{
					ExistingAsset = Asset;
					break;
				}
			}
		}

		if (ExistingAsset.IsValid())
		{
			UBlueprint* ExistingBP = Cast<UBlueprint>(ExistingAsset.GetAsset());
			if (ExistingBP)
			{
				// Build the same result data as a fresh creation
				TSharedPtr<FJsonObject> ResultData = MakeShared<FJsonObject>();
				ResultData->SetStringField(TEXT("asset_path"), ExistingBP->GetPathName());
				ResultData->SetStringField(TEXT("asset_name"), AssetName);
				ResultData->SetStringField(TEXT("parent_class"), ExistingBP->ParentClass ? ExistingBP->ParentClass->GetName() : TEXT("Unknown"));

				TArray<TSharedPtr<FJsonValue>> GraphNames;
				TArray<UEdGraph*> Graphs;
				ExistingBP->GetAllGraphs(Graphs);
				for (UEdGraph* Graph : Graphs)
				{
					GraphNames.Add(MakeShared<FJsonValueString>(Graph->GetName()));
				}
				ResultData->SetArrayField(TEXT("graphs"), GraphNames);

				// Track this blueprint even though we didn't create it
				if (Registry)
				{
					Registry->TrackCreatedBlueprint(AssetName, ExistingBP->GetPathName());
				}

				return FBlueprintActionResult::Success(
					FString::Printf(TEXT("Blueprint '%s' already exists — returning existing. No duplicate created."), *AssetName),
					ResultData
				);
			}
		}
	}

	// Create the Blueprint using the factory
	UBlueprintFactory* Factory = NewObject<UBlueprintFactory>();
	Factory->ParentClass = ParentClass;

	FAssetToolsModule& AssetToolsModule = FModuleManager::LoadModuleChecked<FAssetToolsModule>("AssetTools");
	UObject* CreatedAsset = AssetToolsModule.Get().CreateAsset(AssetName, AssetPath, UBlueprint::StaticClass(), Factory);

	if (!CreatedAsset)
	{
		return FBlueprintActionResult::Failure(
			FString::Printf(TEXT("Failed to create Blueprint '%s' at '%s'"), *AssetName, *AssetPath));
	}

	UBlueprint* Blueprint = Cast<UBlueprint>(CreatedAsset);
	if (!Blueprint)
	{
		return FBlueprintActionResult::Failure(TEXT("Created asset is not a Blueprint"));
	}

	// Compile initially
	FKismetEditorUtilities::CompileBlueprint(Blueprint);

	// Track the created blueprint in the registry
	if (Registry)
	{
		Registry->TrackCreatedBlueprint(AssetName, Blueprint->GetPathName());
	}

	// Build result
	TSharedPtr<FJsonObject> ResultData = MakeShared<FJsonObject>();
	ResultData->SetStringField(TEXT("asset_path"), Blueprint->GetPathName());
	ResultData->SetStringField(TEXT("asset_name"), AssetName);
	ResultData->SetStringField(TEXT("parent_class"), ParentClass->GetName());

	// List available graphs
	TArray<TSharedPtr<FJsonValue>> GraphNames;
	TArray<UEdGraph*> Graphs;
	Blueprint->GetAllGraphs(Graphs);
	for (UEdGraph* Graph : Graphs)
	{
		GraphNames.Add(MakeShared<FJsonValueString>(Graph->GetName()));
	}
	ResultData->SetArrayField(TEXT("graphs"), GraphNames);

	return FBlueprintActionResult::Success(
		FString::Printf(TEXT("Created Blueprint '%s' with parent class '%s'"), *AssetName, *ParentClass->GetName()),
		ResultData
	);
}

UClass* FCreateBlueprintAction::ResolveParentClass(const FString& ClassName)
{
	// Common class lookup table
	static TMap<FString, UClass*> ClassMap;
	if (ClassMap.Num() == 0)
	{
		ClassMap.Add(TEXT("Actor"), AActor::StaticClass());
		ClassMap.Add(TEXT("Pawn"), APawn::StaticClass());
		ClassMap.Add(TEXT("Character"), ACharacter::StaticClass());
		ClassMap.Add(TEXT("GameModeBase"), AGameModeBase::StaticClass());
		ClassMap.Add(TEXT("PlayerController"), APlayerController::StaticClass());
		ClassMap.Add(TEXT("HUD"), AHUD::StaticClass());
		ClassMap.Add(TEXT("ActorComponent"), UActorComponent::StaticClass());
		ClassMap.Add(TEXT("SceneComponent"), USceneComponent::StaticClass());
		ClassMap.Add(TEXT("Object"), UObject::StaticClass());
	}

	// direct match (case-insensitive)
	for (const auto& Pair : ClassMap)
	{
		if (Pair.Key.Equals(ClassName, ESearchCase::IgnoreCase))
		{
			return Pair.Value;
		}
	}

	// Try finding by iterating loaded classes
	for (TObjectIterator<UClass> It; It; ++It)
	{
		UClass* Cls = *It;
		FString Name = Cls->GetName();
		if (Name.Equals(ClassName, ESearchCase::IgnoreCase) ||
			Name.Equals(TEXT("A") + ClassName, ESearchCase::IgnoreCase) ||
			Name.Equals(TEXT("U") + ClassName, ESearchCase::IgnoreCase))
		{
			return Cls;
		}
	}

	return nullptr;
}
