// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#include "Actions/ConnectNodesAction.h"
#include "Actions/ActionRegistry.h"

#include "EdGraph/EdGraphNode.h"
#include "EdGraph/EdGraphPin.h"
#include "EdGraphSchema_K2.h"
#include "Kismet2/BlueprintEditorUtils.h"
#include "Engine/Blueprint.h"

// ─── Pin fuzzy matching ─────────────────────────────────────────────────────

/** Normalize for comparison: lowercase, strip spaces AND underscores */
static FString NormalizePinName(const FString& Name)
{
	return Name.ToLower().Replace(TEXT(" "), TEXT("")).Replace(TEXT("_"), TEXT(""));
}

/** Try to find a pin with fuzzy matching: aliases, case-insensitive, space/underscore agnostic, boolean prefix */
static UEdGraphPin* FindPinFuzzy(UEdGraphNode* Node, const FString& PinName, EEdGraphPinDirection Direction)
{
	// 0. Common AI aliases — map what AIs typically send to actual UE5 pin names
	static TMap<FString, FString> ExecAliases = {
		// Branch output aliases (AI says "True"/"False", UE5 uses "then"/"else")
		{TEXT("true"), TEXT("then")},
		{TEXT("false"), TEXT("else")},
		// Exec input (AI says "execute", many UE5 nodes use "" empty pin name)
		{TEXT("exec"), TEXT("execute")},
	};

	// Try alias first
	FString AliasLower = PinName.ToLower();
	if (const FString* Alias = ExecAliases.Find(AliasLower))
	{
		if (UEdGraphPin* Pin = Node->FindPin(FName(**Alias), Direction))
		{
			return Pin;
		}
	}

	// 1. Exact match (UE FName comparison)
	if (UEdGraphPin* Pin = Node->FindPin(FName(*PinName), Direction))
	{
		return Pin;
	}

	// 1b. "execute" → try empty-string pin (exec inputs on most K2 nodes)
	if (Direction == EGPD_Input && AliasLower == TEXT("execute"))
	{
		if (UEdGraphPin* Pin = Node->FindPin(FName(NAME_None), Direction))
		{
			if (Pin->PinType.PinCategory == UEdGraphSchema_K2::PC_Exec)
			{
				return Pin;
			}
		}
		// Also search for any unnamed exec input
		for (UEdGraphPin* Pin : Node->Pins)
		{
			if (Pin->Direction == Direction && !Pin->bHidden &&
				Pin->PinType.PinCategory == UEdGraphSchema_K2::PC_Exec &&
				Pin->PinName.IsNone())
			{
				return Pin;
			}
		}
	}

	// 2. Normalize: lowercase + strip ALL spaces and underscores
	FString Normalized = NormalizePinName(PinName);

	for (UEdGraphPin* Pin : Node->Pins)
	{
		if (Pin->Direction != Direction || Pin->bHidden) continue;

		FString CandidateNorm = NormalizePinName(Pin->PinName.ToString());
		if (CandidateNorm == Normalized) return Pin;
	}

	// 3. Boolean prefix: 'Looping' matches 'bLooping', 'MaxOncePerFrame' matches 'bMaxOncePerFrame'
	for (UEdGraphPin* Pin : Node->Pins)
	{
		if (Pin->Direction != Direction || Pin->bHidden) continue;

		FString CandidateName = Pin->PinName.ToString();
		// AI sent 'Looping' but pin is 'bLooping'
		if (CandidateName.Equals(TEXT("b") + PinName, ESearchCase::IgnoreCase))
		{
			return Pin;
		}
		// AI sent 'bLooping' but pin is 'Looping' (reverse)
		if (PinName.StartsWith(TEXT("b"), ESearchCase::CaseSensitive) && PinName.Len() > 1)
		{
			FString WithoutB = PinName.Mid(1);
			if (CandidateName.Equals(WithoutB, ESearchCase::IgnoreCase))
			{
				return Pin;
			}
		}
	}

	// 4. Last resort: if there is exactly ONE non-hidden pin in the requested direction
	//    that is NOT an exec pin, and the query also isn't "execute"/"then", return it.
	//    Handles cases like VariableGet nodes where AI sends "ReturnValue" but pin is "MyVar".
	if (AliasLower != TEXT("execute") && AliasLower != TEXT("then") && AliasLower != TEXT("else"))
	{
		UEdGraphPin* OnlyCandidate = nullptr;
		int32 NonExecCount = 0;
		for (UEdGraphPin* Pin : Node->Pins)
		{
			if (Pin->Direction != Direction || Pin->bHidden) continue;
			if (Pin->PinType.PinCategory == UEdGraphSchema_K2::PC_Exec) continue;
			OnlyCandidate = Pin;
			NonExecCount++;
		}
		if (NonExecCount == 1 && OnlyCandidate)
		{
			return OnlyCandidate;
		}
	}

	return nullptr;
}

TSharedPtr<FJsonObject> FConnectNodesAction::GetParameterSchema() const
{
	auto MakeStr = [](const FString& Desc) -> TSharedPtr<FJsonObject> {
		TSharedPtr<FJsonObject> P = MakeShared<FJsonObject>();
		P->SetStringField(TEXT("type"), TEXT("string"));
		P->SetStringField(TEXT("description"), Desc);
		return P;
	};

	TSharedPtr<FJsonObject> Schema = MakeShared<FJsonObject>();
	Schema->SetStringField(TEXT("type"), TEXT("object"));

	TSharedPtr<FJsonObject> Props = MakeShared<FJsonObject>();
	Props->SetObjectField(TEXT("source_node"), MakeStr(TEXT("Source node ID (e.g. 'N1') — the node with the output pin")));
	Props->SetObjectField(TEXT("source_pin"), MakeStr(TEXT("Name of the output pin on the source node (e.g. 'then', 'ReturnValue', 'True')")));
	Props->SetObjectField(TEXT("target_node"), MakeStr(TEXT("Target node ID (e.g. 'N2') — the node with the input pin")));
	Props->SetObjectField(TEXT("target_pin"), MakeStr(TEXT("Name of the input pin on the target node (e.g. 'execute', 'InString', 'Condition')")));

	Schema->SetObjectField(TEXT("properties"), Props);

	TArray<TSharedPtr<FJsonValue>> Required;
	Required.Add(MakeShared<FJsonValueString>(TEXT("source_node")));
	Required.Add(MakeShared<FJsonValueString>(TEXT("source_pin")));
	Required.Add(MakeShared<FJsonValueString>(TEXT("target_node")));
	Required.Add(MakeShared<FJsonValueString>(TEXT("target_pin")));
	Schema->SetArrayField(TEXT("required"), Required);

	return Schema;
}

FBlueprintActionResult FConnectNodesAction::Execute(const TSharedPtr<FJsonObject>& Params)
{
	const FString SourceNodeId = Params->GetStringField(TEXT("source_node"));
	const FString SourcePinName = Params->GetStringField(TEXT("source_pin"));
	const FString TargetNodeId = Params->GetStringField(TEXT("target_node"));
	const FString TargetPinName = Params->GetStringField(TEXT("target_pin"));

	// Find nodes
	UEdGraphNode* SourceNode = Registry->FindTrackedNode(SourceNodeId);
	if (!SourceNode)
	{
		return FBlueprintActionResult::Failure(FString::Printf(TEXT("Source node '%s' not found"), *SourceNodeId));
	}

	UEdGraphNode* TargetNode = Registry->FindTrackedNode(TargetNodeId);
	if (!TargetNode)
	{
		return FBlueprintActionResult::Failure(FString::Printf(TEXT("Target node '%s' not found"), *TargetNodeId));
	}

	// Guard: reject cross-blueprint connections
	UEdGraph* SourceGraph = SourceNode->GetGraph();
	UEdGraph* TargetGraph = TargetNode->GetGraph();
	if (SourceGraph && TargetGraph)
	{
		UBlueprint* SourceBP = FBlueprintEditorUtils::FindBlueprintForGraph(SourceGraph);
		UBlueprint* TargetBP = FBlueprintEditorUtils::FindBlueprintForGraph(TargetGraph);
		if (SourceBP && TargetBP && SourceBP != TargetBP)
		{
			return FBlueprintActionResult::Failure(
				FString::Printf(TEXT("Cannot connect nodes across different Blueprints! %s is in '%s' but %s is in '%s'. Use node IDs from the SAME blueprint."),
					*SourceNodeId, *SourceBP->GetName(), *TargetNodeId, *TargetBP->GetName()));
		}
	}

	// Find pins (with fuzzy matching for case, spaces, and boolean prefixes)
	UEdGraphPin* SourcePin = FindPinFuzzy(SourceNode, SourcePinName, EGPD_Output);
	if (!SourcePin)
	{
		// List available output pins for error message
		FString Available;
		for (UEdGraphPin* Pin : SourceNode->Pins)
		{
			if (Pin->Direction == EGPD_Output && !Pin->bHidden)
			{
				Available += Pin->PinName.ToString() + TEXT(", ");
			}
		}
		FString NodeTitle = SourceNode->GetNodeTitle(ENodeTitleType::FullTitle).ToString();
		return FBlueprintActionResult::Failure(
			FString::Printf(TEXT("Output pin '%s' not found on node %s (\"%s\"). Available outputs: [%s]"),
				*SourcePinName, *SourceNodeId, *NodeTitle, *Available));
	}

	UEdGraphPin* TargetPin = FindPinFuzzy(TargetNode, TargetPinName, EGPD_Input);
	if (!TargetPin)
	{
		// List available input pins for error message
		FString Available;
		for (UEdGraphPin* Pin : TargetNode->Pins)
		{
			if (Pin->Direction == EGPD_Input && !Pin->bHidden)
			{
				Available += Pin->PinName.ToString() + TEXT(", ");
			}
		}
		FString NodeTitle = TargetNode->GetNodeTitle(ENodeTitleType::FullTitle).ToString();
		return FBlueprintActionResult::Failure(
			FString::Printf(TEXT("Input pin '%s' not found on node %s (\"%s\"). Available inputs: [%s]"),
				*TargetPinName, *TargetNodeId, *NodeTitle, *Available));
	}

	// Try to connect
	const UEdGraphSchema_K2* K2Schema = GetDefault<UEdGraphSchema_K2>();
	const bool bConnected = K2Schema->TryCreateConnection(SourcePin, TargetPin);

	if (!bConnected)
	{
		return FBlueprintActionResult::Failure(
			FString::Printf(TEXT("Could not connect %s.%s → %s.%s (incompatible pin types)"),
				*SourceNodeId, *SourcePinName, *TargetNodeId, *TargetPinName));
	}

	// Mark modified
	UEdGraph* Graph = SourceNode->GetGraph();
	if (Graph)
	{
		Graph->NotifyGraphChanged();
		UBlueprint* BP = FBlueprintEditorUtils::FindBlueprintForGraph(Graph);
		if (BP)
		{
			FBlueprintEditorUtils::MarkBlueprintAsModified(BP);
		}
	}

	TSharedPtr<FJsonObject> Data = MakeShared<FJsonObject>();
	Data->SetStringField(TEXT("source"), FString::Printf(TEXT("%s.%s"), *SourceNodeId, *SourcePinName));
	Data->SetStringField(TEXT("target"), FString::Printf(TEXT("%s.%s"), *TargetNodeId, *TargetPinName));

	return FBlueprintActionResult::Success(
		FString::Printf(TEXT("Connected %s.%s → %s.%s"),
			*SourceNodeId, *SourcePinName, *TargetNodeId, *TargetPinName),
		Data);
}
