// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#include "Actions/CompileBlueprintAction.h"

#include "Kismet2/KismetEditorUtilities.h"
#include "Kismet2/BlueprintEditorUtils.h"
#include "Kismet2/CompilerResultsLog.h"
#include "Engine/Blueprint.h"
#include "AssetRegistry/AssetRegistryModule.h"
#include "Engine/BlueprintGeneratedClass.h"

static UBlueprint* FindBPForCompile(const FString& NameOrPath)
{
	if (NameOrPath.StartsWith(TEXT("/")))
	{
		return LoadObject<UBlueprint>(nullptr, *NameOrPath);
	}
	FAssetRegistryModule& AssetReg = FModuleManager::LoadModuleChecked<FAssetRegistryModule>("AssetRegistry");
	TArray<FAssetData> Assets;
	AssetReg.Get().GetAssetsByClass(UBlueprint::StaticClass()->GetClassPathName(), Assets);
	for (const FAssetData& Asset : Assets)
	{
		if (Asset.AssetName.ToString().Equals(NameOrPath, ESearchCase::IgnoreCase))
		{
			return Cast<UBlueprint>(Asset.GetAsset());
		}
	}
	return nullptr;
}

TSharedPtr<FJsonObject> FCompileBlueprintAction::GetParameterSchema() const
{
	TSharedPtr<FJsonObject> Schema = MakeShared<FJsonObject>();
	Schema->SetStringField(TEXT("type"), TEXT("object"));

	TSharedPtr<FJsonObject> Props = MakeShared<FJsonObject>();
	TSharedPtr<FJsonObject> BPProp = MakeShared<FJsonObject>();
	BPProp->SetStringField(TEXT("type"), TEXT("string"));
	BPProp->SetStringField(TEXT("description"), TEXT("Blueprint name or content path"));
	Props->SetObjectField(TEXT("blueprint"), BPProp);

	Schema->SetObjectField(TEXT("properties"), Props);

	TArray<TSharedPtr<FJsonValue>> Required;
	Required.Add(MakeShared<FJsonValueString>(TEXT("blueprint")));
	Schema->SetArrayField(TEXT("required"), Required);

	return Schema;
}

FBlueprintActionResult FCompileBlueprintAction::Execute(const TSharedPtr<FJsonObject>& Params)
{
	const FString BPName = Params->GetStringField(TEXT("blueprint"));

	UBlueprint* BP = FindBPForCompile(BPName);
	if (!BP)
	{
		return FBlueprintActionResult::Failure(FString::Printf(TEXT("Blueprint '%s' not found"), *BPName));
	}

	// Compile
	FKismetEditorUtilities::CompileBlueprint(BP, EBlueprintCompileOptions::None, nullptr);

	TSharedPtr<FJsonObject> Data = MakeShared<FJsonObject>();
	Data->SetStringField(TEXT("blueprint"), BP->GetPathName());

	// Check compilation status
	bool bHasErrors = BP->Status == BS_Error;

	Data->SetBoolField(TEXT("success"), !bHasErrors);
	Data->SetStringField(TEXT("status"), bHasErrors ? TEXT("Error") : TEXT("Success"));

	// Extract compile error/warning messages by scanning graph nodes
	if (bHasErrors)
	{
		TArray<TSharedPtr<FJsonValue>> ErrorArray;
		int32 WarningCount = 0;

		// Nodes with compile errors have bHasCompilerMessage=true, ErrorMsg, and ErrorType set
		TArray<UEdGraph*> AllGraphs;
		BP->GetAllGraphs(AllGraphs);
		for (UEdGraph* Graph : AllGraphs)
		{
			for (UEdGraphNode* Node : Graph->Nodes)
			{
				if (Node->bHasCompilerMessage)
				{
					// ErrorType is a bitmask: 0x01 = Error, 0x02 = Warning, 0x04 = Note
					bool bIsError = (Node->ErrorType <= EMessageSeverity::Error);
					bool bIsWarning = (Node->ErrorType == EMessageSeverity::Warning);

					if (bIsWarning) WarningCount++;

					TSharedPtr<FJsonObject> ErrObj = MakeShared<FJsonObject>();
					ErrObj->SetStringField(TEXT("severity"), bIsError ? TEXT("Error") : TEXT("Warning"));
					ErrObj->SetStringField(TEXT("message"), Node->ErrorMsg);
					ErrObj->SetStringField(TEXT("node_title"), Node->GetNodeTitle(ENodeTitleType::FullTitle).ToString());
					ErrObj->SetStringField(TEXT("node_class"), Node->GetClass()->GetName());
					ErrObj->SetNumberField(TEXT("pos_x"), Node->NodePosX);
					ErrObj->SetNumberField(TEXT("pos_y"), Node->NodePosY);

					// Include pin info for the erroneous node so AI can attempt fixes
					TArray<TSharedPtr<FJsonValue>> PinsArray;
					for (UEdGraphPin* Pin : Node->Pins)
					{
						if (Pin->bHidden) continue;
						TSharedPtr<FJsonObject> PinObj = MakeShared<FJsonObject>();
						PinObj->SetStringField(TEXT("name"), Pin->PinName.ToString());
						PinObj->SetStringField(TEXT("dir"), Pin->Direction == EGPD_Input ? TEXT("In") : TEXT("Out"));
						PinObj->SetBoolField(TEXT("connected"), Pin->LinkedTo.Num() > 0);
						if (Pin->bOrphanedPin)
						{
							PinObj->SetBoolField(TEXT("orphaned"), true);
						}
						PinsArray.Add(MakeShared<FJsonValueObject>(PinObj));
					}
					ErrObj->SetArrayField(TEXT("pins"), PinsArray);

					ErrorArray.Add(MakeShared<FJsonValueObject>(ErrObj));

					// Cap at 10 errors to stay within payload budget
					if (ErrorArray.Num() >= 10) break;
				}
			}
			if (ErrorArray.Num() >= 10) break;
		}

		Data->SetArrayField(TEXT("errors"), ErrorArray);
		Data->SetNumberField(TEXT("error_count"), ErrorArray.Num());
		Data->SetNumberField(TEXT("warning_count"), WarningCount);

		FBlueprintActionResult R;
		R.bSuccess = false;
		R.Message = FString::Printf(TEXT("Blueprint '%s' compiled with %d error(s). See 'errors' array for details — fix broken nodes then recompile."), *BPName, ErrorArray.Num());
		R.Data = Data;
		return R;
	}

	return FBlueprintActionResult::Success(
		FString::Printf(TEXT("Blueprint '%s' compiled successfully"), *BPName),
		Data);
}
