// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#include "Actions/AddVariableAction.h"

#include "Kismet2/BlueprintEditorUtils.h"
#include "Engine/Blueprint.h"
#include "EdGraphSchema_K2.h"
#include "AssetRegistry/AssetRegistryModule.h"
#include "Engine/BlueprintGeneratedClass.h"

// ── Helpers ──────────────────────────────────────────────────────────────────

static UBlueprint* FindBPByName(const FString& NameOrPath)
{
	if (NameOrPath.StartsWith(TEXT("/")))
	{
		return LoadObject<UBlueprint>(nullptr, *NameOrPath);
	}

	FAssetRegistryModule& AssetReg = FModuleManager::LoadModuleChecked<FAssetRegistryModule>("AssetRegistry");
	TArray<FAssetData> Assets;
	AssetReg.Get().GetAssetsByClass(UBlueprint::StaticClass()->GetClassPathName(), Assets);
	for (const FAssetData& Asset : Assets)
	{
		if (Asset.AssetName.ToString().Equals(NameOrPath, ESearchCase::IgnoreCase))
		{
			return Cast<UBlueprint>(Asset.GetAsset());
		}
	}
	return nullptr;
}

static FEdGraphPinType ResolvePinType(const FString& TypeStr)
{
	FEdGraphPinType PinType;
	PinType.PinCategory = UEdGraphSchema_K2::PC_Real; // default — UE5 uses PC_Real for all float/double
	PinType.PinSubCategory = TEXT("double");

	FString Lower = TypeStr.ToLower();

	if (Lower == TEXT("bool") || Lower == TEXT("boolean"))
	{
		PinType.PinCategory = UEdGraphSchema_K2::PC_Boolean;
	}
	else if (Lower == TEXT("int") || Lower == TEXT("integer") || Lower == TEXT("int32"))
	{
		PinType.PinCategory = UEdGraphSchema_K2::PC_Int;
	}
	else if (Lower == TEXT("int64"))
	{
		PinType.PinCategory = UEdGraphSchema_K2::PC_Int64;
	}
	else if (Lower == TEXT("float"))
	{
		PinType.PinCategory = UEdGraphSchema_K2::PC_Real;
		PinType.PinSubCategory = TEXT("double");
	}
	else if (Lower == TEXT("double") || Lower == TEXT("real"))
	{
		PinType.PinCategory = UEdGraphSchema_K2::PC_Real;
		PinType.PinSubCategory = TEXT("double");
	}
	else if (Lower == TEXT("string"))
	{
		PinType.PinCategory = UEdGraphSchema_K2::PC_String;
	}
	else if (Lower == TEXT("text"))
	{
		PinType.PinCategory = UEdGraphSchema_K2::PC_Text;
	}
	else if (Lower == TEXT("name"))
	{
		PinType.PinCategory = UEdGraphSchema_K2::PC_Name;
	}
	else if (Lower == TEXT("byte"))
	{
		PinType.PinCategory = UEdGraphSchema_K2::PC_Byte;
	}
	else if (Lower == TEXT("vector"))
	{
		PinType.PinCategory = UEdGraphSchema_K2::PC_Struct;
		PinType.PinSubCategoryObject = TBaseStructure<FVector>::Get();
	}
	else if (Lower == TEXT("rotator"))
	{
		PinType.PinCategory = UEdGraphSchema_K2::PC_Struct;
		PinType.PinSubCategoryObject = TBaseStructure<FRotator>::Get();
	}
	else if (Lower == TEXT("transform"))
	{
		PinType.PinCategory = UEdGraphSchema_K2::PC_Struct;
		PinType.PinSubCategoryObject = TBaseStructure<FTransform>::Get();
	}
	else if (Lower == TEXT("color"))
	{
		PinType.PinCategory = UEdGraphSchema_K2::PC_Struct;
		PinType.PinSubCategoryObject = TBaseStructure<FColor>::Get();
	}
	else if (Lower == TEXT("linearcolor"))
	{
		PinType.PinCategory = UEdGraphSchema_K2::PC_Struct;
		PinType.PinSubCategoryObject = TBaseStructure<FLinearColor>::Get();
	}
	else if (Lower.StartsWith(TEXT("object:")))
	{
		FString ClassName = TypeStr.Mid(7);
		PinType.PinCategory = UEdGraphSchema_K2::PC_Object;
		for (TObjectIterator<UClass> It; It; ++It)
		{
			if ((*It)->GetName().Equals(ClassName, ESearchCase::IgnoreCase) ||
				(*It)->GetName().Equals(TEXT("A") + ClassName, ESearchCase::IgnoreCase) ||
				(*It)->GetName().Equals(TEXT("U") + ClassName, ESearchCase::IgnoreCase))
			{
				PinType.PinSubCategoryObject = *It;
				break;
			}
		}
	}

	return PinType;
}

// ── Schema ───────────────────────────────────────────────────────────────────

TSharedPtr<FJsonObject> FAddVariableAction::GetParameterSchema() const
{
	auto MakeStr = [](const FString& Desc) -> TSharedPtr<FJsonObject> {
		TSharedPtr<FJsonObject> P = MakeShared<FJsonObject>();
		P->SetStringField(TEXT("type"), TEXT("string"));
		P->SetStringField(TEXT("description"), Desc);
		return P;
	};
	auto MakeBool = [](const FString& Desc) -> TSharedPtr<FJsonObject> {
		TSharedPtr<FJsonObject> P = MakeShared<FJsonObject>();
		P->SetStringField(TEXT("type"), TEXT("boolean"));
		P->SetStringField(TEXT("description"), Desc);
		return P;
	};

	TSharedPtr<FJsonObject> Schema = MakeShared<FJsonObject>();
	Schema->SetStringField(TEXT("type"), TEXT("object"));

	TSharedPtr<FJsonObject> Props = MakeShared<FJsonObject>();
	Props->SetObjectField(TEXT("blueprint"), MakeStr(TEXT("Blueprint name or content path")));
	Props->SetObjectField(TEXT("variable_name"), MakeStr(TEXT("Name of the variable to add")));
	Props->SetObjectField(TEXT("variable_type"), MakeStr(TEXT("Type: bool, int, float, double, string, text, name, byte, Vector, Rotator, Transform, Color, LinearColor, or 'object:ClassName'")));
	Props->SetObjectField(TEXT("is_array"), MakeBool(TEXT("If true, creates an array of the specified type")));
	Props->SetObjectField(TEXT("default_value"), MakeStr(TEXT("Default value as string (e.g. '100', 'true', 'Hello')")));
	Props->SetObjectField(TEXT("category"), MakeStr(TEXT("Variable category for organizing in the Blueprint editor")));
	Props->SetObjectField(TEXT("is_instance_editable"), MakeBool(TEXT("If true, variable is editable per instance in the Details panel")));
	Props->SetObjectField(TEXT("tooltip"), MakeStr(TEXT("Tooltip text for the variable")));

	Schema->SetObjectField(TEXT("properties"), Props);

	TArray<TSharedPtr<FJsonValue>> Required;
	Required.Add(MakeShared<FJsonValueString>(TEXT("blueprint")));
	Required.Add(MakeShared<FJsonValueString>(TEXT("variable_name")));
	Required.Add(MakeShared<FJsonValueString>(TEXT("variable_type")));
	Schema->SetArrayField(TEXT("required"), Required);

	return Schema;
}

// ── Execute ──────────────────────────────────────────────────────────────────

FBlueprintActionResult FAddVariableAction::Execute(const TSharedPtr<FJsonObject>& Params)
{
	const FString BPName = Params->GetStringField(TEXT("blueprint"));
	const FString VarName = Params->GetStringField(TEXT("variable_name"));
	const FString VarType = Params->GetStringField(TEXT("variable_type"));
	const bool bIsArray = Params->HasField(TEXT("is_array")) && Params->GetBoolField(TEXT("is_array"));
	const bool bInstanceEditable = Params->HasField(TEXT("is_instance_editable")) && Params->GetBoolField(TEXT("is_instance_editable"));

	UBlueprint* BP = FindBPByName(BPName);
	if (!BP) return FBlueprintActionResult::Failure(FString::Printf(TEXT("Blueprint '%s' not found"), *BPName));

	// Check if variable already exists — return success (idempotent) if it does
	for (const FBPVariableDescription& Var : BP->NewVariables)
	{
		if (Var.VarName.ToString().Equals(VarName, ESearchCase::IgnoreCase))
		{
			TSharedPtr<FJsonObject> Data = MakeShared<FJsonObject>();
			Data->SetStringField(TEXT("variable_name"), VarName);
			Data->SetStringField(TEXT("status"), TEXT("already_exists"));
			return FBlueprintActionResult::Success(
				FString::Printf(TEXT("Variable '%s' already exists in '%s' (no changes made)"), *VarName, *BPName),
				Data);
		}
	}

	// Resolve type
	FEdGraphPinType PinType = ResolvePinType(VarType);
	if (bIsArray)
	{
		PinType.ContainerType = EPinContainerType::Array;
	}

	// Add the variable
	bool bSuccess = FBlueprintEditorUtils::AddMemberVariable(BP, FName(*VarName), PinType);
	if (!bSuccess)
	{
		return FBlueprintActionResult::Failure(
			FString::Printf(TEXT("Failed to add variable '%s' to '%s'"), *VarName, *BPName));
	}

	// Set additional properties
	int32 VarIdx = FBlueprintEditorUtils::FindNewVariableIndex(BP, FName(*VarName));
	if (VarIdx != INDEX_NONE)
	{
		FBPVariableDescription& Var = BP->NewVariables[VarIdx];

		if (bInstanceEditable)
		{
			Var.PropertyFlags |= CPF_Edit | CPF_BlueprintVisible;
		}

		if (Params->HasField(TEXT("category")))
		{
			Var.Category = FText::FromString(Params->GetStringField(TEXT("category")));
		}

		if (Params->HasField(TEXT("tooltip")))
		{
			FBlueprintEditorUtils::SetBlueprintVariableMetaData(BP, FName(*VarName), nullptr,
				TEXT("tooltip"), Params->GetStringField(TEXT("tooltip")));
		}

		if (Params->HasField(TEXT("default_value")))
		{
			Var.DefaultValue = Params->GetStringField(TEXT("default_value"));
		}
	}

	FBlueprintEditorUtils::MarkBlueprintAsStructurallyModified(BP);

	TSharedPtr<FJsonObject> Data = MakeShared<FJsonObject>();
	Data->SetStringField(TEXT("variable_name"), VarName);
	Data->SetStringField(TEXT("variable_type"), VarType);
	Data->SetBoolField(TEXT("is_array"), bIsArray);

	return FBlueprintActionResult::Success(
		FString::Printf(TEXT("Added variable '%s' (type: %s%s) to '%s'"),
			*VarName, *VarType, bIsArray ? TEXT("[]") : TEXT(""), *BPName),
		Data);
}
